# -*- coding: utf-8 -*-
from collections import defaultdict

import mpfs.engine.process
from mpfs.common.errors import EventDispatcherError
from mpfs.core.event_dispatcher.events import Event


class EventCallback(object):
    """Обертка над callback, помогающая связать с ним доп. информацию"""
    def __init__(self, callback, suppress_exceptions):
        self.callback = callback
        self.suppress_exceptions = suppress_exceptions


class EventDispatcher(object):
    """
    Диспетчер событий.

    Позволяет производить подписку на определенные события.

    При старте MPFS идет поиск и регистрация всех событий(классов) в системе(см. mpfs.engine.process).
    Модуль, содержащий события, должен лежать в mpfs.core и "выше".
    Все регистрируемые события должны наследоваться от класса Event.

    Подписки на события происходят во время инициализации MPFS и в процессе работы не меняются.
    Подписаться можно только на зарегистрированное событие с помощью метода subscribe или декоратора
    mpfs.core.event_dispatcher.event.subscribe

    При срабатывании event-а последовательно дергаются все callback-и, которые подписались на этот event.
    Так же вызываются события вверх по иерархии наследования событий.
    Событие "Event" вызывается при любом событии в MPFS.

    Алгоритм добавления событий:
        1) Создаем модуль с событиям(напр. mpfs.core.albums.events)
        2) В нем создаем классы события(наследники Event)
        3) Создаем модуль с подписками(напр. mpfs.core.albums.event_subscribtions)
        4) Объявлем в нем функции-callback-и, принимающие один аргумент - event и
           декорируем функции декоратором mpfs.core.event_dispatcher.event.subscribe
            @subscribe(MyEvent)
            def callback(event):
                pass
        5) В коде создаем объекты события и вызываем метод send() (MyEvent().send())
    """
    def __init__(self, event_classes):
        # хранит callback-и для каждого события
        self._observers = defaultdict(list)
        for event_class in event_classes:
            if not isinstance(event_class, type):
                raise EventDispatcherError('Not class object. Object: %r' % event_class)
            if not issubclass(event_class, Event):
                raise EventDispatcherError('Event should be subclass of "Event". Object: %r' % event_class)
        # хранит события системы
        self._event_classes = set(event_classes)

    def subscribe(self, event_class, callback, suppress_exceptions=False):
        """
        Подписка на уведомления

        Подписываться можно на конкретное событие - event_class
        callback - вызываемый объект с одним параметром-событием
        suppress_exceptions - подавлять исключения, появляющиеся в процессе работы callback-а
        """
        if not issubclass(event_class, Event):
            raise EventDispatcherError('Event should inherits from "Event" class')
        if not hasattr(callback, '__call__'):
            raise EventDispatcherError('"Callback" should be callable')

        event_callback = EventCallback(callback, suppress_exceptions)
        # добавляем callback событию, а также родительским событиям
        self._observers[event_class].append(event_callback)

    def send(self, event):
        """
        Отправка уведомления
        """
        if not isinstance(event, Event):
            raise EventDispatcherError('Sending obj should be inherits from "Event"')

        for sys_event in self._event_classes:
            if isinstance(event, sys_event):
                for event_callback in self._observers[sys_event]:
                    try:
                        event_callback.callback(event)
                    except Exception:
                        suppress = event_callback.suppress_exceptions
                        mpfs.engine.process.get_error_log().exception('Got exception at event callback, suppress: "%s"' % suppress)
                        if not suppress:
                            raise
