# -*- coding: utf-8 -*-

from mpfs.common.errors import EventError


def subscribe(event_class, suppress_exceptions=False):
    """
    Декоратор, позволяющий подписывать декорируемую функцию на события

    Декоратору должен быть передан класс события(event_class)
    Декорируемая функция принимает в качестве единственного параметра объект события
    suppress_exceptions - подавлять исключения, появляющиеся в процессе работы callback-а, в логах будет traceback

    @subscribe(MyEvent)
    def callback(event):
        pass
    """
    from mpfs.engine.process import event_dispatcher

    def decorator(function):
        event_dispatcher().subscribe(event_class, function, suppress_exceptions=suppress_exceptions)
        return function
    return decorator


class EventMetaClass(type):
    """Метакласс обеспечивающий механизм наследования аттрибутa `required_fields`."""
    def __new__(mcs, name, parents, attrs):
        required_fields = {}
        for parent in parents:
            required_fields.update(getattr(parent, 'required_fields', {}))
        required_fields.update(attrs.get('required_fields', {}))
        attrs['required_fields'] = required_fields
        return super(EventMetaClass, mcs).__new__(mcs, name, parents, attrs)


class Event(object):
    """
    Базовый класс событий

    Все события системы должны наследоваться от него
    При инициализации в конструктор должны быть переданы именованные параметры, описанные в атрибуте required_fields
    required_fields - словарь, содержащий необходимые поля(название параметра: допустимый тип или tuple типов)
    Значения required_fields наследуются


    class ExampleEvent(Event):
        required_fields = (('field_name_1', int),
                           ('field_name_2', (dict, list)))

    my_event = ExampleEvent(field_name_1=42, field_name_2=[1, 2, 3])
    my_event.send()
    """
    __metaclass__ = EventMetaClass
    required_fields = ()

    def __init__(self, **kwargs):
        self._data = {}
        for field_name, field_types in self.required_fields.iteritems():
            if field_name not in kwargs:
                raise EventError('Field "%s" required' % field_name)
            if not isinstance(kwargs[field_name], field_types):
                actual_field_type = type(kwargs[field_name])
                raise EventError('Field "%s" should be: %s, but was: %s' % (field_name, field_types, actual_field_type))
            self._data[field_name] = kwargs[field_name]

    def send(self):
        """
        Отправить событие(себя) диспетчеру
        """
        from mpfs.engine.process import event_dispatcher
        event_dispatcher().send(self)

    @property
    def data(self):
        """
        Получить данные события
        """
        return self._data

    def __repr__(self):
        return "<%s(data=%r)>" % (self.__class__.__name__,
                                  self.data)
