# -*- coding: utf-8 -*-
import time

from collections import Iterable, OrderedDict

import mpfs.engine.process

from mpfs.common.static.tags import billing
from mpfs.common.util.logger import TSKVMessage
from mpfs.config import settings
from mpfs.core.event_dispatcher.events import subscribe
from mpfs.core.albums.models import AlbumItem

from mpfs.core.address import Address
from mpfs.core.albums import events as album_events
from mpfs.core.billing import events as billing_events
from mpfs.core.event_history.logger import log_raw_event, log_raw_event_message, get_log_raw_event_message
from mpfs.core.filesystem import events as fs_events
from mpfs.core.services.common_service import StorageService
from mpfs.core.social.publicator import Publicator
from mpfs.core.social.share import events as share_events
from mpfs.core.social.share.group import Group, LinkToGroup, GroupInvites
from mpfs.core.user.constants import PHOTOUNLIM_AREA_PATH, ADDITIONAL_AREA, HIDDEN_AREA
from mpfs.invite import events as invite_events
from mpfs.core.user import events as space_events
from mpfs.core.filesystem.constants import NOTES_STORAGE_AREA
from mpfs.core.billing.product import Product
from mpfs.core.billing.product.catalog import Catalog
from mpfs.core.filesystem.resources.share import SharedResource
from mpfs.core.lenta.utils import LentaMediaType


error_log = mpfs.engine.process.get_error_log()


SETTINGS_FOLDERS_ATTACH_ALLOWED_FOLDERS = settings.folders['/attach'].get('allowed_folders', [])


class EventType(object):
    ALBUM_CREATE = 'album-create'
    ALBUM_REMOVE = 'album-remove'
    ALBUM_CHANGE_COVER = 'album-change-cover'
    ALBUM_CHANGE_COVER_OFFSET = 'album-change-cover-offset'
    ALBUM_CHANGE_PUBLICITY = 'album-change-publicity'
    ALBUM_CHANGE_TITLE = 'album-change-title'
    ALBUM_ITEMS_APPEND = 'album-items-append'
    ALBUM_ITEMS_REMOVE = 'album-items-remove'
    ALBUM_POST_TO_SOCIAL = 'album-post-to-social'

    BILLING_ORDER_NEW = 'billing-order-new'
    BILLING_BUY_NEW = 'billing-buy-new'
    BILLING_PROLONG = 'billing-prolong'
    BILLING_UNSUBSCRIBE = 'billing-unsubscribe'
    BILLING_DELETE = 'billing-delete'

    FS_MK_DIR = 'fs-mkdir'
    FS_MK_SYS_DIR = 'fs-mksysdir'
    FS_COPY = 'fs-copy'
    FS_MOVE = 'fs-move'
    FS_RM = 'fs-rm'
    FS_STORE = 'fs-store'
    FS_HARDLINK_COPY = 'fs-hardlink-copy'
    FS_TRASH_APPEND = 'fs-trash-append'
    FS_TRASH_DROP_ALL = 'fs-trash-drop-all'
    FS_TRASH_DROP = 'fs-trash-drop'
    FS_TRASH_RESTORE = 'fs-trash-restore'
    FS_SET_PUBLIC = 'fs-set-public'
    FS_SET_PRIVATE = 'fs-set-private'

    SHARE_CREATE_GROUP = 'share-create-group'
    SHARE_UNSHARE_FOLDER = 'share-unshare-folder'
    SHARE_INVITE_USER = 'share-invite-user'
    SHARE_REMOVE_INVITE = 'share-remove-invite'
    SHARE_CHANGE_INVITE_RIGHTS = 'share-change-invite-rights'
    SHARE_CHANGE_RIGHTS = 'share-change-rights'
    SHARE_KICK_FROM_GROUP = 'share-kick-from-group'
    SHARE_CHANGE_GROUP_OWNER = 'share-change-group-owner'
    SHARE_ACTIVATE_INVITE = 'share-activate-invite'
    SHARE_REJECT_INVITE = 'share-reject-invite'
    SHARE_LEAVE_GROUP = 'share-leave-group'

    INVITE_SENT = 'invite-sent'
    INVITE_ACTIVATION = 'invite-activation'

    SPACE_ENLARGE = 'space-enlarge'
    SPACE_REDUCE = 'space-reduce'


EVENT_TYPES = {
    EventType.ALBUM_CREATE: album_events.AlbumCreateEvent,
    EventType.ALBUM_REMOVE: album_events.AlbumRemoveEvent,
    EventType.ALBUM_CHANGE_COVER: album_events.AlbumChangeCoverEvent,
    EventType.ALBUM_CHANGE_COVER_OFFSET: album_events.AlbumChangeCoverOffsetEvent,
    EventType.ALBUM_CHANGE_PUBLICITY: album_events.AlbumChangePublishEvent,
    EventType.ALBUM_CHANGE_TITLE: album_events.AlbumChangeTitleEvent,
    EventType.ALBUM_ITEMS_APPEND: album_events.AlbumChangeItemsAppendEvent,
    EventType.ALBUM_ITEMS_REMOVE: album_events.AlbumChangeItemsRemoveEvent,
    EventType.ALBUM_POST_TO_SOCIAL: album_events.AlbumPostToSocialEvent,

    EventType.BILLING_ORDER_NEW: billing_events.BillingOrderNewEvent,
    EventType.BILLING_BUY_NEW: billing_events.BillingBuyNewEvent,
    EventType.BILLING_PROLONG: billing_events.BillingProlongEvent,
    EventType.BILLING_UNSUBSCRIBE: billing_events.BillingUnsubscribeEvent,
    EventType.BILLING_DELETE: billing_events.BillingDeleteEvent,

    EventType.FS_MK_DIR: fs_events.FilesystemCreateDirectoryEvent,
    EventType.FS_MK_SYS_DIR: fs_events.FilesystemCreateSystemDirectoryEvent,
    EventType.FS_COPY: fs_events.FilesystemCopyEvent,
    EventType.FS_MOVE: fs_events.FilesystemMoveEvent,
    EventType.FS_RM: fs_events.FilesystemRemoveEvent,
    EventType.FS_STORE: fs_events.FilesystemStoreEvent,
    EventType.FS_HARDLINK_COPY: fs_events.FilesystemHardlinkCopyEvent,
    EventType.FS_TRASH_APPEND: fs_events.FilesystemTrashAppendEvent,
    EventType.FS_TRASH_DROP_ALL: fs_events.FilesystemTrashDropAllEvent,
    EventType.FS_TRASH_DROP: fs_events.FilesystemTrashDropEvent,
    EventType.FS_TRASH_RESTORE: fs_events.FilesystemTrashRestoreEvent,
    EventType.FS_SET_PUBLIC: fs_events.FilesystemSetPublicEvent,
    EventType.FS_SET_PRIVATE: fs_events.FilesystemSetPrivateEvent,

    EventType.SHARE_CREATE_GROUP: share_events.ShareCreateGroupEvent,
    EventType.SHARE_UNSHARE_FOLDER: share_events.ShareUnshareFolderEvent,
    EventType.SHARE_INVITE_USER: share_events.ShareCreateInviteEvent,
    EventType.SHARE_REMOVE_INVITE: share_events.ShareRemoveInviteEvent,
    EventType.SHARE_CHANGE_INVITE_RIGHTS: share_events.ShareChangeInviteRightsEvent,
    EventType.SHARE_CHANGE_RIGHTS: share_events.ShareChangeRightsEvent,
    EventType.SHARE_KICK_FROM_GROUP: share_events.ShareKickFromGroupEvent,
    EventType.SHARE_CHANGE_GROUP_OWNER: share_events.ShareChangeGroupOwnerEvent,
    EventType.SHARE_ACTIVATE_INVITE: share_events.ShareActivateInviteEvent,
    EventType.SHARE_REJECT_INVITE: share_events.ShareRejectInviteEvent,
    EventType.SHARE_LEAVE_GROUP: share_events.ShareLeaveGroupEvent,

    EventType.INVITE_SENT: invite_events.InviteSentEvent,
    EventType.INVITE_ACTIVATION: invite_events.InviteActivationEvent,

    EventType.SPACE_ENLARGE: space_events.EnlargeSpaceEvent,
    EventType.SPACE_REDUCE: space_events.ReduceSpaceEvent,
}

# Для быстрого поиска выворачиваем EVENT_TYPES
_EVENT_TYPES_REVERS_LOOKUP = {v: k for k, v in EVENT_TYPES.iteritems()}

# Список классов событий, которые необходимо логировать
_EVENT_TYPES_TO_LOG = _EVENT_TYPES_REVERS_LOOKUP.keys() + [fs_events.FilesystemGroupChangeEvent]

# Отображение наших локалей в локали, которые будут записаны в эвент лог.
MAP_PRODUCT_LANG_TO_LOG_LANG = {
    billing.UK_UA: ['ua', 'uk'],
    billing.RU_RU: ['ru'],
    billing.TR_TR: ['tr'],
    billing.EN_EN: ['en']
}


# Подписываемся на все события. Фильтровать будем в лиснере.
@subscribe(mpfs.core.event_dispatcher.events.Event)
def log_event(event):
    # если событие не логируется в историю, то ничего не делаем
    if not _is_logging_needed(event):
        return

    # Логируем события
    if isinstance(event, (album_events.AlbumChangeEvent, album_events.AlbumPostToSocialEvent)):
        _log_album_event(event)
    elif isinstance(event, fs_events.FilesystemGroupChangeEvent):
        _log_fs_group_event(event)
    elif isinstance(event, fs_events.FilesystemTargetChangeEvent):
        _log_fs_target_event(event)
    elif isinstance(event, share_events.ShareEvent):
        _log_share_event(event)
    elif isinstance(event, billing_events.BillingEvent):
        _log_billing_event(event)
    elif isinstance(event, space_events.SpaceEvent):
        _log_space_event(event)
    else:
        _log_event_default(event)


def _is_logging_needed(event):
    if type(event) not in _EVENT_TYPES_TO_LOG:
        return False
    data = event.data
    try:
        if ('src_resource' in data and data['src_resource'].address.storage_name == ADDITIONAL_AREA
                or 'tgt_resource' in data and data['tgt_resource'].address.storage_name == ADDITIONAL_AREA):
            return False
    except Exception:
        error_log.error('An error occurred during checking storage names', exc_info=True)

    # https://st.yandex-team.ru/CHEMODAN-54396
    # эти события приходят только при загрузке live photo и мы не хотим их логировать
    if isinstance(event, fs_events.FilesystemMoveEvent) and \
            data.get('is_live_photo', False) and \
            'src_address' in data and data['src_address'].storage_name == HIDDEN_AREA: 
        return False
    if isinstance(event, fs_events.FilesystemStoreEvent) and \
            data.get('is_live_photo', False) and \
            'tgt_address' in data and data['tgt_address'].storage_name == HIDDEN_AREA:
        return False
    if isinstance(event, fs_events.FilesystemGroupChangeEvent) and \
            'trigger_event' in data and isinstance(data['trigger_event'], fs_events.FilesystemMoveEvent):
        trigger_event_data = data['trigger_event'].data
        if trigger_event_data.get('is_live_photo', False) and \
                'src_address' in trigger_event_data and trigger_event_data['src_address'].storage_name == HIDDEN_AREA:
            return False

    return True


def _log_album_event(event):
    event_data = event.data

    if not isinstance(event, album_events.AlbumChangeItemsEvent):
        # Для альбомных событий вытаскиваем данные для логирования из атрибута 'album'
        album = event_data['album']
        data = {'uid': album.uid, 'album_id': album.id, 'album_title': album.title}
        if isinstance(event, album_events.AlbumCreateEvent):
            data['album_item_count'] = len(album.items)
        elif isinstance(event, album_events.AlbumChangeTitleEvent):
            data['prev_album_title'] = event_data['prev_title']
        elif isinstance(event, album_events.AlbumChangePublishEvent):
            data['album_is_public'] = int(album.is_public)
        elif isinstance(event, album_events.AlbumPostToSocialEvent):
            data['provider'] = event_data['provider']

        _log_event_raw(event, TSKVMessage.with_special_escaped(**data))

    # Логируем элементы альбомов...
    if isinstance(event, album_events.AlbumCreateEvent):
        # ... добавленные при создании альбома
        _log_album_items(event, event_data['album'].items, event_type='album-create-item')
    elif isinstance(event, album_events.AlbumChangeItemsEvent):
        # ... добавленные после создания альбома или удаленные
        _log_album_items(event, event_data['items'])


def _log_album_items(event, items, event_type=None):
    album = event.data['album']
    for item in items:
        data = {'uid': album.uid, 'album_id': album.id, 'album_title': album.title,
                'album_item_id': item.id, 'album_item_type': item.obj_type}
        if item.obj_type == AlbumItem.ALBUM:
            data['child_album_id'] = item.object.id
            data['child_album_title'] = item.object.title
        else:
            data['resource_address'] = item.object.visible_address.id
            data.update(_extract_common_resource_data(item.object))

        _log_event_raw(event, TSKVMessage.with_special_escaped(**data), event_type=event_type)


def _log_fs_group_event(event):
    base_event = event.data['trigger_event']

    data = {'user_uid': base_event.data['uid']}
    data.update(_extract_common_fs_target_data(base_event))

    # собираем адреса из различных мест, чтобы залогировать максимум доступной информации
    src_addresses = _collect_address_for_each_user(event, FolderIdExtractor.PREFIX_SOURCE)
    tgt_addresses = _collect_address_for_each_user(event, FolderIdExtractor.PREFIX_TARGET)

    folder_id_extractor = FolderIdExtractor()
    for uid in (src_addresses.viewkeys() | tgt_addresses.viewkeys()):
        src_address = src_addresses.get(uid)
        tgt_address = tgt_addresses.get(uid)

        user_data = data.copy()
        user_data['uid'] = (src_address or tgt_address).uid

        if src_address:
            user_data['src_rawaddress'] = src_address.id
            resource = base_event.data.get("src_resource")
            folder_id = folder_id_extractor.extract_from_resource(user_data['uid'], resource)
            if folder_id:
                user_data["src_folder_id"] = folder_id

        if tgt_address:
            user_data['tgt_rawaddress'] = tgt_address.id
            resource = base_event.data.get("tgt_resource")
            folder_id = folder_id_extractor.extract_from_resource(user_data['uid'], resource)
            if folder_id:
                user_data["tgt_folder_id"] = folder_id

        _log_event_raw(base_event, TSKVMessage.with_special_escaped(**user_data))
        _log_fs_delete_subdirs_if_needed(base_event, user_data)


def _collect_address_for_each_user(event, key_prefix):
    group_data = event.data
    base_event_data = group_data['trigger_event'].data

    addresses = {}

    resource = base_event_data.get(key_prefix + '_resource')
    if resource:
        resource_address = resource.address
        addresses[resource_address.uid] = resource_address

    current_user_address = base_event_data.get(key_prefix + '_address')
    if current_user_address:
        addresses[current_user_address.uid] = current_user_address

    group = group_data.get(key_prefix + '_group')
    if group and current_user_address:
        relative_path = group.get_relative_path(current_user_address)

        # адрес владельца
        addresses[group.owner] = Address.Make(group.owner, group.path + relative_path)

        # адреса подписчиков
        for link in group.iterlinks():
            addresses[link.uid] = Address.Make(link.uid, link.path + relative_path)

    return addresses


def should_drop_event_message_based_on_data(data):
    """Надо ли отбросить переданное сообщение на основе данных (дикт) и не записывать его в лог.

    Сейчас мы отбрасываем:
    * все сообщения, в которых участвует раздел /notes.
    * все сообщения, принадлежащие разрешенным папкам в /attach/
    """
    attach_folders_to_filter = set()
    for folder in SETTINGS_FOLDERS_ATTACH_ALLOWED_FOLDERS:
        folder = StorageService.add_beginning_and_trailing_slashes(folder)
        attach_folders_to_filter.add('/attach' + folder)

    def should_drop_by_address(address_key):
        if address_key in data:
            address = Address(data[address_key])
            if address.storage_name == NOTES_STORAGE_AREA:
                return True

            for folder_to_filter in attach_folders_to_filter:
                if StorageService.add_beginning_and_trailing_slashes(address.path).startswith(folder_to_filter):
                    return True

        return False

    try:
        tgt_raw_address_key = 'tgt_rawaddress'
        src_raw_address_key = 'src_rawaddress'
        if should_drop_by_address(tgt_raw_address_key) or should_drop_by_address(src_raw_address_key):
            return True

        return False
    except Exception as e:
        error_log.exception('Unexpected error in `should_drop_event_message_based_on_data`: %s' % e.message)
        # в случае ошибки в логике - не падаем
        return False


def get_fs_target_event_messages(event):
    data = event.data
    result = {'uid': data['uid'], 'tgt_rawaddress': data['tgt_address'].id}

    if 'src_resource' in data:
        result['src_rawaddress'] = data['src_resource'].address.id

    result.update(_extract_common_fs_target_data(event))

    folder_id_extractor = FolderIdExtractor()
    for prefix in FolderIdExtractor.PREFIXES:
        resource = data.get("%s_resource" % prefix)
        if not resource:
            continue
        folder_id = folder_id_extractor.extract_from_resource(data['uid'], resource)
        if not folder_id:
            continue
        result["%s_folder_id" % prefix] = folder_id

    # TODO: если понадбятся таймстемпы для src ресурса - перенести в цикл выше
    tgt_resource = data.get('tgt_resource')
    if tgt_resource:
        result['tgt_ctime'] = data['tgt_resource'].ctime
        result['tgt_utime'] = data['tgt_resource'].utime
        if 'etime' in data['tgt_resource'].meta:
            result['tgt_etime'] = data['tgt_resource'].meta['etime']

    r = OrderedDict()
    event_type = _get_event_type(event=event)
    if not should_drop_event_message_based_on_data(result):
        r.setdefault(event_type, []).append(
            get_log_raw_event_message(event_type, result)
        )
        for event_type, messages in get_fs_delete_subdir_messages(event, result).items():
            for message in messages:
                r.setdefault(event_type, []).append(message)

    return r


def _log_all_messages(msg_dict):
    for event_type, messages in msg_dict.items():
        for message in messages:
            log_raw_event_message(message)


def _log_fs_target_event(event):
    """Залогировать событие изменения целевого ресурса."""
    msg_dict = get_fs_target_event_messages(event)
    _log_all_messages(msg_dict)


def _log_fs_delete_subdirs_if_needed(event, logging_data):
    msg_dict = get_fs_delete_subdir_messages(event, logging_data)
    _log_all_messages(msg_dict)


def get_fs_delete_subdir_messages(event, logging_data):
    prefix = None

    if isinstance(event, fs_events.FilesystemRemoveEvent):
        prefix = 'tgt'
    elif isinstance(event, fs_events.FilesystemTrashAppendEvent):
        prefix = 'src'

    ret = OrderedDict()

    if prefix and logging_data.get(prefix + '_rawaddress'):
        resource = event.data.get(prefix + '_resource')

        if resource and resource.is_folder and resource.full_index_map:
            for key, child in resource.full_index_map.iteritems():
                if key != resource.id and child['type'] == 'dir' and child['meta'].get('file_id'):
                    result = {
                        'uid': logging_data['uid'],
                        'resource_id': str(resource.resource_id)
                    }
                    if 'user_uid' in logging_data:
                        result.update({'user_uid': logging_data['user_uid']})

                    event_type = 'fs-delete-subdir'
                    message = get_log_raw_event_message(
                        event_type,
                        result
                    )
                    ret.setdefault(event_type, []).append(message)

    return ret


def _extract_common_fs_target_data(event):
    event_data = event.data
    data = {}

    for key in ('type', 'subtype', 'force', 'auto', 'overwritten'):
        if key in event_data:
            data[key] = event_data[key]

    if isinstance(event, fs_events.FilesystemStoreHardlinkEvent):
        changes = []
        if event_data.get('set_public', False):
            changes.append('public')
        changes_dict = event_data.get('changes', {})
        if changes_dict and int(changes_dict.get('screenshot', '0')) == 1:
            changes.append('screenshot')
        data['changes'] = ','.join(changes)
        data['size'] = event_data.get('size', None)

    extract_public = isinstance(event, fs_events.FilesystemSetPublicEvent) or event_data.get('set_public', False)
    resource = event_data.get('tgt_resource', None) or event_data.get('src_resource', None)
    data.update(_extract_common_resource_data(resource, extract_public))

    return data


def _extract_common_resource_data(resource, extract_public=False):
    data = {'resource_type': resource.type}

    if resource.type == 'file':
        data['resource_media_type'] = resource.media_type
        data['lenta_media_type'] = LentaMediaType.convert_to_lenta_media_type(resource.media_type)

    if 'file_id' in resource.meta:
        data['resource_file_id'] = resource.meta['file_id']

    data['owner_uid'] = resource.storage_address.uid

    if extract_public:
        data.update({'public_key': resource.get_public_hash(), 'short_url': Publicator.get_short_url(resource)})

    return data


class FolderIdExtractor(object):
    """
    Содержит логику извлечения folder_id из ресурса

    Folder_id имеет следующие особенности:
        * Для файла folder_id == resource.get_parent().resource_id
        * Для папки folder_id == resource.resource_id
        * Для системного ресурса folder_id == resource.address.id
    Особое внимание уделяем рутовым(входным) папкам в механизме ОП.
    У каждого пользователя она своя, поэтому для каждого участника ОП folder_id будет свой!
    """
    PREFIX_SOURCE = 'src'
    PREFIX_TARGET = 'tgt'
    PREFIXES = (PREFIX_TARGET, PREFIX_SOURCE)

    def __init__(self):
        # мапинг вида: gid -> uid -> root_folder_resource
        self._shared_root_folders_cache = {}

    def _get_group_mapping(self, group):
        """
        Для группы(ОП), возвращает мапинг uid -> uid_root_folder
        """
        gid = group.gid
        if gid not in self._shared_root_folders_cache:
            # можно было применить загрузку ресурсов пачкой, но она разбивает
            # адреса по uid-ам, поэтому в данном случае смысла нет
            self._shared_root_folders_cache[gid] = {}
            for root_folder in group.get_all_root_folders(cached=True):
                uid = root_folder.visible_address.uid
                self._shared_root_folders_cache[gid][uid] = root_folder
        return self._shared_root_folders_cache[gid]

    def extract_from_resource(self, uid, resource):
        """
        Извлечь из ресурса folder_id
        """
        if not resource:
            return

        lenta_block_resource = resource.get_parent() if resource.is_file else resource
        if lenta_block_resource is None:
            return

        # если это рутовая ОП, то нужно её перезагрузить для конкретного пользователя
        if lenta_block_resource.is_group_root or lenta_block_resource.is_shared_root:
            group = resource.group
            group_mapping = self._get_group_mapping(group)
            if uid in group_mapping:
                lenta_block_resource = group_mapping[uid]
            else:
                # для публичного копирования корневой ОП, пользователь может не быть
                # участником группы. Поэтому берем папку владельца.
                lenta_block_resource = group_mapping[group.owner]

        # костыль для корневых папок: /disk, /trash - у них не было file_id и как следствие resource_id
        if lenta_block_resource.address.is_system and lenta_block_resource.address.path != PHOTOUNLIM_AREA_PATH:
            return "%s:%s" % (lenta_block_resource.uid, lenta_block_resource.address.path)
        return lenta_block_resource.resource_id.serialize()


def _log_share_event(event):
    event_data = event.data

    # извлекаем группу, ссылку на группу (если есть) и приглашение (если есть)
    link = None
    invite = None
    if isinstance(event, share_events.ShareGroupEvent):
        group = event_data['group']
    elif isinstance(event, share_events.ShareLinkToGroupEvent):
        if isinstance(event, share_events.ShareChangeGroupOwnerEvent):
            link = event_data['new_link']
        else:
            link = event_data['link']
        group = event_data['link'].group
    elif isinstance(event, share_events.ShareGroupInvitesEvent):
        invite = event_data['invite']
        group = invite.group
    else:
        # don't know how to log that
        return

    common_data = {'gid': group.gid, 'owner_uid': group.owner}

    if invite and invite.uid and invite.uid != group.owner:
        invite_uid = invite.uid
    else:
        invite_uid = None

    # извлекаем данные о пользователе общей папки (не владельце)
    if isinstance(event, share_events.ShareUserEvent):
        common_data['user_uid'] = event_data['uid']
    elif isinstance(event, share_events.ShareOwnerEvent):
        if link:
            common_data['user_uid'] = link.uid
        elif invite:
            if invite_uid:
                common_data['user_uid'] = invite_uid
            else:
                common_data['user_universe_login'] = invite.universe_login
                common_data['user_universe_service'] = invite.universe_service

    # извлекаем информацию о правах доступа
    if link is not None or invite is not None:
        common_data['rights'] = (link or invite).rights

    if 'prev_rights' in event_data:
        common_data['prev_rights'] = event_data['prev_rights']

    links = list(group.iterlinks())

    common_data['subscriber_count'] = len(links)

    logged_uids = []

    # логируем событие для владельца, подписчиков и текущего приглашенного пользователя
    _log_share_events_basic(event, common_data, logged_uids, [group, link, links, invite])

    # в случае события удаления общей папки ...
    if isinstance(event, share_events.ShareUnshareFolderEvent):
        invites = list(group.iterinvites())

        # ... логируем событие для всех приглашенных пользователей
        _log_share_events_basic(event, common_data, logged_uids, invites)

        # ... логируем для владельца список подписчиков и список инвайтов
        owner_common_data = common_data.copy()
        owner_common_data.update({'uid': group.owner, 'path': group.path, 'is_invite': False})
        logged_user_uids = []
        _log_share_event_users(event, owner_common_data, 'user', logged_user_uids, [link, links])
        _log_share_event_users(event, owner_common_data, 'invitee', logged_user_uids, [invite, invites])


def _log_share_events_basic(event, common_data, logged_uids, share_objects):
    for share_object in flatten(share_objects):
        _log_share_event_basic(event, share_object, common_data, logged_uids)


def _log_share_event_basic(event, share_object, common_data, logged_uids):
    uid, path, invite_hash = _extract_share_object_data(share_object)

    # не логируем, если не хватает обязательных атрибутов
    if uid is None or path is None:
        return

    # не логируем, если информация для этого пользователя уже была залогирована
    if uid in logged_uids:
        return

    data = common_data.copy()
    data.update({'uid': uid, 'path': path, 'is_invite': invite_hash is not None})

    if invite_hash:
        data['invite_hash'] = invite_hash

    elif isinstance(event, share_events.ShareActivateInviteEvent) and event.data['invite'].uid == uid:
        data['invite_hash'] = event.data['invite'].hash

    if isinstance(event, share_events.ShareUnshareFolderEvent) and \
            isinstance(share_object, (LinkToGroup, GroupInvites)):
        data['rights'] = share_object.rights

    _log_event_raw(event, TSKVMessage.with_special_escaped(**data))

    logged_uids.append(uid)


def _log_share_event_users(event, common_data, event_type_suffix, logged_uids, share_objects):
    base_event_type = _EVENT_TYPES_REVERS_LOOKUP.get(type(event))

    for share_object in flatten(share_objects):
        user_uid, _, _ = _extract_share_object_data(share_object)

        if user_uid in logged_uids:
            continue

        data = common_data.copy()
        data['user_uid'] = user_uid

        if isinstance(share_object, (LinkToGroup, GroupInvites)):
            data['rights'] = share_object.rights

        event_type = '%s-%s' % (base_event_type, event_type_suffix)
        _log_event_raw(event, TSKVMessage.with_special_escaped(**data), event_type=event_type)

        logged_uids.append(user_uid)


def _extract_share_object_data(share_object):
    if isinstance(share_object, Group):
        return share_object.owner, share_object.path, None
    elif isinstance(share_object, LinkToGroup):
        return share_object.uid, share_object.path, None
    elif isinstance(share_object, GroupInvites):
        if share_object.uid != share_object.group.owner:
            uid = share_object.uid
        else:
            uid = None
        return uid, share_object.group.path, share_object.hash
    else:
        # передан объект неизвестного типа
        return None, None, None


def flatten(alist):
    for item in alist:
        if isinstance(item, Iterable) and not isinstance(item, basestring):
            for sub_item in flatten(item):
                yield sub_item
        elif item is not None:
            yield item


def _log_billing_event(event):
    data = event.data
    product = data.pop('product')
    data.update(_extract_product_data(product))
    _log_event_raw(event, TSKVMessage.with_special_escaped(**data))


def _log_space_event(event):
    data = event.data
    catalog = Catalog()
    if data['reason'] and data['reason'] in catalog.all_products():
        product = Product(data['reason'])
        data.update(_extract_product_data(product))

    _log_event_raw(event, TSKVMessage.with_special_escaped(**data))


def _extract_product_data(product):
    data = {'product_id': product.pid, 'product_is_free': product.free()}

    for lang, name in product.get_names().iteritems():
        for log_lang in MAP_PRODUCT_LANG_TO_LOG_LANG[lang]:
            data['product_name_' + log_lang] = name

    if product.period:
        period_chunks = ['%s:%s' % (duration, amount) for duration, amount in product.period.iteritems()]
        data['product_period'] = ','.join(period_chunks)

    return data


def _log_event_default(event):
    _log_event_raw(event, TSKVMessage.with_special_escaped(**event.data))


def _get_event_type(event=None, event_type=None):
    if event is None and event_type is None:
        raise ValueError("Either event or event type must be specified")

    if event_type is None:
        event_type = _EVENT_TYPES_REVERS_LOOKUP.get(type(event))

    return event_type


def _log_event_raw(event, message, event_type=None):
    event_type = _get_event_type(event, event_type)
    log_raw_event(event_type, message)
