# -*- coding: utf-8 -*-

import re
import time

import mpfs.engine.process
from mpfs.common.util.logger import TSKVMessage

SAVE_FILE_FROM_PUBLIC_LOG_MESSAGE_RE = re.compile(r'event_type=fs-copy')
SAVE_FILE_FROM_MAIL_LOG_MESSAGE_RE = re.compile(r'event_type=fs-store|event_type=fs-hardlink-copy')
SAVE_FILE_FROM_WEB_LOG_MESSAGE_RE = re.compile(r'(event_type=fs-store|event_type=fs-hardlink-copy)')
EXTRACT_FILE_FROM_ARCHIVE_LOG_MESSAGE_RE = re.compile(r'(event_type=fs-store|event_type=fs-hardlink-copy)')


def get_pattern(operation):
    """Получить регулярное выражение, соответствующее строке в эвент логе,
     из которой можно получить идентификатор блока Ленты для операции."""
    # TODO: Думаю это всё же лучше перенести в core.lenta..., т.к. слишком специфично.
    from mpfs.core.operations.filesystem.copy import CopyOnDisk, CopyWebDisk, CopyAttachDisk, CopyMailDisk
    from mpfs.core.operations.filesystem.store import ExtractFileFromArchive
    if operation.__class__ in (CopyAttachDisk, CopyOnDisk):
        return SAVE_FILE_FROM_PUBLIC_LOG_MESSAGE_RE
    elif operation.__class__ in (ExtractFileFromArchive,):
        return EXTRACT_FILE_FROM_ARCHIVE_LOG_MESSAGE_RE
    elif operation.__class__ in (CopyMailDisk,):
        return SAVE_FILE_FROM_MAIL_LOG_MESSAGE_RE
    elif operation.__class__ in (CopyWebDisk,):
        return SAVE_FILE_FROM_WEB_LOG_MESSAGE_RE
    # важно возвращать None


class CatchHistoryLogging(object):
    """
    Прицепляет обработчик для записей, которые история событий пишет в лог.

    Позволят отключить запись в лог и получить записи в виде списка.
    Может использоваться как декоратор или как контекст менеджер.
    """

    @classmethod
    def is_logging_disabled(cls):
        """Возвращает включено или отключено сейчас логирование в лог истории."""
        return cls.__disable_logging

    @classmethod
    def is_message_catching_enabled(cls):
        """Возвращает включён или отключён сейчас сбор записей от истории."""
        return cls.__catch_messages

    @classmethod
    def get_messages(cls, pattern=None):
        """Возвращает список сообщений истории собранный с начала контекста.

        При передаче параметра `pattern` будут возвращены только сообщения,
        соответствующие переданному регулярному выражению (или списку).
        Параметр `pattern` может быть не только скомпилированным регулярным
        выражением, но и списком/кортежем скомпилированных регулярных выражений.
        """
        if pattern is None:
            return cls.__messages
        elif isinstance(pattern, (tuple, list)):
            result = []
            for message in cls.__messages:
                for p in pattern:
                    if p.search(message):
                        result.append(message)
                        break
            return result
        else:
            return [message for message in cls.__messages if pattern.search(message)]

    def __init__(self, catch_messages=False, disable_logging=False):
        """
        :param bool catch_messages: Если True, то атрибут контекст-менеджера будет содержать список сообщений
                                    сгенерированных историей.
        :param bool disable_logging: Отключить запись событий в лог. При этом записи до
        """
        self._catch_messages = catch_messages
        self._disable_logging = disable_logging

    def __call__(self, f):
        """
        Декоратор возвращающий метод обёрнутый в контекст менеджер.

        :param f: Метод который следует обернуть.
        :return: Декоратор.
        """

        def wrapper(*args, **kwargs):
            with self:
                result = f(*args, **kwargs)
            return result

        return wrapper

    def __enter__(self):
        """Метод необходимый для поддержки протокола контекст-менеджера."""
        self.__class__.__catch_messages = self._catch_messages
        self.__class__.__disable_logging = self._disable_logging
        return self

    def __exit__(self, exc_type, exc_val, exc_tb):
        """
        Замалчивает и логирует или пробрасывает дальше по стеку исключение.

        Метод необходимый для поддержки протокола контекст-менеджера.
        """
        # Возвращаем всё параметры логирования на место.
        self.__class__.__catch_messages = False
        self.__class__.__disable_logging = False
        # Дропаем ссылку на список, ибо все кто надо уже её сохранили.
        self.__class__.__messages = []
        return False

    __disable_logging = False
    """Выключатьель логирования в лог истории событиий."""

    __catch_messages = False
    """Собирать сообщения истории событий в списочек."""

    __messages = []
    """Список сообщений сгенерированных лентой. Пишется если включён флаг `__catch_messages`."""

    @classmethod
    def append_message(cls, msg):
        """
        Добавляет запись в список собранных контекст-менеджером сообщений истории.

        !!!ВНИМАНИЕ: Этот метод руками лучше не дёргать. Его дёргает логгер истории и тольпо поэтому он не приватный.
        """
        if cls.is_message_catching_enabled():
            cls.__messages.append(msg)


def get_log_raw_event_message(event_type, data):
    timestamp = int(time.time())
    prefix_message = TSKVMessage(
        tskv_format='ydisk-event-history-log',
        event_type=event_type,
        unixtime=timestamp,
        req_id=(mpfs.engine.process.get_cloud_req_id() or '-'),
        hostname=mpfs.engine.process.hostname().split('.')[0]
    )
    if isinstance(data, basestring):
        postfix_message = data
    elif isinstance(data, TSKVMessage):
        postfix_message = data
    elif isinstance(data, dict):
        postfix_message = TSKVMessage.with_special_escaped(**data)
    else:
        raise TypeError('Unexpected `data` type. Expected dict or string, got %s.' % data.__class__.__name__)

    message = u'tskv\t%s\t%s' % (prefix_message, postfix_message)
    return message


def log_raw_event(event_type, data):
    message = get_log_raw_event_message(event_type, data)
    log_raw_event_message(message=message)


def log_raw_event_message(message):
    _log_raw_event_message(message)


def _log_raw_event_message(message):
    # FIXME: если импортировать эту функцию вне этого модуля, то у меня не
    # FIXME: получается ее замокать, поэтому она сделана приватной
    CatchHistoryLogging.append_message(message)
    if not CatchHistoryLogging.is_logging_disabled():
        mpfs.engine.process.get_event_history_log().info(message)
