# -*- coding: utf-8 -*-
"""

MPFS
CORE

Фабрика папок и файлов (:

"""
import exceptions
import itertools

from multiprocessing.pool import ThreadPool
from collections import defaultdict, OrderedDict

import mpfs.engine.process

import mpfs.core.filesystem.resources.narod
import mpfs.core.filesystem.resources.notes
import mpfs.core.filesystem.resources.root
import mpfs.core.filesystem.resources.disk
import mpfs.core.filesystem.resources.fotki
import mpfs.core.filesystem.resources.yavideo
import mpfs.core.filesystem.resources.mail
import mpfs.core.filesystem.resources.trash
import mpfs.core.filesystem.resources.stock
import mpfs.core.filesystem.resources.hidden
import mpfs.core.filesystem.resources.attach
import mpfs.core.filesystem.resources.misc
import mpfs.core.filesystem.resources.mulca
import mpfs.core.filesystem.resources.photounlim
import mpfs.core.filesystem.resources.photostream
import mpfs.core.filesystem.resources.yareader
import mpfs.core.filesystem.resources.additional
import mpfs.core.filesystem.resources.client

import mpfs.core.services.narod_service
import mpfs.core.services.notes_storage_service
import mpfs.core.services.disk_service
import mpfs.core.services.fotki_service
import mpfs.core.services.yavideo_service
import mpfs.core.services.mail_service
import mpfs.core.services.trash_service
import mpfs.core.services.stock_service
import mpfs.core.services.attach_service
import mpfs.core.services.misc_service
import mpfs.core.services.hidden_files_service
import mpfs.core.services.photounlim_service
import mpfs.core.services.photostream_service
import mpfs.core.services.mulca_service
import mpfs.core.services.yareader_service
import mpfs.core.services.additional_files_service
import mpfs.core.services.client_service

from mpfs.config import settings
from mpfs.common import errors
from mpfs.common.errors import share, ResourceNotFound
from mpfs.common.util.mappers import ListMapper
from mpfs.core.filesystem.constants import NOTES_STORAGE_PATH
from mpfs.core.filesystem.resources.disk import DiskFolder, DiskFile
from mpfs.core.filesystem.symlinks import Symlink
from mpfs.core.office.static import OfficeAccessStateConst
from mpfs.core.services.search_service import DiskSearch, SearchService, SearchResults
from mpfs.core.address import Address
from mpfs.core.user.constants import (
    ADDITIONAL_AREA_PATH,
    DISK_AREA,
    PHOTOUNLIM_AREA,
    PHOTOUNLIM_AREA_PATH,
    CLIENT_AREA,
    CLIENT_AREA_PATH,
    PUBLIC_UID,
)

from mpfs.metastorage.mongo.binary import Binary
from mpfs.metastorage.mongo.collections.all_user_data import AllUserDataCollection
from mpfs.core.filesystem.dao.file import FileDAO
from mpfs.core.filesystem.dao.folder import FolderDAO


log = mpfs.engine.process.get_default_log()
error_log = mpfs.engine.process.get_error_log()
requests_log = mpfs.engine.process.get_requests_log()


MAX_THREADS_LIMIT = 20

SYS_FOLDER_MAP = OrderedDict((
    ('/', {
        'classes': {
            'folder': mpfs.core.filesystem.resources.root.RootFolder,
        },
        'service': mpfs.core.services.common_service.Service,
    }),
    ('/attach', {
        'classes': {
            'folder': mpfs.core.filesystem.resources.attach.AttachFolder,
            'file': mpfs.core.filesystem.resources.attach.AttachFile,
        },
        'service': mpfs.core.services.attach_service.Attach,
    }),
    ('/disk', {
        'classes': {
            'folder': mpfs.core.filesystem.resources.disk.DiskFolder,
            'file': mpfs.core.filesystem.resources.disk.DiskFile,
        },
        'service': mpfs.core.services.disk_service.Disk,
    }),
    ('/fotki', {
        'classes': {
            'folder': mpfs.core.filesystem.resources.fotki.FotkiFolder,
            'file': mpfs.core.filesystem.resources.fotki.FotkiFile,
        },
        'service': mpfs.core.services.fotki_service.Fotki,
    }),
    ('/hidden', {
        'classes': {
            'folder': mpfs.core.filesystem.resources.hidden.HiddenFolder,
            'file': mpfs.core.filesystem.resources.hidden.HiddenFile,
        },
        'service': mpfs.core.services.hidden_files_service.Hidden,
    }),
    ('/lnarod', {
        'classes': {
            'folder': mpfs.core.filesystem.resources.narod.LegacyNarodFolder,
            'file': mpfs.core.filesystem.resources.narod.LegacyNarodFile,
        },
        'service': mpfs.core.services.narod_service.LegacyNarod,
    }),
    ('/mail', {
        'classes': {
            'folder': mpfs.core.filesystem.resources.mail.MailFolder,
            'file': mpfs.core.filesystem.resources.mail.MailFile,
        },
        'service': mpfs.core.services.mail_service.Mail,
    }),
    ('/mulca', {
        'classes': {
            'folder': mpfs.core.filesystem.resources.mulca.MulcaFolder,
            'file': mpfs.core.filesystem.resources.mulca.MulcaFile,
        },
        'service': mpfs.core.services.mulca_service.Mulca,
    }),
    ('/narod', {
        'classes': {
            'folder': mpfs.core.filesystem.resources.narod.NarodFolder,
            'file': mpfs.core.filesystem.resources.narod.NarodFile,
        },
        'service': mpfs.core.services.narod_service.Narod,
    }),
    (PHOTOUNLIM_AREA_PATH, {
        'classes': {
            'folder': mpfs.core.filesystem.resources.photounlim.PhotounlimFolder,
            'file': mpfs.core.filesystem.resources.photounlim.PhotounlimFile,
        },
        'service': mpfs.core.services.photounlim_service.Photounlim,
    }),
    (ADDITIONAL_AREA_PATH, {
        'classes': {
            'folder': mpfs.core.filesystem.resources.additional.AdditionalFolder,
            'file': mpfs.core.filesystem.resources.additional.AdditionalFile,
        },
        'service': mpfs.core.services.additional_files_service.AdditionalFilesService,
    }),
    (CLIENT_AREA_PATH, {
        'classes': {
            'folder': mpfs.core.filesystem.resources.client.ClientFolder,
            'file': mpfs.core.filesystem.resources.client.ClientFile,
        },
        'service': mpfs.core.services.client_service.ClientService,
    }),
    ('/photostream', {
        'classes': {
            'folder': mpfs.core.filesystem.resources.photostream.PhotostreamFolder,
            'file': mpfs.core.filesystem.resources.photostream.PhotostreamFile,
        },
        'service': mpfs.core.services.photostream_service.Photostream,
    }),
    ('/settings', {
        'classes': {
            'folder': mpfs.core.filesystem.resources.misc.MiscFolder,
            'file': mpfs.core.filesystem.resources.misc.MiscFile,
        },
        'service': mpfs.core.services.misc_service.Misc,
    }),
    ('/share', {
        'classes': {
            'folder': mpfs.core.filesystem.resources.stock.StockFolder,
            'file': mpfs.core.filesystem.resources.stock.StockFile,
        },
        'service': mpfs.core.services.stock_service.Stock,
    }),
    ('/trash', {
        'classes': {
            'folder': mpfs.core.filesystem.resources.trash.TrashFolder,
            'file': mpfs.core.filesystem.resources.trash.TrashFile,
        },
        'service': mpfs.core.services.trash_service.Trash,
    }),
    ('/yareader', {
        'classes': {
            'folder': mpfs.core.filesystem.resources.yareader.YaReaderFolder,
            'file': mpfs.core.filesystem.resources.yareader.YaReaderFile,
        },
        'service': mpfs.core.services.yareader_service.YaReader,
    }),
    ('/yavideo', {
        'classes': {
            'folder': mpfs.core.filesystem.resources.yavideo.YaVideoFolder,
            'file': mpfs.core.filesystem.resources.yavideo.YaVideoFile,
        },
        'service': mpfs.core.services.yavideo_service.YaVideo,
    }),
    (NOTES_STORAGE_PATH, {
        'classes': {
            'folder': mpfs.core.filesystem.resources.notes.NotesFolder,
            'file': mpfs.core.filesystem.resources.notes.NotesFile,
        },
        'service': mpfs.core.services.notes_storage_service.NotesStorageService,
    }),
))


def get_collection_for_root_folder(root_folder):
    for folder, attrs in SYS_FOLDER_MAP.iteritems():
        if folder == root_folder:
            if 'service' in attrs:
                control = getattr(attrs['service'], 'control', None)
                if control is not None:
                    return control.name
    raise LookupError('collection for folder "%s" not found' % root_folder)


def get_folder_for_collection(coll_name):
    for folder, attrs in SYS_FOLDER_MAP.iteritems():
        if 'service' in attrs:
            control = getattr(attrs['service'], 'control', None)
            if control is not None and control.name == coll_name:
                return folder
    raise LookupError('folder for collection "%s" not found' % coll_name)


def get_service_by_root_folder_path(root_folder_path):
    for folder, attrs in SYS_FOLDER_MAP.iteritems():
        if folder == root_folder_path and 'service' in attrs:
            return attrs['service']()
    raise LookupError('service for root folder path "%s" not found' % root_folder_path)


def get_sys_folder_info(address):
    try:
        return SYS_FOLDER_MAP[address.storage_path]
    except exceptions.KeyError:
        raise errors.ServiceNotFound(address.storage_path)


def get_service(address):
    if address.is_root:
        raise errors.PermissionDenied
    return get_sys_folder_info(address)['service']()


def get_class(address, force_dir=False):
    classes = get_sys_folder_info(address)['classes']
    if (address.is_system or address.is_folder or force_dir):
        return classes['folder']
    else:
        return classes['file']


def get_resource_class_by_sys_folder(uid, path, type_):
    address = Address.Make(uid, path)
    try:
        sys_folder_info = get_sys_folder_info(address)
    except errors.ServiceNotFound:
        raise errors.ResourceNotFound()

    try:
        resource_class = sys_folder_info['classes'][type_]
    except KeyError:
        raise errors.ResourceNotFound()
    return resource_class


def _select_disk_resource_class(type_, data):
    """Выбрать подходящий класс дисковых ресурсов на основе
    типа ``type`` и данных ``data``.

    :type type_: str
    :type data: dict
    """
    # цикл. импорт
    from mpfs.core.filesystem.resources.share import SharedRootFolder, SharedFolder, SharedFile
    from mpfs.core.filesystem.resources.group import GroupRootFolder, GroupFolder, GroupFile

    if 'link' in data:
        resource_class = SharedFolder
        if type_ == 'file':
            resource_class = SharedFile
        elif data['link'].path == data['key']:
            resource_class = SharedRootFolder
    elif 'group' in data:
        resource_class = GroupFolder
        if type_ == 'file':
            resource_class = GroupFile
        elif data['group'].path == data['key']:
            resource_class = GroupRootFolder
    else:
        resource_class = DiskFolder
        if type_ == 'file':
            resource_class = DiskFile

    return resource_class


def select_resource_class(data):
    """Выбрать подходящий класс ресурсов на основе данных ``data``.

    :type data: dict
    """

    type_ = 'folder'
    if data['type'] == 'file':
        type_ = 'file'

    resource_class = get_resource_class_by_sys_folder(data['uid'], data['key'], type_)

    if resource_class in (DiskFolder, DiskFile):
        return _select_disk_resource_class(type_, data)

    return resource_class


def set_group_or_link(uid, data, find_links_correctly=False):
    # цикл. импорт
    from mpfs.core.social.share.group import Group

    group = None
    try:
        group = Group.find(uid=data['uid'], path=data['key'], find_links_correctly=find_links_correctly)
    except share.GroupNotFound:
        if uid != data['uid']:
            raise

    link = None
    if group:
        try:
            link = group.get_link_by_uid(uid)
        except errors.share.ShareNotFound:
            if uid != data['uid']:
                raise

    if link:
        data['link'] = link
    elif group:
        data['group'] = group


def get_service_class_by_sys_folder(uid, path):
    address = Address.Make(uid, path)
    try:
        service_class = get_sys_folder_info(address)['service']
    except KeyError:
        raise errors.ResourceNotFound()
    return service_class


def get_resource_from_doc(user_principal, doc, find_links_correctly=False):
    if user_principal is None:
        user_principal = doc['uid']
    service_class = get_service_class_by_sys_folder(user_principal, doc['key'])

    uid = user_principal if user_principal != PUBLIC_UID else doc['uid']
    data = service_class().get_data_for_uid(uid, doc)
    set_group_or_link(uid, data, find_links_correctly=find_links_correctly)

    resource_class = select_resource_class(data)
    return resource_class.from_dict(data, user_principal=user_principal)


def _get_resource_by_spec(uid, spec, enable_collections=None):
    if 'uid' in spec:
        db_result = AllUserDataCollection(enable_collections=enable_collections).find_one_on_uid_shard(spec['uid'], spec)
    else:
        db_result = AllUserDataCollection(enable_collections=enable_collections).find_one(spec)
    if not db_result:
        raise errors.ResourceNotFound()
    return get_resource_from_doc(uid, db_result.record)


def get_resource_by_file_id(uid, file_id, enable_collections=None, user_principal=None):
    user_principal = user_principal or uid
    return _get_resource_by_spec(
        user_principal, {'uid': uid, 'data.file_id': file_id}, enable_collections=enable_collections)


def get_resource_by_uid_and_office_doc_short_id(uid, owner_uid, office_doc_short_id, enable_collections=('user_data',),
                                                skip_access_check=False, access_check_func=any):
    symlinks = Symlink.find_by_office_doc_short_id(owner_uid, office_doc_short_id)
    if not symlinks:
        raise ResourceNotFound()

    resource = get_not_removed_resource_by_file_id(owner_uid, symlinks[0].get_file_id())

    if (not skip_access_check and
            uid != owner_uid and
            not access_check_func(symlink.get_office_access_state() == OfficeAccessStateConst.ALL
                                  for symlink in symlinks)):
        raise ResourceNotFound()

    return resource


def get_not_removed_resource_by_file_id(uid, file_id, is_for_public=False):
    """
    Ищем файл по file_id по всем коллекциям, кроме hidden_data и trash
    """
    user_principal = PUBLIC_UID if is_for_public else uid
    if mpfs.engine.process.usrctl().is_user_in_postgres(uid):
        item = FileDAO().find_by_file_id_with_oldest_version_not_removed(uid, file_id)
        if item is None:
            item = FolderDAO().find_by_file_id_with_oldest_version_not_removed(uid, file_id)
        if item is None:
            raise errors.ResourceNotFound()
        return get_resource_from_doc(user_principal, item.get_mongo_representation(), find_links_correctly=True)
    else:
        return get_resource_by_file_id(
            uid, file_id,
            enable_collections={c for c in AllUserDataCollection.DATA_COLLECTIONS if c not in ('trash', 'hidden_data')},
            user_principal=user_principal
        )


def get_resource_by_resource_id(uid, resource_id, enable_collections=('user_data', 'trash')):
    return _get_resource_by_spec(
        uid,
        {'uid': resource_id.uid, 'data.file_id': resource_id.file_id},
        enable_collections=enable_collections
    )


def get_resource_by_path(uid, path):
    return _get_resource_by_spec(
        uid, {'uid': uid, 'key': path, 'path': path})


def get_resource_by_address(uid, address):
    return _get_resource_by_spec(
        uid, {'uid': address.uid, 'key': address.path, 'path': address.path})


def get_resource(uid, address, data=None, version=None, **kwargs):
    if data is None:
        data = {}
    if not isinstance(address, Address):
        try:
            address = Address(address)
        except errors.AddressError:
            address = Address.Make(uid, address)

    if address.is_system or address.storage_path in settings.folder_types['static']:
        class_ = get_class(address, force_dir=data.get('type') == 'dir')
        return class_(uid, address, data=data, version=version, **kwargs)
    elif address.storage_path in settings.folder_types['dynamic']:
        address.clean_id()
        service = get_service(address)
        return service.get_resource(uid, address, version=version, **kwargs)
    else:
        raise errors.ResourceNotFound()


def get_resource_by_uid_and_hid(uid, hid, enable_collections=('user_data', 'photounlim_data')):
    return _get_resource_by_spec(uid, {'uid': uid, 'hid': Binary(str(hid))}, enable_collections=enable_collections)


def get_resources(uid, addresses, available_service_ids=None):
    """
    Получение пачки ресурсов из разных сервисов/коллекций

    addresses - список адресов
    available_service_ids - список id сервисов, к которым идет обращение. Default:
        ['/disk', '/attach', '/share', '/yavideo', '/yareader', '/trash',
        '/hidden', '/', '/mulca', '/narod', '/lnarod', '/mail', '/settings', '/fotki', '/photostream', '/client']
    """
    if available_service_ids is None:
        available_service_ids = SYS_FOLDER_MAP.keys()

    # разбиваем ресурсы по сервисам
    splited_addresses = defaultdict(list)
    for i, address in enumerate(addresses):
        service_id = address.storage_path
        if service_id in available_service_ids:
            splited_addresses[address.storage_path].append(address)

    # получаем ресурсы из сервисов
    unsorted_resources = []
    for _, service_addresses in splited_addresses.iteritems():
        service_obj = get_service(service_addresses[0])
        unsorted_resources += service_obj.get_resources(uid, service_addresses)

    # сортируем ресурсы в порядке, полученном из addresses
    def key_getter(resource):
        """Для шаренных ресурсов отдаем visible_address.id"""
        from mpfs.core.filesystem.resources.share import SharedResource
        if isinstance(resource, SharedResource):
            return resource.visible_address.id
        else:
            return resource.address.id
    address_map = ListMapper([a.id for a in addresses])
    resources = address_map.map(unsorted_resources, key_getter=key_getter)
    return [r for r in resources if r is not None]


def get_resources_with_dummies(uid, addresses, dummies):
    service_obj = get_service(addresses[0])
    return service_obj.get_resources_with_dummies(uid, addresses, dummies)


def search(resource, query, **kwargs):
    """
    Search string occurrences in specified resource.

    :param resource:
    :param query: string Search query.
    :param kwargs:
    :return:
    """
    if resource.address.storage_name in ('disk',):
        return DiskSearch().search(resource, query, search_folders=['disk', 'photounlim'], **kwargs)
    if resource.address.storage_name in ('trash', 'attach', ):
        return DiskSearch().search(resource, query, **kwargs)
    elif resource.address.storage_name in ('mail',):
        # FIXME: Скорее всего этот код вообще не работает,
        # FIXME: тк далее по коду идет обращение к атрибуту query у результатов
        return SearchService().listing(resource, query, **kwargs)
    else:
        ret = SearchResults()
        ret.query = query
        return ret


def geo_search(uid, start_date, end_date, latitude=None, longitude=None, distance=None, count_lost_results=False):
    return DiskSearch().geo_search(uid, start_date, end_date, latitude, longitude, distance, count_lost_results)


def is_unique(address):
    return get_service(Address(address)).unique_items


def reload_resources_for_user(uid, resources):
    """
    Перезагружает ресурсы для нужного uid-a с сохранением порядка.

    :param str uid: uid пользователя, от чего имени хотим получить ресурсы.
    :param list resources: ресурсы, которые нужно переполучить.
    :return list: Список ресурсов с сохранением исходного порядка.
                  Если ресурс невозможно получить, то на его месте будет None.
    """
    ret = [None] * len(resources)
    if resources:
        result_resources = []
        order_file_ids = [r.meta['file_id'] for r in resources]
        # "Входные папки" мы не можем перезагрузить, т.к. они не являются общими
        import mpfs.core.filesystem.resources.share
        import mpfs.core.filesystem.resources.group
        not_reloadable_resources = (
            mpfs.core.filesystem.resources.share.SharedRootFolder,
            mpfs.core.filesystem.resources.group.GroupRootFolder
        )
        resources = [r for r in resources if not isinstance(r, not_reloadable_resources)]
        # разбиваем ресурсы по владельцам
        uid_resources_map = defaultdict(list)
        for resource in resources:
            uid_resources_map[resource.uid].append(resource)

        # свои оставляем как есть
        if uid in uid_resources_map:
            result_resources += uid_resources_map.pop(uid)

        # обрабатываем чужие файлы
        ## собираем чужие групповые(GroupFile) ресурсы
        group_map = defaultdict(list)
        for foreign_uid, foreign_uid_resources in uid_resources_map.iteritems():
            for resource in foreign_uid_resources:
                if resource.meta.get('group'):
                    gid = resource.meta['group']['gid']
                    group_map[gid].append(resource)

        ## преобразуем пути в "свои" и получаем шаренные(SharedFile) ресурсы
        from mpfs.core.social.share import LinkToGroup
        uid_shared_addresses = []
        for group_link in LinkToGroup.load_all(uid)['id'].itervalues():
            gid = group_link.gid
            for resource in group_map.get(gid, []):
                uid_shared_folder_path = group_link.path
                owner_shared_folder_path = group_link.group.get_folder().path
                if uid_shared_folder_path == owner_shared_folder_path:
                    uid_shared_resource_path = resource.path
                else:
                    uid_shared_resource_path = resource.path.replace(owner_shared_folder_path, uid_shared_folder_path, 1)
                uid_shared_addresses.append(Address.Make(uid, uid_shared_resource_path))
        if uid_shared_addresses:
            result_resources += get_resources(uid, uid_shared_addresses, available_service_ids=['/disk'])

        # ставим ресурс на свое место в ответе
        for resource in result_resources:
            try:
                ret_index = order_file_ids.index(resource.meta['file_id'])
            except ValueError:
                # не нашли такой file_id
                pass
            else:
                ret[ret_index] = resource
    return ret


def _get_owner_resources(uid, owner_uid, uid_resource_ids, mpfs_collections, enable_service_ids, enable_optimization):
    all_user_data_coll = AllUserDataCollection(enable_collections=mpfs_collections)
    uid_file_ids = [i.file_id for i in uid_resource_ids]
    spec = {'uid': owner_uid, 'data.file_id': {'$in': uid_file_ids}}
    if enable_optimization:
        uid_raw_resources = (i.record for i in all_user_data_coll.find_on_uid_shard(owner_uid, spec))
    else:
        uid_raw_resources = (i.record for i in all_user_data_coll.find_on_uid_shard(owner_uid, spec, fields=['key']))
    uid_raw_resources = list(uid_raw_resources)

    if not uid_raw_resources:
        return []

    if enable_optimization:
        result = []
        for doc in uid_raw_resources:
            try:
                resource = get_resource_from_doc(uid, doc, find_links_correctly=True)
                result.append(resource)
            except (errors.share.GroupNotFound, errors.share.ShareNotFound, errors.ResourceNotFound):
                pass
        return result
    else:
        addresses = [Address.Make(owner_uid, raw['key']) for raw in uid_raw_resources]
        resources = get_resources(owner_uid, addresses, available_service_ids=enable_service_ids)
        resources = [r for r in resources if r is not None]

        if owner_uid == uid:
            return resources
        else:
            return [r for r in reload_resources_for_user(uid, resources) if r is not None]


def get_resources_by_resource_ids(uid, resource_ids, enable_service_ids=None, enable_optimization=False,
                                  enable_multithreading=False):
    """
    Получение ресурсов по ResourceId

    :param uid: от чьего имени получаем ресурсы
    :param resource_ids: список объектов ResourceId
    :param enable_service_ids: список id сервисов, к которым идет обращение. Доступные см. в SYS_FOLDER_MAP -> mpfs_collection_name
    Если ресурс невозможно получить, возвращает в списке `None`
    :param enable_optimization: отключить повторное хождение в базу за ресурсами.
    :param enable_multithreading: использовать мультитрединг для распаралеливания запросов в БД.
    """
    unsorted_resources = find_all_resources_by_resource_ids(uid, resource_ids, enable_service_ids=enable_service_ids,
                                                            enable_optimization=enable_optimization,
                                                            enable_multithreading=enable_multithreading)
    # отдаем в запрошенном порядке
    file_ids = [i.file_id for i in resource_ids]
    lm = ListMapper(file_ids)
    return lm.map(unsorted_resources, key_getter=lambda i: i.meta['file_id'])


def find_all_resources_by_resource_ids(uid, resource_ids, enable_service_ids=None, enable_optimization=False,
                                       enable_multithreading=False, filter_duplicates=True):
    """
    Найти все ресурсы по ResourceIds

    Отличается от get_resources_by_resource_ids тем, что ответ неупорядочен и может не фильтровать дубликаты
    resource_id. Если ресурса не было в базе, то вместо None он попросту будет отсутствовать в результирующем списке.
    :param filter_duplicates: фильтровать оставлять все файлы с одинаковым resource_id.
    """
    # подготавливаем названия сервисов и коллекций, в которые будем ходить
    available_service_ids = {k: get_collection_for_root_folder(k) for k in
                             ('/attach', '/disk', '/trash', PHOTOUNLIM_AREA_PATH, NOTES_STORAGE_PATH)}
    if enable_service_ids:
        mpfs_collections = []
        for service_id in enable_service_ids:
            if service_id in available_service_ids:
                mpfs_collections.append(available_service_ids[service_id])
            else:
                raise errors.ServiceNotFound('Unsupported service_id in `enable_service_ids`. Got: "%s" choice: %s' % (
                    service_id, available_service_ids.keys()))
    else:
        mpfs_collections = available_service_ids.values()
        enable_service_ids = available_service_ids.keys()

    # сортировка упорядочивает адреса по uid-ам
    sorted_resource_ids = sorted(resource_ids, key=lambda i: i.uid)
    grouped_resource_ids = []
    for owner_uid, owner_uid_items_iter in itertools.groupby(sorted_resource_ids, lambda x: x.uid):
        grouped_resource_ids.append((owner_uid, list(owner_uid_items_iter)))
    # не используем мультитрединг, если только один владелец
    if len(grouped_resource_ids) <= 1:
        enable_multithreading = False
    unsorted_resources = []
    if enable_multithreading:
        pool = ThreadPool(min(len(grouped_resource_ids), MAX_THREADS_LIMIT))
        results = pool.map(
            lambda x: _get_owner_resources(uid, x[0], x[1], mpfs_collections, enable_service_ids, enable_optimization),
            grouped_resource_ids
        )
        pool.close()
        pool.join()
        for result in results:
            unsorted_resources.extend(result)
    else:
        for owner_uid, uid_resource_ids in grouped_resource_ids:
            unsorted_resources.extend(
                _get_owner_resources(uid, owner_uid, uid_resource_ids, mpfs_collections, enable_service_ids,
                                     enable_optimization)
            )

    if not filter_duplicates:
        return unsorted_resources
    filtered_resources = []
    encountered_file_ids = set()
    for resource in unsorted_resources:
        if resource.meta['file_id'] in encountered_file_ids:
            continue
        encountered_file_ids.add(resource.meta['file_id'])
        filtered_resources.append(resource)
    return filtered_resources


def iter_resources_by_stids(stids):
    """Получить ресурсы по stid-ам"""
    if not isinstance(stids, list):
        raise TypeError()
    if not stids:
        raise StopIteration()

    spec = {'data.stids': {
        '$elemMatch': {'stid': {'$in': stids}}
    }}
    for db_result in AllUserDataCollection().find(spec):
        yield get_resource_from_doc(None, db_result.record)
