# -*- coding: utf-8 -*-
"""

Занимается тем, что:
    * Получает json представление репорта ПО
    * Приводит json к объетам MPFS
    * Находит соотвествующий обработчик репорта
    * Запускает обработку репорта
"""
import collections

from mpfs.config import settings
from mpfs.engine.process import get_default_log
from mpfs.core.file_recovery.logic.reports import RecoveryReport
from mpfs.core.file_recovery.logic.processors import (
    FileNotFoundReportProcessor,
    HashConflictReportProcessor
)
from mpfs.core.file_recovery.errors import UnsupportedRecoveryReport

default_log = get_default_log()
FILE_RECOVERY_ENABLE_REPORT_PROCESSING = settings.file_recovery['enable_report_processing']


AVAILABLE_REPORT_PROCESSORS_MAP = {
    rp.alias: rp
    for rp in (
        FileNotFoundReportProcessor,
        HashConflictReportProcessor
    )
}


def get_report_processor_class(report):
    if report.name not in AVAILABLE_REPORT_PROCESSORS_MAP:
        raise UnsupportedRecoveryReport("Got: %r" % report)
    return AVAILABLE_REPORT_PROCESSORS_MAP[report.name]


def process_raw_report(raw_json):
    """Обработать один json-репорт

    :rtype: Exception
    """
    report_name = None
    process_status = None
    put_status = None

    try:
        report = RecoveryReport.from_json(raw_json)
        report_processor_class = get_report_processor_class(report)
    except Exception as e:
        report_name = 'Unknown'
        process_status = "Exc %s" % e.__class__.__name__
    else:
        report_name = report_processor_class.alias
        try:
            process_status, put_status = report_processor_class(report).process()
        except Exception as e:
            process_status = "Exc %s" % e.__class__.__name__
        else:
            process_status = process_status.value
            put_status = put_status.value if put_status else None

    default_log.info("%s %s %s %s", report_name, process_status, put_status, raw_json)
    return report_name, process_status, put_status


def process_raw_reports(raw_json_reports):
    """Обработать поток json-репортов"""
    if not FILE_RECOVERY_ENABLE_REPORT_PROCESSING:
        get_default_log().info("Report processing disabled")
        return

    counter = collections.Counter()
    for raw_json in raw_json_reports:
        report_name, process_status, put_status = process_raw_report(raw_json)
        counter['Report %s' % report_name] += 1
        counter['Status %s' % process_status] += 1
        counter['Put to recovery %s' % put_status] += 1
        counter['_total_'] += 1
    default_log.info("Processed %i reports." % counter['_total_'])
    return counter
