# -*- coding: utf-8 -*-
from enum import Enum

from mpfs.engine.process import get_default_log
from mpfs.common.errors import ResourceNotFound, NotFile
from mpfs.core.file_recovery.errors import ResourceIDMissmatch
from mpfs.core.factory import get_resource_by_resource_id, get_resource_by_address
from mpfs.core.filesystem.hardlinks.controllers import HardlinksController
from mpfs.core.services.previewer_service import Previewer
from mpfs.core.services.restore_db_service import restore_db_service
from mpfs.core.services.mulca_service import Mulca

default_log = get_default_log()
mulca_service = Mulca()
previewer_service = Previewer()


class PutOnRecoveryStatus(Enum):
    already_public = 'already_public'
    already_personal = 'already_personal'
    put_public = 'put_public'
    put_personal = 'put_personal'


def put_file_on_recovery(resource):
    """Поставить ресурс на восстановление"""
    if not resource.is_file:
        raise NotFile()

    checksums = resource.get_checksums()
    if restore_db_service.is_public_exists(checksums.hid):
        status = PutOnRecoveryStatus.already_public
    elif restore_db_service.is_personal_exists(resource.uid, checksums.hid):
        status = PutOnRecoveryStatus.already_personal
    else:
        if (resource.is_group or resource.is_shared or
                not HardlinksController.is_one_owner(resource.uid, checksums.hid)):
            status = PutOnRecoveryStatus.put_public
            restore_db_service.put_public(checksums)
        else:
            status = PutOnRecoveryStatus.put_personal
            restore_db_service.put_personal(resource.uid, checksums)

    default_log.info("%r %r", status, checksums)
    return status


class ProcessStatus(Enum):
    already_on_recovery = 'already_on_recovery'
    no_file_in_storage = 'no_file_in_storage'
    false_alarm = 'false_alarm'
    hash_missmatch = 'hash_missmatch'
    storage_hash_missmatch = 'storage_hash_missmatch'
    external_hash_missmatch = 'external_hash_missmatch'
    mpfs_hash_missmatch = 'mpfs_hash_missmatch'


class BaseReportProcessor(object):
    alias = None

    def __init__(self, report):
        self.report = report

    def process(self):
        raise NotImplementedError()

    def _load_file_resource(self):
        """Получаем файл-ресурс, который хотим восстановить"""
        try:
            resource = get_resource_by_address(self.report.address.uid, self.report.address)
        except ResourceNotFound:
            if not self.report.resource_id:
                raise
            resource = get_resource_by_resource_id(self.report.resource_id.uid, self.report.resource_id)
        else:
            if self.report.resource_id and resource.resource_id != self.report.resource_id:
                raise ResourceIDMissmatch()
        if not resource.is_file:
            raise NotFile()
        return resource


class FileNotFoundReportProcessor(BaseReportProcessor):
    """Обработчик репорта о ненайденом файле

    Cторадж отдает 404
    """
    alias = 'download_file_404'

    def process(self):
        resource = self._load_file_resource()
        resource_checksums = resource.get_checksums()
        put_status = None
        if restore_db_service.is_exists(resource.uid, resource_checksums.hid):
            return ProcessStatus.already_on_recovery, put_status

        report_checksums = self.report.external_mpfs_checksums or self.report.external_checksums
        if resource_checksums != report_checksums:
            return ProcessStatus.hash_missmatch, put_status

        if mulca_service.is_file_exist(resource.file_mid()):
            return ProcessStatus.false_alarm, put_status
        else:
            put_status = put_file_on_recovery(resource)
            return ProcessStatus.no_file_in_storage, put_status


class HashConflictReportProcessor(BaseReportProcessor):
    """Обработчик репорта о несовпадающих хешах

    ПО пересчитывает хеши файла и сравнивает их с хешами из индекса ПО
    """
    alias = 'hash_conflict'

    def process(self):
        resource = self._load_file_resource()
        resource_checksums = resource.get_checksums()
        put_status = None
        if restore_db_service.is_exists(resource.uid, resource_checksums.hid):
            return ProcessStatus.already_on_recovery, put_status

        report_checksums = self.report.external_checksums
        if resource_checksums == report_checksums:
            return ProcessStatus.false_alarm, put_status

        if not mulca_service.is_file_exist(resource.file_mid()):
            put_status = put_file_on_recovery(resource)
            return ProcessStatus.no_file_in_storage, put_status

        storage_checksums = previewer_service.get_file_checksums(resource.file_mid())
        if resource_checksums == storage_checksums:
            # ПО отдает другой хеш.
            return ProcessStatus.external_hash_missmatch, put_status
        else:
            # Побитый стораджем файл.
            put_status = put_file_on_recovery(resource)
            return ProcessStatus.storage_hash_missmatch, put_status
