# -*- coding: utf-8 -*-
import types

from mpfs.common.util import from_json
from mpfs.core.address import Address, ResourceId
from mpfs.core.filesystem.hardlinks.common import FileChecksums


class RecoveryReport(object):
    def __init__(self, name, address, external_checksums=None, resource_id=None, external_mpfs_checksums=None):
        if not isinstance(address, Address):
            raise TypeError("address should be `Address`. Got %r" % address)
        if not isinstance(external_checksums, (types.NoneType, FileChecksums)):
            raise TypeError("external_checksums should be `FileChecksums` or None. Got %r" % external_checksums)
        if not isinstance(resource_id, (types.NoneType, ResourceId)):
            raise TypeError("resource_id should be `ResourceId` or None. Got %r" % resource_id)
        if not isinstance(external_mpfs_checksums, (types.NoneType, FileChecksums)):
            raise TypeError("external_mpfs_checksums should be `FileChecksums` or None. Got %r" % external_mpfs_checksums)

        self.name = name
        self.address = address
        self.external_checksums = external_checksums
        self.resource_id = resource_id
        self.external_mpfs_checksums = external_mpfs_checksums

    @classmethod
    def from_json(cls, raw_json):
        raw_report = from_json(raw_json)

        try:
            address = Address.Make(raw_report['uid'], raw_report['mpfs_path'])
        except Exception:
            address = None

        try:
            resource_id = ResourceId.parse(raw_report['resource_id'])
        except Exception:
            resource_id = None

        try:
            external_checksums = FileChecksums(raw_report['local_md5'],
                                               raw_report['local_sha256'],
                                               int(raw_report['local_size']))
        except Exception:
            external_checksums = None

        try:
            external_mpfs_checksums = FileChecksums(raw_report['remote_md5'],
                                                    raw_report['remote_sha256'],
                                                    int(raw_report['remote_size']))
        except Exception:
            external_mpfs_checksums = None

        return cls(raw_report['report_name'], address, external_checksums,
                   resource_id=resource_id, external_mpfs_checksums=external_mpfs_checksums)
