# -*- coding: utf-8 -*-
import time

import mpfs.engine.process
from mpfs.common.static import codes
from mpfs.common.errors import KladunConflict, StorageError, KladunBadResponse
from mpfs.core.address import ResourceId
from mpfs.core.factory import get_resource_by_resource_id
from mpfs.core.operations.base import DiskUploadOperation
from mpfs.core.filesystem.hardlinks.common import FileChecksums
from mpfs.core.file_recovery.errors import FileNotOnRecoveryError
from mpfs.core.filesystem.hardlinks.controllers import HardlinksController
from mpfs.core.services.restore_db_service import restore_db_service


log = mpfs.engine.process.get_default_log()
error_log = mpfs.engine.process.get_error_log()


class RestoreFileOperation(DiskUploadOperation):
    type = 'store'
    subtype = 'restore_file'

    @classmethod
    def Create(classname, uid, odata, **kw):
        checksums = FileChecksums(odata['store_md5'], odata['store_sha256'], odata['store_size'])
        hid = checksums.hid

        if not restore_db_service.is_exists(uid, checksums.hid):
            raise FileNotOnRecoveryError()

        resource = HardlinksController.get_resource_by_uid_hid_for_user_data(uid, hid)
        kladun_data = {
            'service': 'disk',
            'uid': uid,
            'path': resource.visible_address.id,
            'oid': odata['id'],
            'file-id': resource.meta.get('file_id'),
        }
        odata['kladun_data'] = kladun_data.copy()
        odata['resource_id'] = resource.resource_id.serialize()
        odata['file_id'] = resource.resource_id.serialize()
        odata['path'] = resource.visible_address.id
        odata['put_to_hardlink_updater'] = False

        ext_upload_url, int_status_url = classname.post_request_to_kladun(kladun_data, **kw)
        odata['upload_url'] = ext_upload_url
        odata['status_url'] = int_status_url
        odata['state'] = codes.EXECUTING
        return super(RestoreFileOperation, classname).Create(uid, odata, **kw)

    def handle_callback_1(self):
        kladun_checksums = self._get_kladun_checksums()
        if kladun_checksums.size == 0:
            log.info('Attemp to restore 0 size file. %r', kladun_checksums)
            self.data['hardlinked'] = True
            self.set_completed()
            return
        if not restore_db_service.is_exists(self.uid, kladun_checksums.hid):
            log.info('File not found in restore db. %r', kladun_checksums)
            self.data['hardlinked'] = True
            self.set_completed()
            return
        store_checksums = FileChecksums(self.data['store_md5'], self.data['store_sha256'], self.data['store_size'])
        if store_checksums != kladun_checksums:
            log.info('Hash missmatch. Store: %r Kladun: %r', store_checksums, kladun_checksums)
            self.data['hardlinked'] = True
            self.set_completed()
            return

    def handle_callback_2_and_3(self):
        if not self.data['put_to_hardlink_updater']:
            kladun_checksums = self._get_kladun_checksums()
            if not restore_db_service.is_exists(self.uid, kladun_checksums.hid):
                log.info('File not found in restore db. %r', kladun_checksums)
                raise KladunConflict()
            store_checksums = FileChecksums(self.data['store_md5'], self.data['store_sha256'], self.data['store_size'])
            if store_checksums != kladun_checksums:
                log.info('Hash missmatch. Store: %r Kladun: %r', store_checksums, kladun_checksums)
                raise KladunConflict()

            self.data['put_to_hardlink_updater'] = True
            self.save()
            file_data = self.data['filedata']['meta']

            file_mid = file_data['file_mid']
            digest_mid = file_data['digest_mid']

            if not file_mid or not digest_mid:
                log.info('Invalid stids received from kladun (file_mid: %s, digest_mid: %s)' % (file_mid, digest_mid))
                raise KladunBadResponse()

            new_stids = {
                'hid': kladun_checksums.hid,
                'file_mid': file_mid,
                'digest_mid': digest_mid,
            }
            user_resource_updated = False
            try:
                resource = get_resource_by_resource_id(self.uid, ResourceId.parse(self.data['resource_id']))
                is_changed, _ = resource.update_mids(new_stids)
                if is_changed:
                    resource.save()
                    user_resource_updated = True
            except Exception:
                error_log.error("Failed to update users file", exc_info=True)

            log.info("File recovery ended: user_resource_updated: %s", user_resource_updated)

            restore_db_service.delete_public(kladun_checksums.hid)
            restore_db_service.delete_personal(self.uid, kladun_checksums.hid)

    def _get_kladun_checksums(self):
        md5, sha256, size = self._get_sha_size_md5()
        return FileChecksums(md5, sha256, int(size))

    def certain_infoupdate(self):
        self.data['filedata'].update({
            'size': self.kladun_parse.content_length,
            'mimetype': self.kladun_parse.mimetype,
        })

        self.data['filedata']['meta'].update({
            'md5': self.kladun_parse.md5,
            'sha256': self.kladun_parse.sha256,
        })
