# -*- coding: utf-8 -*-
"""

MPFS
CORE

Файловая система

"""
import copy
import hashlib
import re
import traceback
import operator
import os
import sys
import time
import math
import datetime
import inspect as inspect_module
import uuid

from copy import deepcopy
from collections import defaultdict, OrderedDict
from itertools import ifilter, imap

from psycopg2 import ProgrammingError

import mpfs.engine.process
import mpfs.core.factory as factory
import mpfs.common.errors as errors
import mpfs.common.errors.share as share_errors
import mpfs.core.filesystem.resources.base as base_resources
import mpfs.core.filesystem.events as fs_events

from mpfs.common.forms.mixed import MixedForm
from mpfs.common.static.tags import FILE
from mpfs.common.static.tags.push import DIFF
from mpfs.common.util import convert_gps_deg_dec, to_json, logger, chunks, format_log_message, safe_to_int, \
    normalize_unicode, ctimestamp
from mpfs.common.util.experiments.logic import experiment_manager
from mpfs.common.util.filetypes import getGroupByName
from mpfs.common.util.mobile_client import MobileClientVersion
from mpfs.common.util.user_agent_parser import UserAgentParser
from mpfs.core.albums.static import GeneratedAlbumType
from mpfs.core.albums.errors import AlbumsWrongGeneratedType

from mpfs.core.filesystem import hardlinks
from mpfs.core.filesystem.constants import *
from mpfs.core.filesystem.dao.folder import FolderDAO
from mpfs.core.filesystem.dao.legacy import CollectionRoutedDatabase, is_new_fs_spec_required
from mpfs.core.filesystem.dao.trash_cleaner_queue import TrashCleanerQueueDAO
from mpfs.core.filesystem.hardlinks.common import construct_hid, FileChecksums
from mpfs.core.filesystem.live_photo import LivePhotoFilesManager
from mpfs.core.filesystem.resources.base import Resource
from mpfs.core.filesystem.resources.disk import MPFSFile
from mpfs.config import settings
from mpfs.core.filesystem.hardlinks.sharded import ShardedHardLink
from mpfs.core.filesystem.resources.narod import LegacyNarodFolder
from mpfs.core.filesystem.resources.photounlim import (
    is_converting_address_to_photounlim_for_uid_needed,
    is_photounlim_fraudulent_store, is_unlim_forbidden_due_to_experiment,
    is_converting_address_to_newunlim_for_uid_needed)
from mpfs.core.global_gallery.logic.controller import GlobalGalleryController
from mpfs.core.history import History
from mpfs.core.filesystem.quota import Quota
from mpfs.core.office.static import OfficeAccessStateConst
from mpfs.core.photoslice.albums.logic import resolve_photoslice_album_type
from mpfs.core.services.djfs_albums import djfs_albums_legacy
from mpfs.core.user.constants import (
    PHOTOUNLIM_AREA,
    PHOTOUNLIM_AREA_PATH,
    TRASH_AREA,
    ADDITIONAL_AREA_PATH,
    DISK_AREA,
    HIDDEN_AREA,
    PHOTOSTREAM_AREA,
    PHOTOSTREAM_AREA_PATH,
    LNAROD_AREA,
)
from mpfs.core.user.dao.user import UserDAO
from mpfs.core.user.utils import ignores_shared_folders_space, is_correct_space_checks_in_dry_mode_for
from mpfs.dao.base import DAOPath
from mpfs.dao.session import Session
from mpfs.metastorage.mongo.binary import Binary
from mpfs.core.filesystem.symlinks import Symlink
from mpfs.core.filesystem.dao.resource import AdditionalDataDAO
from mpfs.core.services.video_service import video
from mpfs.core.filesystem.dao.file import FileDAO
from mpfs.core.filesystem.indexer import DiskDataIndexer
from mpfs.core.filesystem.helpers.counter import Counter
from mpfs.core.filesystem.helpers.lock import LockHelper
from mpfs.core.filesystem.helpers.notify import XivaHelper
from mpfs.core.address import Address, SymlinkAddress, change_parent, ResourceId
from mpfs.core.metastorage.control import (
    hidden_data,
    support_blocked_hids,
    recount,
)
from mpfs.core.services import trash_service, kladun_service, mulca_service, geocoder_service, previewer_service
from mpfs.metastorage.mongo.collections.changelog import ChangelogCollection
from mpfs.metastorage.mongo.collections.filesystem import UserDataCollection, is_quick_move_enabled, \
    UserCollectionZipped
from mpfs.metastorage.mongo.util import id_for_key, parent_for_key, name_for_key, generate_version_number, compress_data, is_subpath
from mpfs.core.filesystem.cleaner.models import StorageCleanCheckStid, DeletedStid, DeletedStidSources
from mpfs.core.versioning.logic.version import Version
from mpfs.core.versioning.logic.version_manager import ResourceVersionManager, async_remove_versions_by_full_index
from mpfs.core.filesystem.resources.disk import append_meta_to_office_files, DiskFolder
from mpfs.core.queue import mpfs_queue
from mpfs.metastorage.postgres.exceptions import SetAutocommitError
from mpfs.metastorage.postgres.queries import SQL_SET_FILE_ID_TO_NULL_BY_PATH, SQL_SET_FILE_ID_FOR_FILE_BY_PATH
from mpfs.metastorage.postgres.query_executer import PGQueryExecuter

STORAGE_CLEANER_WORKER_ENABLE_STID_ACCESS_CHECK = settings.storage_cleaner['worker']['enable_stid_access_check']
STORAGE_CLEANER_WORKER_ENABLE_INCREMENT_USING_UPDATE = \
    settings.storage_cleaner['worker']['enable_increment_using_update']
FEATURE_TOGGLES_DISABLE_OLD_PREVIEWS = settings.feature_toggles['disable_old_previews']
SYSTEM_SYSTEM_FILESYSTEM_LOCK_AUTOUPDATE_PERIOD = settings.system['system']['filesystem_lock_autoupdate_period']
FEATURE_TOGGLES_USE_FILESYSTEM_LOCK_AUTOUPDATE_PERIOD = \
    settings.feature_toggles['use_filesystem_lock_autoupdate_period']
FEATURE_TOGGLES_TRASH_DROP_ALL_LOCK_CHECK = settings.feature_toggles['trash_drop_all_lock_check']
SYSTEM_ATTACH_SYS_FOLDERS = settings.system['attach_sys_folders']
FEATURE_TOGGLES_REMOVE_FOLDER_FIRST = settings.feature_toggles['remove_folder_first']
FEATURE_TOGGLES_CORRECT_SPACE_CHECKS_FOR_SHARED_FOLDERS = \
    settings.feature_toggles['correct_space_checks_for_shared_folders']
FEATURE_TOGGLES_IGNORE_SHARED_FOLDER_IN_QUOTA = settings.feature_toggles['ignore_shared_folder_in_quota']
FEATURE_TOGGLES_SETTING_YAROVAYA_MARK_ENABLED = settings.feature_toggles['setting_yarovaya_mark_enabled']
POSTGRES_QUICK_MOVE_FOLDER_RESOURCES_LIMIT = settings.postgres['quick_move']['folder_resources_limit']
LIVE_PHOTO_ROBUST_STORE_IOS_VERSION = settings.live_photo['robust_store_ios_version']
MAX_FILE_SIZE = settings.services['disk']['filesize_limit']
ALBUMS_DJFS_ALLOWED_LIST = settings.album['allowed_methods_list']
DOCS_REMOVE_ALLOWED_METHODS_LIST = settings.docs['allowed_methods_list']


log = mpfs.engine.process.get_default_log()
error_log = mpfs.engine.process.get_error_log()


def check_path(address, **kw):
    """
    Проверка корректности пути
    """
    Address(address, **kw)


def check_destination(address, **kw):
    if Address(address, **kw).is_storage:
        raise errors.StorageAddressError()


def notify_storage_clean_check_stids(stids):
    if not STORAGE_CLEANER_WORKER_ENABLE_STID_ACCESS_CHECK:
        return

    if not stids:
        return

    if not STORAGE_CLEANER_WORKER_ENABLE_INCREMENT_USING_UPDATE:
        filtered_stids = StorageCleanCheckStid.controller.filter(**{'_id': {'$in': stids}})
        stids = [filtered_stid.stid for filtered_stid in filtered_stids]
        if not stids:
            return

    StorageCleanCheckStid.controller.collection.update({'_id': {'$in': stids}},
                                                       {'$inc': {'counter': 1}},
                                                       multi=True)


def notify_storage_clean_check_resource(item):
    if not STORAGE_CLEANER_WORKER_ENABLE_STID_ACCESS_CHECK:
        return

    if isinstance(item, dict):
        meta = item
    elif isinstance(item, Resource):
        meta = item.meta
    else:
        raise TypeError()

    stid_types = ['file_mid', 'digest_mid', 'pmid']
    stids = [meta[stid_type] for stid_type in stid_types if meta.get(stid_type) is not None]

    if 'previews' in meta:
        previews = meta['previews']
        if isinstance(previews, dict):
            stids += previews.values()

    notify_storage_clean_check_stids(stids)


class Filesystem(object):
    def __init__(self, *args, **kwargs):
        self.request = None
        self.job_content = defaultdict(list)
        self.push_content = []
        self.email_content = defaultdict(list)
        self.xiva_helper = XivaHelper(self)
        self.lock_helper = LockHelper()
        self.counter_helper = Counter()
        self.async = False
        self.quota = Quota()
        self.disk_indexer = DiskDataIndexer()
        self.method = None
        self.connection_id = ''
        for k, v in kwargs.iteritems():
            setattr(self, k, v)
        from mpfs.core.social.publicator import Publicator
        self.publicator = Publicator(request=self.request)
        self.fotki_proxy_urls = set()

    def check_rights(self, uid, rawaddress=None, resource=None):
        """
        Базовый метод проверки прав
        Пока запиливаем для каталога дистрибутивов
        """
        if not resource and rawaddress:
            address = Address(rawaddress)
            try:
                resource = factory.get_resource(address.uid, address.path)
            except errors.ResourceNotFound:
                try:
                    resource = factory.get_resource(address.uid, address.parent_path)
                except errors.ResourceNotFound:
                    return

        if str(resource.uid) != str(uid):
            raise errors.PermissionDenied()

    def check_address(self, rawaddress, **kw):
        if Address(rawaddress, **kw).is_storage:
            raise errors.StorageAddressError()

    @staticmethod
    def _get_blocked_hids(hids):
        # В монго тело запроса не может превышать 16МБ (16777216 байт)
        # Запрос с 375000 записей занимает 16763915 байт
        #
        # https://st.yandex-team.ru/CHEMODAN-34551
        CHUNK_SIZE = 375000
        for hid_chunk in chunks(hids, CHUNK_SIZE):
            for blocked_hid in support_blocked_hids.get(hid_chunk):
                yield blocked_hid

    def filter_by_blocked_hids(self, resources):
        """
        Отфильтровывает заблокированные по HID'у ресурсы.

        :param list resources: Список ресурсов, из которого следует отфильтровать заблокированные.

        :return: Список незаблокированных ресурсов.
        :rtype: list
        """
        hids = [r.hid for r in resources if r.type == 'file']
        blocked_hids = [x['hid'] for x in self._get_blocked_hids(hids)]
        return [r for r in resources if r.type == 'dir' or r.hid not in blocked_hids]

    def check_hids_blockings(self, hids, method=None):
        """Проверяет hid'ы на наличение в таблице заблокированных"""
        if method is None:
            method = self.method
        checking_methods = [
            'copy_resource',
            'move_resource',
            'hardlink_copy',
            'check_hids_blockings',
            'export_photos',
            'video_url',
        ]

        if method in checking_methods:
            hid = next(self._get_blocked_hids(hids), None)
            if hid:
                log.info("Operation %s with blocked hid %s found" % (method, hid))
                raise errors.HidBlocked

    @staticmethod
    def _check_full_file_upload_process_needed_by_hashes(uid, size, md5, sha256, is_live_photo):
        """
        Проверка необходимости полного процесса загрузки для нового файла по хешам

        Если файл существует по хэшам - грузить не надо
        Особенная логика для live photo. Если есть лайв фото файл с такими же хешами, а грузим обычную фотку - грузить
        можно.
        """
        try:
            hardlinked_resource = factory.get_resource_by_uid_and_hid(
                uid, FileChecksums(md5, sha256, size).hid,
                enable_collections=('user_data', 'photounlim_data')
            )
        except errors.ResourceNotFound:
            return

        if experiment_manager.is_feature_active('disable_live_photo'):
            # ОТРЫВАЕМ: Залитый файл это лайвфотка, а грузим не лайвфотку
            # (при отключенном на мобилах экспе, все фотки приходят будто не лайв)
            pass
        else:
            if hardlinked_resource.meta.get('is_live_photo', False) and not is_live_photo:
                return

        exception_info = {
            'path': hardlinked_resource.id,
            'resource_id': hardlinked_resource.resource_id.serialize(),
        }
        if not hardlinked_resource.meta.get('is_live_photo', False) and is_live_photo:
            raise errors.LivePhotoUploadAttemptRegularPhotoExists(headers=exception_info)
        raise errors.FileAlreadyExist(headers=exception_info)

    def check_photostream_file_existing(self, uid, rawaddress, size, md5=None, sha256=None,
                                        is_live_photo=False, replace_hid=None):
        """

        Логика обработки файлов, загружаемых через photostream

        1) Проверить хардлинк, если передали md5 и sha256
        Если хардлинк найден и uid совпадает - не разрешаем загрузку
        Если хардлинк найден и uid не совпадает - разрешаем загрузку
        Если хардлинк не найден - разрешаем загрузку

        2) Проверить имя.
        Взять имя и размер.
        Если имени не существует - разрешаем загрузку
        Если имя существует - разрешаем загрузку и меняем имя файла если не было передано значение replace_md5,
                              если было передано — то просто разрешаем

        """
        if size == 0:
            raise errors.PreconditionsFailed()

        if size and md5 and sha256:
            self._check_full_file_upload_process_needed_by_hashes(uid, size, md5, sha256, is_live_photo)

        path = Address(rawaddress)

        # резрешаем загрузку, если передан replace_hid безотносительно того, есть файл или нет
        # https://st.yandex-team.ru/CHEMODAN-77309
        if replace_hid:
            return path.id

        try:
            factory.get_resource(uid, path)
            path = path.add_suffix('_' + str(int(time.time())))
        except errors.ResourceNotFound:
            pass
        return path.id

    def check_photostream_live_photo_video_part(self, uid, size, md5, sha256):
        if size == 0:
            raise errors.PreconditionsFailed()

        if size and md5 and sha256:
            try:
                hardlinked_resource = factory.get_resource_by_uid_and_hid(
                    uid, FileChecksums(md5, sha256, size).hid,
                    enable_collections=('additional_data',)
                )
                raise errors.FileAlreadyExist(headers={
                        'path': hardlinked_resource.id,
                        'resource_id': hardlinked_resource.resource_id.serialize(),
                    })
            except errors.ResourceNotFound:
                pass

    def preprocess_path(self, uid, rawaddress):
        """
        Преобразование пути фотострима к нужному
        """
        path = Address(rawaddress)
        if path.storage_name == 'photostream':
            photostream_addr = self.get_photostream_address(uid)
            path.change_parent(photostream_addr)
            return path.id
        else:
            return rawaddress

    def check_moveability(self, source, target):
        src_address = Address(source)
        tgt_address = src_address.clone(target)

        if source == target or src_address.id == tgt_address.parent_id:
            raise errors.MoveSame(source, target)

        if tgt_address.is_subfolder(src_address):
            raise errors.MoveParentToChild(source, target)

    def check_lock(self, rawaddress, skip_self_lock=False, **kw):
        self.lock_helper.check(rawaddress, skip_self_lock)

    def set_lock(self, rawaddress, data=None, time_offset=0):
        """
        :return: `_id` документа из коллекции
        :rtype: str
        """
        return self.lock_helper.lock(rawaddress, data, time_offset)

    def get_lock(self, rawaddress):
        """
        :return: запись из коллекции локов
        :rtype: dict
        """
        return self.lock_helper.get_lock(rawaddress)

    def update_lock(self, rawaddress, data=None, time_offset=0):
        self.lock_helper.update(rawaddress, data, time_offset)

    def unset_lock(self, rawaddress):
        self.lock_helper.unlock(rawaddress)

    def set_trash_lock(self, uid, oid=None, lock_data=None):
        address = Address.Make(uid, '/trash')

        if (not FEATURE_TOGGLES_TRASH_DROP_ALL_LOCK_CHECK or
                oid is None or
                FEATURE_TOGGLES_USE_FILESYSTEM_LOCK_AUTOUPDATE_PERIOD is False):
            # если пришли не из операции или отключено использование счетчика, который обновляет локи, то
            # просто пытаемся поставить лок
            self.lock_helper.lock(address.id, lock_data, operation=self.method)
        else:
            lock = self.get_lock(address.id)

            if lock is None:
                # если лок еще не стоит, то добавляем к нему oid операции и лочим
                self.lock_helper.lock(address.id, data={'oid': oid}, operation=self.method)
                return

            lock_oid = lock.get('data', {}).get('oid')

            if lock_oid != oid:
                # проверяем, от текущей ли операции был поставлен лок, если нет, то фейлимся
                raise errors.ResourceLockFailed(uid, address.id)

            # проверяем, не выполняется ли эта операция в другом воркере
            lock_dtime = lock['dtime']
            now = datetime.datetime.utcnow()
            if now < lock_dtime + datetime.timedelta(seconds=SYSTEM_SYSTEM_FILESYSTEM_LOCK_AUTOUPDATE_PERIOD):
                # если время меньше времени апдейта таймера, обновляющего локи, то ждем:
                # не фейлим таску, просто ставим в конец очереди, для этого тут достаточно кинуть особенное исключение
                # и поймать его в TrashDrop, который, в свою очередь, сделает reenque для операции
                raise errors.ResourceCheckLockFailed()

            self.lock_helper.update(address.id, data={'oid': oid})

    def unset_trash_lock(self, uid):
        address = Address.Make(uid, '/trash')
        self.lock_helper.unlock(address.id)

    def check_trash_lock_for(self, rawaddress):
        """
        Проверит лок для адреса, который предполагают разместить в корзине

        :param rawaddress: сырой адрес вида UID:PATH
        """
        address = self.get_trash_address(rawaddress)
        self.lock_helper.check(address.id)

    def is_trash_append_process_running(self, uid):
        # для эффективности наличие бегущих трэш операций определяем по наличию локов
        return self.lock_helper.do_trash_append_locks_exist(uid)

    def check_preconditions(self, uid, rawaddress, **kw):
        if 'replace_md5' in kw and kw['replace_md5'] is not None:
            try:
                address = Address(rawaddress)
                resource = factory.get_resource(uid, address)
            except errors.ResourceNotFound:
                if kw['replace_md5'] != '':
                    raise errors.PreconditionsFailed()
            else:
                if resource.md5() != kw['replace_md5']:
                    raise errors.PreconditionsFailed()

    def check_existing_md5(self, uid, rawaddress, md5):
        if not md5:
            return
        address = Address(rawaddress)
        resource = factory.get_resource(uid, address)
        if not isinstance(resource, MPFSFile):
            raise errors.MD5CheckNotSupportedError()
        if resource.md5() != md5:
            raise errors.PreconditionsFailed()

    def check_file_hash(self, uid, rawaddress, md5, sha256, size):
        address = Address(rawaddress)
        resource = factory.get_resource(uid, address)
        if md5 != resource.md5() or sha256 != resource.sha256() or size != resource.size:
            raise errors.PreconditionsFailed()

    @classmethod
    def exists(cls, uid, rawaddress):
        try:
            cls.resource(uid, rawaddress, check_parent=False)
            return True
        except errors.ResourceNotFound:
            return False

    def info(self, uid, rawaddress, unzip_file_id=False, load_source_ids=False, **kw):
        """Получить метаинформацию о ресурсе

        :param bool unzip_file_id: Если ``True``, то file_id будет раззипован
                                   из meta в data. А ресурс пересохранен

        .. warning::
            Раззиповка работает только для файлов.
            Используется для альбомов и OfficeOnline.
        """

        try:
            try:
                address = Address(rawaddress)
            except errors.AddressError:
                address = Address.Make(uid, rawaddress)

            try:
                parent_address = address.get_parent()
                factory.get_resource(uid, parent_address)
            except errors.ResourceNotFound:
                raise errors.FolderNotFound(parent_address.id)

            resource = self.get_resource(uid, address, unzip_file_id)
            if load_source_ids:
                GlobalGalleryController.load_and_set_source_ids_for_resources([resource])
        except errors.AddressError:
            raise errors.InfoPathError(rawaddress)

        return resource.info()

    def get_resource(self, uid, address, unzip_file_id=False):
        """Получить ресурс оп uid и адресу, при необходимости раззиповать file_id.

        :type uid: str
        :type address: :class:`~Address`
        :type unzip_file_id: bool
        :param unzip_file_id: Если ``True``, то file_id будет раззипован
                              из meta в data. А ресурс пересохранен
        :param load_source_ids: Подкачать информацию о source_id для ресурса
        :rtype: :class:`~Resource`
        """

        resource = factory.get_resource(uid, address, request=self.request)

        if unzip_file_id and resource.is_file_id_zipped():
            resource.save()

        resource.set_request(self.request)
        resource.load_views_counter()
        return resource

    def resources_by_resource_ids(self, uid, resource_ids, enable_service_ids=('/disk',), enable_optimization=False,
                                  load_source_ids=False):
        """
        Получить пачку ресуров по ResourceIdes и представить информацию от лица uid-a

        :param uid: uid пользователя, от которого приходи запрос
        :param resource_ids: список объектов ResourceId
        :param enable_service_ids: где искать ресурс. Доступные варианты ищи в factory
        :param enable_optimization: отключить повторное хождение в базу за ресурсами.
        :param load_source_ids: подгрузить для всех файлов source_ids
        """
        resources = []
        for resource in factory.get_resources_by_resource_ids(
                uid, resource_ids, enable_service_ids=enable_service_ids, enable_optimization=enable_optimization):
            if resource is None:
                continue
            resource.set_request(self.request)
            resources.append(resource)
        filtered_resources = self.filter_by_blocked_hids(resources)
        if load_source_ids:
            GlobalGalleryController.load_and_set_source_ids_for_resources(filtered_resources)
        return filtered_resources

    def resources_by_resource_ids_filtered(self, uid, resource_ids, enable_service_ids=('/disk',), enable_optimization=False,
                                          filter_duplicates=True, filter_values=None, load_source_ids=False):
        """
        resources_by_resource_ids с фильтрацией по десериализованным ресурсам
        :param filter_values dict: словарь характеристик, по которым надо отфильтровать результат
        """
        all_resources = factory.find_all_resources_by_resource_ids(uid, resource_ids, enable_service_ids=enable_service_ids,
                                                                   enable_optimization=enable_optimization,
                                                                   filter_duplicates=filter_duplicates)
        if not filter_values:
            return all_resources
        filtered_resources = []
        for resource in all_resources:
            if filter_values.get('md5') and resource.meta['md5'] != filter_values['md5']:
                continue
            if filter_values.get('sha256') and resource.meta['sha256'] != filter_values['sha256']:
                continue
            if filter_values.get('size') and resource.size != filter_values['size']:
                continue
            filtered_resources.append(resource)
        if load_source_ids:
            GlobalGalleryController.load_and_set_source_ids_for_resources(filtered_resources)
        return filtered_resources

    def resource_by_file_id(self, uid, file_id, owner_uid=None, enable_service_ids=('/disk',), load_source_ids=False):
        """
        Получить ресурс по file_id
        Представить информацию от лица uid'а.

        :type uid: str
        :type file_id: str
        :type owner_uid: str
        """
        if not owner_uid:
            owner_uid = uid
        resource_id = ResourceId(owner_uid, file_id)
        resources = self.resources_by_resource_ids(uid, [resource_id], enable_service_ids=enable_service_ids)
        if load_source_ids:
            GlobalGalleryController.load_and_set_source_ids_for_resources(resources)
        if resources:
            return resources[0]
        else:
            raise errors.ResourceNotFound(resource_id)

    def info_by_file_id(self, uid, file_id, owner_uid=None, enable_service_ids=('/disk',)):
        """
        Получить информацию о ресурсе по owner_uid, file_id.
        Представить информацию от лица uid'а.

        :type uid: str
        :type file_id: str
        :type owner_uid: str
        """
        return self.resource_by_file_id(uid, file_id, owner_uid=owner_uid, enable_service_ids=enable_service_ids).info()

    def bulk_info(self, uid, raw_addresses, load_source_ids=False, **kw):
        """
        Получение метаинформации о пачке ресурсов
        """
        addresses = []
        for i, raw_address in enumerate(raw_addresses):
            try:
                address = Address(raw_address)
            except errors.AddressError:
                try:
                    address = Address.Make(uid, raw_address)
                except errors.AddressError as e:
                    address = None
            if address is not None:
                addresses.append(address)
        resources = factory.get_resources(uid, addresses)
        if load_source_ids:
            GlobalGalleryController.load_and_set_source_ids_for_resources(resources)

        result = []
        for resource in resources:
            resource.set_request(self.request)
            resource.load_views_counter()
            result.append(resource)
        return result

    def content(self, uid, rawaddress, params=None, load_source_ids=False, **kw):
        """
        Получение листинга ресурса
        """
        try:
            address = Address(rawaddress)
            resource = factory.get_resource(uid, address)
            resource.set_request(self.request)
            if self.request and self.request.meta and 'views_counter' in self.request.meta:
                resource.load_views_counter()
            result = resource.list()
            if load_source_ids:
                children_files = [] if resource.type == FILE else resource.children_items['files']
                GlobalGalleryController.load_and_set_source_ids_for_resources([resource] + children_files)
        except errors.AddressError:
            raise errors.ListPathError(rawaddress)
        except errors.ResourceNotFound:
            raise errors.ListNotFound(rawaddress)
        except errors.NotFolder:
            raise errors.ListNotFolder(rawaddress)
        else:
            return result

    def tree(self, uid, rawaddress, deep_level, sort, order, parents=False):
        """
        Получение дерева каталога
        """
        try:
            address = Address(rawaddress, is_folder=True)
            folder = factory.get_resource(uid, address)
            folder.set_request(self.request)
            if parents:
                result = folder.parents(level=deep_level, sort=sort, order=order)
            else:
                result = folder.tree(level=deep_level, sort=sort, order=order)
        except errors.AddressError:
            raise errors.TreePathError(rawaddress)
        except errors.ResourceNotFound:
            raise errors.TreeNotFound(rawaddress)
        except errors.NotFolder:
            raise errors.TreeNotFolder(rawaddress)
        else:
            return result

    def fulltree(self, uid, rawaddress, **kw):
        """
        Получение полного дерева ресурсов
        """
        address = Address(rawaddress, is_folder=True)
        folder = factory.get_resource(uid, address)
        prefix = kw.pop('prefix', uid)

        return self._recurse_tree(folder, root=folder.visible_address.path, prefix=prefix, **kw)

    def _recurse_tree(self, resource, **kw):
        deep_level = int(kw.get('deep_level') or PUBLIC_TREE_MAX_LEVEL)
        clean = kw.get('clean', False)
        rfunc = kw.get('rfunc', Address.MakeRelative)
        relative = kw.get('relative', False)
        prefix = kw.get('prefix', '')
        root = kw.get('root', resource.visible_address.path)
        symlink_read_only = kw.get('symlink_read_only', False)

        def fall_next_level(resource, root, prefix, level=None):
            level = 1 if level is None else level + 1

            _this = resource
            if relative:
                relative_addr = rfunc(prefix, root, _this.visible_address.path)
                _this.path = relative_addr.path
                _this.id = relative_addr.id

            _list = []
            if (not deep_level) or (deep_level and level <= deep_level):
                resource.load()
                if symlink_read_only:
                    resource.clean_urls()

                for child in resource.child_files:
                    child_file = resource.subfile(child)

                    if clean:
                        if child_file.is_blocked() or child_file.is_infected():
                            continue

                    if relative:
                        relative_addr = rfunc(prefix, root, child_file.visible_address.path)
                        child_file.path = relative_addr.path
                        child_file.id = relative_addr.id
                    if symlink_read_only:
                        child_file.clean_urls()

                    _list.append({'this': child_file, 'list': []})

                for child in resource.child_folders:
                    child_folder = resource.subfolder(child)
                    _list.append(fall_next_level(child_folder, root, prefix, level))
            return {'this': _this, 'list': _list}

        return fall_next_level(resource, root, prefix)

    def services(self, uid, rawaddress, deep_level, sort, order):
        """
        Получение дерева каталога
        """
        try:
            address = Address(rawaddress, is_folder=True)
            folder = factory.get_resource(uid, address)
            folder.set_request(self.request)
            return folder.services()
        except errors.AddressError:
            raise errors.ServicesPathError(rawaddress)
        except errors.ResourceNotFound:
            raise errors.ServicesNotFound(rawaddress)
        except errors.NotFolder:
            raise errors.ServicesNotFolder(rawaddress)

    def timeline(self, uid, rawaddress):
        """
        Получение timeline
        """
        try:
            address = Address(rawaddress, is_folder=True)
        except errors.AddressError:
            raise errors.ListPathError( rawaddress)
        else:
            try:
                folder = factory.get_resource(uid, address)
                folder.set_request(self.request)
                try:
                    folders = bool(int(self.request.args['filter']['public']))
                except (AttributeError, KeyError):
                    folders = False
                return folder.timeline(folders=folders)
            except errors.ResourceNotFound:
                raise errors.ListNotFound(rawaddress)
            except errors.NotFolder:
                raise errors.ListNotFolder(rawaddress)

    def set_last_files(self, uid, rawaddress):
        """Установить последние файлы для папки ``rawaddress``.

        Форматтер сериализует установленные ресурсы в виде ответа.
        """
        try:
            address = Address(rawaddress, is_folder=True)
        except errors.AddressError:
            raise errors.ListPathError(rawaddress)
        try:
            folder = factory.get_resource(uid, address)
        except errors.ResourceNotFound:
            raise errors.ListNotFound(rawaddress)
        except errors.NotFolder:
            raise errors.ListNotFolder(rawaddress)
        folder.set_request(self.request)
        folder.set_last_files()

    def url(self, uid, rawaddress, params):
        """
        Получение ссылки на скачивание файла.
        """
        try:
            address = Address(rawaddress, is_file=True)
            resource = factory.get_resource(uid, address)
            if isinstance(resource, LegacyNarodFolder):
                raise errors.UrlPathError(rawaddress)
            resource.set_request(self.request)
            return resource.get_url(**params)
        except errors.UrlPathError:
            raise
        except errors.AddressError:
            raise errors.InfoPathError(rawaddress)
        except errors.NotFile:
            raise errors.UrlNotFile(rawaddress)
        except errors.ResourceNotFound:
            raise errors.UrlNotFound(rawaddress)

    def public_download_url(self, uid, rawaddress, params):
        """
        Получение незалогиновой ссылки на скачивание файла.
        """
        try:
            address = Address(rawaddress, is_file=True)
            resource = factory.get_resource(uid, address)
            resource.set_request(self.request)
            params['owner_uid'] = resource.owner_uid
            return resource.get_public_download_url(**params)
        except errors.AddressError:
            raise errors.InfoPathError(rawaddress)
        except errors.NotFile:
            raise errors.UrlNotFile(rawaddress)
        except errors.ResourceNotFound:
            raise errors.UrlNotFound(rawaddress)

    def video_url(self, uid, rawaddress):
        """
        Получение ссылки на стриминг видео
        """
        address = Address(rawaddress)
        resource = factory.get_resource(uid, address)
        self.check_hids_blockings([resource.hid, ])
        return video.generate_url(uid, address.uid, resource.file_mid(), resource.hid,
                                  video_info=resource.get_video_info())

    def direct_url(self, uid, rawaddress, modified):
        """
        Получение редиректа на прямую ссылку с содержимым файла
        Если modified
        """
        try:
            address = Address(rawaddress, is_file=True)
            resource = factory.get_resource(uid, address)
            return resource.get_direct_url(modified)
        except errors.AddressError:
            raise errors.InfoPathError(rawaddress)
        except errors.NotFile:
            raise errors.UrlNotFile(rawaddress)
        except errors.ResourceNotFound:
            raise errors.UrlNotFound(rawaddress)

    def mkdir(self, uid, rawaddress, notify=False, notify_search=True,
              copy_symlinks=True, rm_symlinks=False, sysdir=False, skip_check_rights=False, keep_lock=False,
              **folderdata):
        """Создание каталога"""
        try:
            address = Address(rawaddress, is_folder=True)

            parent_addr = address.get_parent()
            parent_folder = factory.get_resource(uid, parent_addr)

            # Если родительский каталог -- не каталог вовсе, а файл, то сразу швыряемся исключением.
            if parent_folder.type == 'file':
                raise errors.MkdirNotFound()

            if not skip_check_rights:
                self.check_rights(uid, resource=parent_folder)
            parent_folder.check_rw()
            try:
                existing_resource = factory.get_resource(uid, address)
            except Exception:
                existing_resource = None
            else:
                """
                следующий if нужен для сохранения логики работы, как в файловой системе,
                где при перемещении destination директория наполняется содержимым
                source, вместо удаления и замещения, как описано в RFC
                """
                if isinstance(existing_resource, base_resources.File):
                    if self.method != 'mkdir':
                        self.rm(uid, rawaddress, notify=False, drop=False, keep_lock=keep_lock)
                    else:
                        raise errors.ResourceExist()
                else:
                    self.mkdir_existing_element = existing_resource
                    raise errors.ResourceExist()

            if parent_folder.is_shared or parent_folder.is_group:
                uids = parent_folder.group.all_uids()
                old_version = parent_folder._service.get_version(uids)
            else:
                old_version = parent_folder._service.get_version(parent_addr.uid)

            if not parent_folder.folders_allowed(method=self.method) and not sysdir:
                raise errors.MkdirNotPermitted()

            if self.method == 'mkdir':
                folder = parent_folder.create_new_child_folder(uid, address, **folderdata)
            else:
                folder = parent_folder.create_child_folder(uid, address, **folderdata)

            # Обработка симлинков
            folder.treat_symlink(copy_symlinks, rm_symlinks)

            folder.set_request(self.request)

            # Пушим данные в xiva
            if (inspect_module.getmodule(sys._getframe(1)).__name__ != __name__ or notify) \
                    and address.storage_name == 'disk':
                xiva_data = {
                    'op': 'new',
                    'fid': folder.meta['file_id'],
                    'resource_type': 'dir',
                }
                if parent_folder.is_shared or parent_folder.is_group:
                    xiva_data['key'] = folder.address.path
                    self.xiva_helper.add_to_xiva_group(
                        uid,
                        parent_folder.group.gid,
                        xiva_data,
                        old_version,
                        folder.version
                    )
                else:
                    xiva_data['key'] = address.path
                    self.xiva_helper.add_to_xiva(
                        address.uid,
                        xiva_data,
                        old_version,
                        folder.version
                    )

            if notify_search:
                self.disk_indexer.push(folder, 'modify', operation=self.method)

            self.mkdir_new_element = folder

            if self.method == 'mkdir':
                event = fs_events.FilesystemCreateDirectoryEvent(uid=str(uid), tgt_resource=folder, tgt_address=address)
                event.send_self_or_group(target_parent=parent_folder)

            return folder.dict()
        except errors.ResourceNotFound:
            raise errors.MkdirNotFound(rawaddress)
        except errors.ResourceExist:
            try:
                autosuffix_path = self.autosuffix_address(Address(rawaddress, is_folder=True)).path
            except Exception:
                error_log.exception('mkdir autosuffix_address failed')
                raise errors.MkdirFolderAlreadyExist(rawaddress)
            raise errors.MkdirFolderAlreadyExist(rawaddress, data=dict(autosuffix_path=autosuffix_path))
        except errors.NotPermitted:
            raise errors.MkdirNotPermitted(rawaddress)
        except errors.AddressError:
            raise errors.MkdirPathError(rawaddress)
        except errors.NotFolder:
            raise errors.MkdirNotFolder(rawaddress)

    def check_available_space(self, uid=None, address=None, required_space=None, group=None,
                              operation_type=None, operation_user_uid=None):
        """Проверяет свободное место указанному Пользователю.

        :param uid: Пользователь, для которого проверяем место
        :param address: Адрес, по которому проверяем место (приоритетен перед uid, т.к. несет в себе и uid и area)
        :param required_space: сколько место необходимо
        :param group: Шаренная группа. Передается если проверка нужна Владельца папки,
                      в которой Приглашенный делает операцию.
        :param operation_type: Тип операции. Для выдачи в ошибке. Используется только для ошибки при проверке
                               свободного места Владельца ОП.
        :param operation_user_uid: uid совершающего операцию. Нужен при проверке места Владельца, для опредления
                                   корневой папки.

        :raises :class:`mpfs.common.errors.NoFreeSpace`: если недостаточно места

        Нужно передать или адрес, или uid. is_group_owner_check передается, если проверяется место Владельца ОП,
        при операции Приглашенным.
        """
        if address is None:
            if uid is None:
                raise TypeError('Address or uid must be specified')
            # Если не указан адрес, то проверяем место пользователя в основном разделе /disk
            free_space = self.quota.free(uid=uid)
        else:
            if isinstance(address, (unicode, str)):
                address = Address(address)
            uid = address.uid
            free_space = self.quota.free(address=address)

        # Проверяем нехватку места
        if (free_space <= 0 or
                required_space is not None and int(required_space) > free_space):
            if group is None:
                raise errors.NoFreeSpace()
            # ALERT: Передаем json-строку в title ошибки для:
            #   * совместимости с передачей ошибок через WebDAV для клиентов в title при 4xx ошибках
            #   * возможности передать контекст ошибки для клиента, чтобы показать пользователю понятную ошибку
            details = {'owner': {'uid': uid,
                                 'display_name': group.owner_user.get_user_info().get('display_name', '')},
                       'user_path': group.get_link_by_uid(operation_user_uid).path,
                       'operation': operation_type,
                       'code': errors.OwnerHasNoFreeSpace.code,
                       'message': errors.OwnerHasNoFreeSpace.__name__}
            raise errors.OwnerHasNoFreeSpace(title=to_json(details))

        return free_space

    def check_user_space_on_move(self, uid, src_resource):
        if ((src_resource.is_group or src_resource.is_shared) and self.method != 'trash_append' and
                    Quota().free(uid=uid) <= 0):
            raise errors.NoFreeSpaceMoveFromShared('Tried to move from shared folder but free space quota exceeded')

    def log_failed_space_checks(self, uid, type, target_parent_resource=None):
        log_data = {'uid': uid,
                    'free': self.quota.free(uid=uid),
                    'limit': self.quota.limit(uid=uid),
                    'type': type}
        if target_parent_resource is not None and (target_parent_resource.is_shared or target_parent_resource.is_group):
            log_data.update({'owner_uid': target_parent_resource.owner_uid,
                             'owner_limit': self.quota.limit(uid=target_parent_resource.owner_uid),
                             'owner_free': self.quota.free(uid=target_parent_resource.owner_uid),
                             'group_id': target_parent_resource.group.gid,
                             'group_size': target_parent_resource.group.size,
                             'group_path': target_parent_resource.group.path})

        log.info(format_log_message(msg='Owner space checks failed', **log_data))

    def check_available_space_on_move_by_resource(self, uid, src_resource, dst_parent_resource):
        if not FEATURE_TOGGLES_CORRECT_SPACE_CHECKS_FOR_SHARED_FOLDERS:
            self.check_user_space_on_move(uid, src_resource)
            return

        try:
            # Проверяем есть ли свободное место у пользователей, у которых потребится место, а именно:
            if self.method == 'trash_append':
                # 0. Всегда разрешаем удалять файл
                return
            elif dst_parent_resource.is_shared:
                should_ignore_shared = ignores_shared_folders_space(uid=uid)
                # 1. Перемещаем в ОП Приглашенным
                # a) проверяем Владельца всегда
                self.check_available_space(uid=dst_parent_resource.owner_uid,
                                           group=dst_parent_resource.group,
                                           operation_type='move',
                                           operation_user_uid=uid)
                if not should_ignore_shared:
                    # b) если Приглашенный без Честного Шаринга, то проверяем и его
                    self.check_available_space(uid=uid)
            elif (src_resource.is_shared or
                      src_resource.is_group or dst_parent_resource.is_group):
                # 2.a Перемещаем из ОП Приглашенным
                # Даже если есть Честный Шаринг, т.к. перемещаем не в ОП, где пользователь Приглашенный
                # 2.b Перемещаем из/в ОП Владельцем
                self.check_available_space(uid=uid)
        except errors.NoFreeSpace:
            if not is_correct_space_checks_in_dry_mode_for(uid):
                raise
            self.check_user_space_on_move(uid, src_resource)
            self.log_failed_space_checks(uid, 'move', dst_parent_resource)

    def check_available_space_on_copy_by_raw_address(self, uid, dst_raw_address, required_space=None):
        if not FEATURE_TOGGLES_CORRECT_SPACE_CHECKS_FOR_SHARED_FOLDERS:
            self.check_available_space(address=dst_raw_address, required_space=required_space)
            return

        parent_resource = None
        try:
            address = Address(dst_raw_address)
            if address.storage_name == 'disk':
                try:
                    parent_resource = factory.get_resource(uid, address.get_parent())
                except errors.ResourceNotFound:
                    raise errors.CopyParentNotFound()

                if parent_resource.is_shared:
                    self.check_available_space(uid=parent_resource.owner_uid,
                                               group=parent_resource.group,
                                               operation_type='copy',
                                               operation_user_uid=uid)

                    if ignores_shared_folders_space(uid=uid):
                        required_space = 0

            self.check_available_space(address=address, required_space=required_space)
        except errors.NoFreeSpace:
            if not is_correct_space_checks_in_dry_mode_for(uid):
                raise
            # Выполняем старую проверку
            self.check_available_space(address=dst_raw_address, required_space=required_space)
            self.log_failed_space_checks(uid, 'copy', parent_resource)

    def check_available_space_on_store_by_raw_address(self, uid, raw_address, required_space=None):
        """
        Проверка наличия свободного места
        """
        address = Address(raw_address)

        if not FEATURE_TOGGLES_CORRECT_SPACE_CHECKS_FOR_SHARED_FOLDERS:
            return self.check_available_space(address=address, required_space=required_space)

        parent_resource = None
        try:
            # Если ресурс находится в расшаренной папке, то у Владельца должно быть достаточно места
            dir_is_shared = False
            new_required_space = required_space
            if address.storage_name == 'disk':
                parent_resource = factory.get_resource(uid, address.get_parent())
                dir_is_shared = parent_resource.is_shared
                if dir_is_shared:
                    owner_address = parent_resource.group.get_folder()
                    self.check_available_space(address=owner_address.address,
                                               required_space=required_space,
                                               group=parent_resource.group,
                                               operation_type='store',
                                               operation_user_uid=uid)

            if dir_is_shared and ignores_shared_folders_space(uid):
                new_required_space = 0

            available_space = self.check_available_space(address=address, required_space=new_required_space)
        except errors.NoFreeSpace:
            if not is_correct_space_checks_in_dry_mode_for(uid):
                raise
            # Выполняем старую проверку
            available_space = self.check_available_space(address=address, required_space=required_space)
            self.log_failed_space_checks(uid, 'store', parent_resource)

        return available_space

    def mkfile(self, uid, rawaddress, copy_symlinks=True, rm_symlinks=False,
               keep_symlinks=False, notify=True, replace_md5=None, notify_search=True,
               notify_search_type='modify', keep_lock=False, is_store=False, **filedata):
        """
        Создание файла
        """
        # TODO: copy_symlinks и keep_symlinks одинаковые параметры, нужно оставить только один
        if not isinstance(rawaddress, Address):
            address = Address(rawaddress, is_file=True)
        else:
            address = rawaddress
            address.is_file = True

        try:
            parent_folder = factory.get_resource(uid, address.get_parent())
        except errors.ResourceNotFound:
            raise errors.ParentNotFound(rawaddress)
        else:
            parent_folder.check_rw()
            self.check_preconditions(uid, rawaddress, replace_md5=replace_md5)

            dstore_update = False
            try:
                existing_resource = factory.get_resource(uid, address)
            except Exception:
                existing_resource = None
            else:
                dstore_update = existing_resource and existing_resource.meta['file_id'] == filedata.get('data', {}).get(
                    'meta', {}).get('file_id')
                if existing_resource.is_shared_root or existing_resource.is_group_root or existing_resource.with_shared:
                    raise share_errors.GroupNoPermit()

                if isinstance(existing_resource, MPFSFile):
                    ResourceVersionManager.add_binary_version(existing_resource)
                self.rm(
                    existing_resource.address.uid,
                    existing_resource.address.id,
                    notify=False,
                    keep_symlinks=keep_symlinks,
                    changelog=False,
                    lock=False,
                    keep_lock=keep_lock,
                )

            if parent_folder.is_shared or parent_folder.is_group:
                uids = parent_folder.group.all_uids()
                old_version = parent_folder._service.get_version(uids)
            else:
                old_version = parent_folder._service.get_version(uid)

            if self.method in ('mkfile', 'hardlink_copy'):
                _file = parent_folder.create_new_child_file(uid, address, **filedata)
            else:
                _file = parent_folder.create_child_file(uid, address, **filedata)

            _file.update_used()

            # версионирование
            if (self.method in ('mkfile', 'hardlink_copy', 'store') and
                    _file.address.storage_name == 'disk' and
                    existing_resource is None):
                ResourceVersionManager.bind_versions_from_trash(_file)

            # Ставим таск на обновление кеша последних файлов
            if parent_folder.is_shared or parent_folder.is_group:
                from mpfs.core.last_files.logic import SharedLastFilesProcessor
                # Будет игнорировать ошибки постановки таска на обновление кэша,
                # чтобы выполнились последующие callback'и
                SharedLastFilesProcessor().update_for_group_async(parent_folder.group.gid)

            # Обработка симлинков
            _file.treat_symlink(copy_symlinks, rm_symlinks)

            # Обрабатываем оверрайд ресурса
            if existing_resource and existing_resource.is_file and existing_resource.is_smth_public() and keep_symlinks:
                _file.copy_public_info(existing_resource)

            # Пушим данные в xiva
            if notify:
                if existing_resource:
                    op = 'changed'
                else:
                    op = 'new'
                data = {
                    'op': op,
                    'key': _file.address.path,
                    'md5': _file.meta['md5'],
                    'sha256': _file.meta['sha256'],
                    'size': _file.size,
                    'fid': _file.meta['file_id'],
                    'resource_type': 'file',
                }
                if parent_folder.is_shared or parent_folder.is_group:
                    self.xiva_helper.add_to_xiva_group(
                        uid,
                        parent_folder.group.gid,
                        data,
                        old_version,
                        _file.version
                    )
                    self.xiva_helper.add_to_xiva_space_group(uid, parent_folder.group.gid)
                elif address.storage_name == 'disk':
                    self.xiva_helper.add_to_xiva(
                        address.uid,
                        data,
                        old_version,
                        _file.version
                    )
                    self.xiva_helper.add_to_xiva_space(uid)
                elif address.storage_name == PHOTOUNLIM_AREA:
                    self.xiva_helper.add_to_xiva(
                        address.uid,
                        data,
                        old_version,
                        _file.version
                    )

            # Пушим в индексаторы
            if notify_search:
                if dstore_update:
                    self.disk_indexer.push(
                        _file,
                        notify_search_type,
                        search_data_modify=True,
                        operation=self.method,
                        append_djfs_callbacks=is_store,
                    )
                else:
                    source_item = None
                    if 'original_resource' in filedata:
                        source_item = filedata['data']
                    self.disk_indexer.push(
                        _file,
                        notify_search_type,
                        operation=self.method,
                        source_item=source_item,
                        append_djfs_callbacks=is_store,
                    )

            self.mkfile_new_element = _file
            _file.set_parent(parent_folder)
            return _file

    def rm(self, uid, rawaddress, drop=False, notify=True, keep_symlinks=False,
           keep_group=False, changelog=True, notify_search=True, mv=False, lock=True, keep_lock=False, lock_data=None):
        """
        Удаление ресурса
        Если drop != True, то переносит данные удаляемого ресурса в hidden
        Если drop == True, то просто затрет данные
        """
        try:
            address = Address(rawaddress)
            service = factory.get_service(address)
            try:
                resource = factory.get_resource(uid, address)
            except errors.ResourceNotFound:
                raise
            else:
                try:
                    resource.set_remove_date()
                    if lock and not keep_lock:
                        self.lock_helper.lock(resource, lock_data, operation=self.method)
                    self.check_rights(uid, resource=resource)
                    resource.check_rw(operation='rm')

                    if resource.is_group or resource.is_shared:
                        # нам обязательно нужно загнать данные в сторадж, тк нам нужна эта информация
                        # когда из базы удалятся все объекты
                        resource.group.get_all_root_folders(cached=False)

                    resource.set_request(self.request)
                    if notify:
                        if resource.is_shared or resource.is_group and not (resource.is_group_root or resource.is_shared_root):
                            uids = resource.group.all_uids()
                            old_version = resource._service.get_version(uids)
                        else:
                            old_version = resource.version

                    if resource.type == 'file':
                        notify_storage_clean_check_resource(resource)

                    full_index = None
                    if notify_search or notify and address.storage_name == 'disk' and self.method == 'rm':
                        # Используется и для логгирования событий о удалении каждой подпапки
                        # в event_history/event_subscriptions.py#_log_fs_delete_subdirs_if_needed
                        full_index = resource.get_full_index(safe=False)

                    if FEATURE_TOGGLES_REMOVE_FOLDER_FIRST:
                        size = resource.rm(drop=drop, remove_links=True, keep_group=keep_group, changelog=changelog)
                        service.commit(address.uid)
                        if drop:
                            resource.update_used(-1 * int(size))
                    else:
                        if keep_group is False and isinstance(resource, DiskFolder):
                            resource.remove_internal_shared_groups_and_links()

                    if address.storage_name in ('disk', 'attach', 'trash', NOTES_STORAGE_AREA, PHOTOUNLIM_AREA):
                        # Ставим команду создания записи файла в hidden
                        if not drop:
                            if resource.type == 'dir':
                                _listing = resource.tree_index()
                            else:
                                _listing = []

                            remove_data = {
                                'this': {
                                    'key': resource.id,
                                    'version': resource.version,
                                    'data': resource.dict(safe=True),
                                    'type': resource.type,
                                    'uid': resource.uid,
                                },
                                'list': _listing,
                            }

                            if resource.is_shared or resource.is_group:
                                remove_data['this']['data']['gid'] = resource.group.gid
                                remove_data['this']['data']['owner_uid'] = resource.owner_uid

                            force_having_yarovaya_mark = (resource.should_yarovaya_mark_be_set()
                                                          if FEATURE_TOGGLES_SETTING_YAROVAYA_MARK_ENABLED
                                                          else False)

                            self.tree_to_post_remove(
                                remove_data,
                                resource._service,
                                symlinks=not keep_symlinks,
                                remove_top_level_folder=FEATURE_TOGGLES_REMOVE_FOLDER_FIRST,
                                set_yarovaya_mark_for_subresources=force_having_yarovaya_mark,
                            )

                    if not FEATURE_TOGGLES_REMOVE_FOLDER_FIRST:
                        size = resource.rm(drop=drop, remove_links=True, keep_group=keep_group, changelog=changelog)
                        service.commit(address.uid)
                        if drop:
                            resource.update_used(-1 * int(size))

                    if (self.method in ('rm', 'trash_drop_element', 'trash_drop_all') and
                            address.storage_name in ('disk', 'trash') and
                            full_index):
                        async_remove_versions_by_full_index(full_index, owner_uid=resource.owner_uid)

                    if address.storage_name in ('disk', 'attach'):
                        # Пушим данные по ксиве, если пользователь подписан
                        if notify:
                            parent_folder = factory.get_resource(uid, address.get_parent())
                            resource.set_parent(parent_folder)
                            data = {
                                'op': 'deleted',
                                'key': resource.address.path,
                                'resource_type': resource.type,
                            }
                            try:
                                data['fid'] = resource.meta['file_id']
                            except KeyError:
                                if resource.address.is_storage:
                                    pass
                                else:
                                    raise
                            if resource.type == 'file':
                                data['md5'] = resource.meta['md5']
                                data['sha256'] = resource.meta['sha256']
                                data['size'] = resource.size
                            elif resource.is_group_root:
                                data['shared_rights'] = 'owner'
                                data['is_group_root'] = 1
                            elif resource.is_shared_root:
                                data['shared_rights'] = str(resource.link.rights)
                                data['is_group_root'] = 1
                            elif resource.with_shared:
                                data['with_shared'] = 1

                            try:
                                new_version = resource.deleted_version
                            except AttributeError:
                                new_version = parent_folder._service.get_version(uid)

                            if parent_folder.is_shared or parent_folder.is_group:
                                self.xiva_helper.add_to_xiva_group(
                                    uid,
                                    parent_folder.group.gid,
                                    data,
                                    old_version,
                                    new_version
                                )
                            else:
                                self.xiva_helper.add_to_xiva(
                                    address.uid,
                                    data,
                                    old_version,
                                    new_version
                                )

                            if self.method == 'rm':
                                event = fs_events.FilesystemRemoveEvent(uid=str(uid), tgt_resource=resource,
                                                                        tgt_address=address)
                                event.send_self_or_group(target_parent=parent_folder)

                        # Пушим в индексаторы
                        if notify_search:
                            self.disk_indexer.push_tree(
                                uid,
                                full_index.values(),
                                'delete',
                                version=resource._service.get_version(uid),
                                operation=self.method
                            )

                        # Удаляем фотки в проксе
                        # https://st.yandex-team.ru/CHEMODAN-39934
                        if address.path.startswith('/attach/YaFotki/'):
                            if full_index is None:
                                full_index = resource.get_full_index(safe=False)
                            for k, v in full_index.iteritems():
                                if v.get('type') == 'file' and v.get('meta', {}).get('fotki_proxy_url') is not None:
                                    self.fotki_proxy_urls.add(v['meta']['fotki_proxy_url'])

                        if address.is_storage:
                            service.make_storage_folder(address)

                    if self.method in ALBUMS_DJFS_ALLOWED_LIST:
                        from mpfs.core.albums.logic.common import send_djfs_albums_callback
                        send_djfs_albums_callback(uid, resource)

                    if self.method in DOCS_REMOVE_ALLOWED_METHODS_LIST:
                        mpfs_queue.put({'uid': uid}, 'delete_objects_in_docs', deduplication_id='delete_objects_in_docs__%s' % uid)
                    if not keep_lock:
                        self.lock_helper.unlock(resource)

                except errors.ResourceLocked:
                    raise
                except Exception:
                    error_log.info(traceback.format_exc())
                    address = Address(rawaddress)
                    service.rollback(address.uid)
                    if resource:
                        self.lock_helper.unlock(resource)
                    else:
                        self.lock_helper.unlock(rawaddress)
                    raise
        except errors.ResourceNotFound:
            raise errors.RmNotFound(rawaddress)
        except errors.NotPermitted:
            raise errors.RmNotPermitted(rawaddress)
        except errors.AddressError:
            raise errors.RmPathError(rawaddress)

    def check_copy_prerequisites(self, uid, path, resources):
        """
        Проверяет предусловия копирования ресурсов.

        Взято из mpfs.core.social.publicator.Publicator#_grab

        :param uid: Владелец диска в который копируются ресурсы.
        :param path: Путь куда копируются ресурсы.
        :param resources: Список копируемых ресурсов.
        """
        if isinstance(path, Address):
            target_address = path
        else:
            target_address = Address(path, uid=uid, is_folder=True)

        required_space = reduce(lambda s, r: s + getattr(r, 'size', 0), resources, 0)

        self.check_available_space_on_copy_by_raw_address(uid=uid, dst_raw_address=target_address.id, required_space=required_space)

        for res in resources:
            if res.is_infected():
                raise errors.CopyInfectedFileError()

        target_parent_resource = factory.get_resource(uid, target_address.get_parent())
        target_parent_resource.check_rw()
        self.check_lock(target_address.id)

    def copy_resources(self, uid, path, resources, new_names=None, async=False, force_djfs_albums_callback=False):
        """
        Копирует пачку ресурсов в папку path или один ресурс в новый path.

        :param uid: UID пользователя которому принадлежит path.
        :param path: Путь к целевой родительской папке.
        :param resources: Ресурсы которые следует скопировать.
        :param new_names: Имена под которыми будут сохранены ресурсы.
        :param async: Асинхронно или синхронно копировать ресурсы.
        :return: Список скопированных ресурсов или асинхронную операцию.
        """
        assert isinstance(resources, (list, tuple))
        assert not [r for r in resources if not isinstance(r, (base_resources.Resource))]  # все элементы -- ресурсы
        assert new_names is None or (isinstance(new_names, (list, tuple)) and len(new_names) == len(resources))

        if isinstance(path, Address):
            target_dir_address = path
        else:
            target_dir_address = Address(path, uid=uid, is_file=False, is_folder=True)

        self.check_copy_prerequisites(target_dir_address.uid, target_dir_address, resources)

        if not self.exists(target_dir_address.uid, target_dir_address.id):
            self.mkdir(target_dir_address.uid, target_dir_address.id, notify=True)

        ret = []
        if not async:
            for i, r in enumerate(resources):
                r_new_address = target_dir_address.get_child(r.address.name)
                if new_names:
                    r_new_address = r_new_address.rename(new_names[i])
                result = self.copy_resource(r_new_address.uid, r.address.id, r_new_address.id, force=True,
                                            return_resource=True, force_djfs_albums_callback=force_djfs_albums_callback)
                ret.append(result)
            return ret
        else:
            from mpfs.core.operations import manager

            odata_cmd = []
            for i, r in enumerate(resources):
                r_new_address = target_dir_address.get_child(r.address.name)
                if new_names:
                    r_new_address = r_new_address.rename(new_names[i])
                odata_cmd.append({
                    'action': 'copy',
                    'params': {
                        'uid': uid,
                        'src': r.address.id,
                        'dst': r_new_address.id,
                        'force': True,
                    }
                })
            operation = manager.create_operation(
                uid,
                'bulk',
                'filesystem',
                odata={'cmd': to_json(odata_cmd)},
            )
            return operation

    def async_copy_resources(self, uid, path, resources, new_names=None, force_djfs_albums_callback=False):
        """
        Асинхронно копирует пачку ресурсов в папку path или один ресурс в новый path.

        :param uid: UID пользователя которому принадлежит path.
        :param path: Путь к целевой родительской папке.
        :param resources: Ресурсы которые следует скопировать.
        :param new_names: Имена под которыми будут сохранены ресурсы.
        :param force_djfs_albums_callback: Надо ли форсировать постановку таска для djfs альбомов.
        :return: Асинхронную операцию.
        """
        return self.copy_resources(uid, path, resources, new_names=new_names, async=True,
                                   force_djfs_albums_callback=force_djfs_albums_callback)

    def copy_resource(self, uid, source, target, force, return_resource=False, lock_source=True, make_visible=False,
                      src_uid=None, lock_data=None, force_djfs_albums_callback=False, **kw):
        """
        Копирование ресурса
        """
        if source == target:
            raise errors.CopySame()

        src_address = Address(source)
        try:
            resource_uid = src_uid or uid
            src_resource = factory.get_resource(resource_uid, src_address)
        except errors.ResourceNotFound:
            raise errors.CopyNotFound(source)
        else:
            tgt_address = src_address.clone(target)
            if tgt_address.storage_name == PHOTOUNLIM_AREA:
                raise errors.Forbidden('Copying to %s not permitted' % PHOTOUNLIM_AREA_PATH)
            try:
                tgt_parent = factory.get_resource(uid, tgt_address.get_parent())
            except errors.ResourceNotFound:
                raise errors.CopyParentNotFound()
            tgt_parent.check_rw()
            try:
                if lock_source:
                    self.lock_helper.lock(src_resource, lock_data, operation=self.method)
            except (errors.ResourceLocked, errors.ResourceLockFailed):
                raise
            else:
                src_parent = factory.get_resource(uid, src_address.get_parent())
                src_resource.set_parent(src_parent)
                copy_infected_files = not src_resource.is_fully_public()
                try:
                    if tgt_parent.is_group or tgt_parent.is_shared:
                        uids = tgt_parent.group.all_uids()
                        old_version = tgt_parent._service.get_version(uids)
                    else:
                        old_version = src_resource.version

                    tgt_service = factory.get_service(tgt_address)
                    src_diff, dst_diff, _changes, copy_version, overwritten, _ = self.base_copy(
                        uid,
                        source,
                        target,
                        force,
                        copy_symlinks=False,
                        rm_symlinks=False,
                        copy_infected_files=copy_infected_files,
                        exception_keyset=('ctime', 'mtime',),
                        src_uid=src_uid,
                        **kw
                    )
                    tgt_service.commit(tgt_address.uid)
                    tgt_resource = factory.get_resource(uid, tgt_address)
                    tgt_resource.set_parent(tgt_parent)

                    # Пушим данные для djfs альбомов
                    if self.method in ALBUMS_DJFS_ALLOWED_LIST or force_djfs_albums_callback:
                        from mpfs.core.albums.logic.common import send_djfs_albums_callback
                        send_djfs_albums_callback(uid, tgt_resource)

                    # Пушим данные в xiva
                    if tgt_parent.is_group or tgt_parent.is_shared:
                        self.xiva_helper.add_to_xiva_group(
                            uid,
                            tgt_parent.group.gid,
                            _changes,
                            old_version,
                            copy_version
                        )
                        self.xiva_helper.add_to_xiva_space_group(uid, tgt_parent.group.gid)
                    else:
                        self.xiva_helper.add_to_xiva(
                            tgt_address.uid,
                            _changes,
                            old_version,
                            copy_version
                        )
                        self.xiva_helper.add_to_xiva_space(uid)

                    if self.request:
                        self.request.form = tgt_resource.form

                    if make_visible is True:
                        tgt_resource.meta['visible'] = 1
                        tgt_resource.save()

                    if not return_resource:
                        result = tgt_resource.list()
                    else:
                        result = tgt_resource
                except errors.MPFSError:
                    if lock_source:
                        self.lock_helper.unlock(src_resource)
                    tgt_service.rollback(tgt_address.uid)
                    raise
                except Exception:
                    if lock_source:
                        self.lock_helper.unlock(src_resource)
                    raise
                else:
                    if lock_source:
                        self.lock_helper.unlock(src_resource)

                    if self.method == 'copy_resource':
                        event = fs_events.FilesystemCopyEvent(uid=str(uid), force=bool(force),
                                                              src_resource=src_resource, src_address=src_address,
                                                              tgt_resource=tgt_resource, tgt_address=tgt_address,
                                                              overwritten = overwritten)
                        event.send_self_or_group(target_parent=tgt_parent)

                    return result

    def move_resource(self, uid, source, target, force, copy_symlinks=True,
                      rm_symlinks=False, parent_check=True, lock=True, exception_keyset=(), lock_data=None,
                      photounlim_allowed=False, check_hids_blockings=True, notify_search=True,
                      is_live_photo=False):
        self.check_moveability(source, target)

        src_address = Address(source)
        tgt_address = src_address.clone(target)
        if src_address.storage_name != tgt_address.storage_name == PHOTOUNLIM_AREA and not photounlim_allowed:
            raise errors.Forbidden('Moving to %s not permitted' % PHOTOUNLIM_AREA_PATH)

        src_service = factory.get_service(src_address)
        tgt_service = factory.get_service(tgt_address)
        source_parent = factory.get_resource(uid, src_address.get_parent())

        src_resource = factory.get_resource(uid, src_address)
        try:
            parent_addr = tgt_address.get_parent()
            target_parent = factory.get_resource(uid, parent_addr)
        except errors.ResourceNotFound:
            raise errors.CopyParentNotFound()

        self.check_available_space_on_move_by_resource(uid=uid,
                                                       src_resource=src_resource,
                                                       dst_parent_resource=target_parent)

        src_resource.check_rw(operation='move')

        if src_resource.type == 'file':
            notify_storage_clean_check_resource(src_resource)

        src_resource.set_parent(source_parent)

        overwritten = False

        force_having_yarovaya_mark_to_dst = (src_resource.should_yarovaya_mark_be_set()
                                             if FEATURE_TOGGLES_SETTING_YAROVAYA_MARK_ENABLED
                                             else False)
        try:
            if lock:
                self.lock_helper.lock(src_resource, lock_data, operation=self.method)
        except (errors.ResourceLocked, errors.ResourceLockFailed):
            raise
        else:
            try:
                tgt_resource = None

                target_parent.check_rw()

                src_shared = src_resource.is_shared or src_resource.is_group or src_resource.with_shared
                if src_shared and (target_parent.is_fully_public() or
                                   target_parent.is_public_internal()):
                    src_resource.check_rw(operation='set_public')

                if src_resource.is_public_internal() and not src_resource.is_fully_public():
                    if not src_resource.has_same_public_parent(parent_addr.path):
                        rm_symlinks = True

                try:
                    tgt_resource = factory.get_resource(uid, tgt_address)
                except errors.ResourceNotFound:
                    pass

                if source_parent and (source_parent.is_group or source_parent.is_shared):
                    uids = source_parent.group.all_uids()
                else:
                    uids = uid
                if source_parent:
                    old_version_src = source_parent._service.get_version(uids)
                else:
                    old_version_src = src_resource._service.get_version(uids)

                if target_parent.is_group or target_parent.is_shared:
                    uids = target_parent.group.all_uids()
                else:
                    uids = uid
                old_version_tgt = target_parent._service.get_version(uids)

                tgt_service.start(src_address.uid)

                def notify(new_version_tgt, new_version_src=None, tgt_diff=None, src_diff=None):
                    tgt_diff = [] if tgt_diff is None else tgt_diff
                    src_diff = [] if src_diff is None else src_diff
                    diff = {
                        'op': 'deleted',
                        'key': src_resource.id,
                        'resource_type': src_resource.type,
                    }
                    try:
                        diff['fid'] = src_resource.meta['file_id']
                    except KeyError:
                        if not (self.method == 'trash_append' and src_resource.path == '/disk'):
                            error_log.error(traceback.format_exc())
                    if src_resource.type == 'file':
                        diff.update({
                            'md5': src_resource.meta['md5'],
                            'sha256': src_resource.meta['sha256'],
                            'size': src_resource.size,
                        })
                    elif src_resource.is_group_root:
                        diff['shared_rights'] = 'owner'
                        diff['is_group_root'] = 1
                    elif src_resource.is_shared_root:
                        diff['shared_rights'] = str(src_resource.link.rights)
                        diff['is_group_root'] = 1
                    elif src_resource.with_shared:
                        diff['with_shared'] = 1
                    src_diff.append(diff)
                    source_parent_shared = source_parent and (source_parent.is_shared or source_parent.is_group)
                    target_parent_shared = target_parent.is_shared or target_parent.is_group

                    # Пушим данные в xiva
                    if source_parent_shared or target_parent_shared:
                        src_gid = source_parent.group.gid if source_parent_shared else None
                        tgt_gid = target_parent.group.gid if target_parent_shared else None
                        src_diff = {'data': src_diff, 'gid': src_gid}
                        tgt_diff = {'data': tgt_diff, 'gid': tgt_gid}

                        if src_resource.is_shared and tgt_service.name == 'trash':
                            self.xiva_helper.add_to_xiva_group_trash_append(uid, src_diff, tgt_diff,
                                                                            old_version_src, new_version_src)
                        else:
                            self.xiva_helper.add_to_xiva_group_move(uid, src_diff, tgt_diff,
                                                                    old_version_src, old_version_tgt,
                                                                    new_version_src, new_version_tgt,)
                        if target_parent_shared:
                            self.xiva_helper.add_to_xiva_space_group(uid, target_parent.group.gid)
                    else:
                        src_diff.extend(tgt_diff)
                        self.xiva_helper.add_to_xiva(
                            tgt_address.uid,
                            src_diff,
                            old_version_tgt,
                            new_version_tgt,
                        )

                # https://jira.yandex-team.ru/browse/CHEMODAN-7425
                source_is_shared = \
                    src_resource.is_group_root or \
                    src_resource.is_shared_root or \
                    src_resource.with_shared

                destination_is_shared = \
                    target_parent.is_shared or \
                    target_parent.is_group or \
                    (tgt_resource and
                        (tgt_resource.is_shared or
                         tgt_resource.is_group,
                         tgt_resource.with_shared))

                source_not_shared_and_target_shared_root = \
                    not source_is_shared and tgt_resource and \
                    (tgt_resource.is_shared_root or
                     tgt_resource.is_group_root or
                     tgt_resource.with_shared)

                is_quick_move_done = False
                skip_trash_append = False
                src_diff = []
                if (source_is_shared and destination_is_shared) or \
                        source_not_shared_and_target_shared_root:
                    raise errors.MoveWrongDestination()
                elif (src_resource.is_group_root or src_resource.is_shared_root) and tgt_service.name != 'trash':
                    tgt_diff = []

                    if lock:
                        self.lock_helper.lock_by_address(tgt_address, lock_data, operation=self.method)
                    try:
                        src_resource.rename(tgt_address)
                    finally:
                        if lock:
                            # чтобы не прятать настоящее исключение
                            try:
                                self.lock_helper.unlock(tgt_address.id)
                            except Exception:
                                error_log.exception('unlocking %s failed' % tgt_address)


                    # Добавляем дифф
                    tgt_resource = factory.get_resource(uid, tgt_address)
                    tgt_index = tgt_resource.get_full_index()

                    for resource_key, resource_data in tgt_index.iteritems():
                        if resource_key == tgt_address.path:
                            continue
                        else:
                            diff = {
                                'op': 'new',
                                'key': resource_key,
                                'fid': resource_data['meta']['file_id'],
                                'resource_type': 'dir',
                            }
                            if resource_data['type'] == 'file':
                                diff.update({
                                    'md5': resource_data['meta']['md5'],
                                    'sha256': resource_data['meta']['sha256'],
                                    'size': resource_data['size'],
                                    'resource_type': 'file',
                                })
                            tgt_diff.append(diff)

                    diff = {
                        'op': 'new',
                        'key': tgt_address.path,
                        'is_group_root': 1,
                        'fid': tgt_resource.meta['file_id'],
                        'resource_type': 'dir',
                    }
                    if src_resource.is_group_root:
                        diff['shared_rights'] = 'owner'
                    elif src_resource.is_shared_root:
                        diff['shared_rights'] = str(src_resource.link.rights)

                    tgt_diff.append(diff)
                    if notify_search:
                        self.disk_indexer.push(tgt_resource, 'modify', operation=self.method)
                    new_version_tgt = src_service.get_version(tgt_address.uid)
                    new_version_src = None
                elif src_resource.is_shared_root and tgt_service.name == 'trash':
                    src_resource.rm()
                    self._send_move_event(uid, force, src_resource, src_address, tgt_address,
                                          source_parent, target_parent)

                    # TODO: send delete to search for user
                    return {}
                else:
                    if (FEATURE_TOGGLES_CORRECT_SPACE_CHECKS_FOR_SHARED_FOLDERS and
                            src_resource.is_shared and tgt_service.name == 'trash' and
                            ignores_shared_folders_space(uid=uid) and self.quota.free(uid=uid) <= 0):
                        skip_trash_append = True

                    is_dst_merge_operation = force and tgt_resource is not None

                    if (is_quick_move_enabled(uid) and isinstance(src_resource, DiskFolder) and
                            not is_dst_merge_operation):
                        if lock:
                            self.lock_helper.lock_by_address(tgt_address, lock_data, operation=self.method)
                        new_version_tgt = None
                        try:
                            is_quick_move_done, new_version_tgt, tgt_diff = self._quick_move_folder(
                                uid, src_resource, tgt_address, target_parent,
                                force_having_yarovaya_mark=force_having_yarovaya_mark_to_dst, force=force
                            )
                        finally:
                            if lock:
                                # чтобы не прятать настоящее исключение
                                try:
                                    self.lock_helper.unlock(tgt_address.id)
                                except Exception:
                                    error_log.exception('unlocking %s failed' % tgt_address)
                        new_version_src = None

                    if not is_quick_move_done:
                        if lock:
                            self.lock_helper.lock_by_address(tgt_address, lock_data, operation=self.method)
                        try:
                            new_version_copy, overwritten, tgt_diff = self._copy_and_remove(
                                copy_symlinks, exception_keyset,
                                force, overwritten, rm_symlinks,
                                source, src_resource, target,
                                tgt_service, uid,
                                skip_copy=skip_trash_append,
                                check_hids_blockings=check_hids_blockings,
                                notify_search=notify_search,
                                force_having_yarovaya_mark=force_having_yarovaya_mark_to_dst,
                            )

                        finally:
                            if lock:
                                # чтобы не прятать настоящее исключение
                                try:
                                    self.lock_helper.unlock(tgt_address.id)
                                except Exception:
                                    error_log.exception('unlocking %s failed' % tgt_address)

                        new_version_remove = None
                        if not skip_trash_append:
                            new_version_remove = src_service.get_version(tgt_address.uid)
                        source_parent_shared = source_parent and (source_parent.is_shared or source_parent.is_group)

                        target_parent_shared = (target_parent.is_shared or target_parent.is_group)

                        target_group_equal_source = False
                        target_group_not_equal_source = False
                        if target_parent_shared and source_parent_shared:
                            if target_parent.group.gid != source_parent.group.gid:
                                target_group_not_equal_source = True
                            else:
                                target_group_equal_source = True

                        if (notify_search and
                                source_parent_shared and
                                (not target_parent_shared or target_group_not_equal_source)):
                            self.disk_indexer.push(
                                src_resource,
                                'delete',
                                operation='move'
                            )

                        if target_group_equal_source:
                            new_version_src = None
                            new_version_tgt = new_version_remove
                        elif target_group_not_equal_source:
                            new_version_src = new_version_remove
                            new_version_tgt = new_version_copy
                        elif source_parent_shared:
                            new_version_src = new_version_remove
                            new_version_tgt = new_version_copy
                        elif target_parent_shared:
                            new_version_src = new_version_remove
                            new_version_tgt = new_version_copy
                        else:
                            new_version_src = None
                            new_version_tgt = new_version_remove
                    else:
                        if self.method in ALBUMS_DJFS_ALLOWED_LIST:
                            from mpfs.core.albums.logic.common import send_djfs_albums_callback
                            send_djfs_albums_callback(uid, tgt_resource)
                tgt_service.commit(src_address.uid)

                if not skip_trash_append:
                    tgt_resource = factory.get_resource(uid, tgt_address)

                notify(new_version_tgt, new_version_src, tgt_diff=tgt_diff, src_diff=src_diff)

                result = {}
                if not skip_trash_append:
                    if is_quick_move_done:
                        result = {
                            'this': {
                                'id': tgt_resource.path,
                                'id_old': src_address.path,
                            }
                        }
                    else:
                        tgt_resource.id_old = src_address.path
                        if self.request:
                            self.request.form = tgt_resource.form
                        result = tgt_resource.list()
                        result['this']['id_old'] = src_address.path

                self.lock_helper.unlock(src_resource)

                self._send_move_event(uid, force, src_resource, src_address, tgt_address,
                                      source_parent, target_parent, tgt_resource, overwritten, is_live_photo)

                return result

            except errors.MPFSError:
                self.lock_helper.unlock(src_resource)
                tgt_service.rollback(tgt_address.uid)
                raise

    @staticmethod
    def _is_folder_contains_shared_folders(uid, path):
        from mpfs.core.social.share import ShareProcessor
        spare_processor = ShareProcessor()

        groups = spare_processor.list_owned_groups(uid)
        for group in groups:
            if group.path.startswith(path):
                return True
        links = spare_processor.list_owned_links(uid)
        for link in links:
            if link.path.startswith(path):
                return True

        return False

    @staticmethod
    def _check_target_resource_exists(uid, tgt_address):
        try:
            factory.get_resource(uid, tgt_address)
        except errors.ResourceNotFound:
            pass
        else:
            raise errors.CopyTargetExists()

    def _quick_move_folder(self, uid, src_resource, tgt_address, tgt_parent, force_having_yarovaya_mark=False,
                           force=False):
        is_quick_move_done = False
        tgt_diff = None
        new_folder_version = None
        usrctl = mpfs.engine.process.usrctl()

        if (usrctl.is_user_in_postgres(uid) and
                not (src_resource.is_group or src_resource.is_shared) and
                not (tgt_parent.is_group or tgt_parent.is_shared) and
                src_resource.address.storage_name == tgt_address.storage_name):
            # прототип - только для перемещений внутри диска пользователя, только в своих папках,
            # которые не являются общими, и только в пределах одного раздела, т.к. при перемещениях
            # между разными разделами часто нужно делать доп. работу, например, при перемещении в trash
            # или hidden надо проставлять время удаления

            if self._is_folder_contains_shared_folders(uid, src_resource.address.path):
                return is_quick_move_done, new_folder_version, tgt_diff
            if not force:
                self._check_target_resource_exists(uid, tgt_address)

            service = factory.get_service_by_root_folder_path(src_resource.address.storage_path)
            resources = list(service.control.iter_subtree(
                uid, src_resource.address.path, limit=POSTGRES_QUICK_MOVE_FOLDER_RESOURCES_LIMIT)
            )

            tgt_diff = []
            changelog_deltas = []
            new_folder_version = generate_version_number()
            last_changelog_version = new_folder_version
            should_update_last_quick_move_version = False

            tgt_diff.append({
                'op': 'new',
                'resource_type': 'dir',
                'key': tgt_address.path,
                'fid': src_resource.meta['file_id'],
            })

            if len(resources) < POSTGRES_QUICK_MOVE_FOLDER_RESOURCES_LIMIT:
                # do the same quick move as in the condition below
                # gather all the data needed for changelog and insert it in one query
                # gather all the data needed for xiva push and make tasks for it (move xiva push tasks creation from
                #  outer scope function here)
                # increase user disk version but do not increase last quick move version
                changelog_deltas.append({
                    'uid': uid,
                    'version': new_folder_version,
                    'zdata': compress_data({
                        'op': 'new',
                        'type': 'dir',
                        'key': tgt_address.path,
                        'fid': src_resource.meta['file_id'],
                        'public': int(bool(src_resource.meta.get('public'))),
                        'visible': int(src_resource.meta.get('visible', 1)),
                    }),
                    'dtime': datetime.datetime.now(),
                })

                for resource in resources:
                    new_address = Address.Make(uid, resource['key'])
                    new_address.change_parent(tgt_address, old_parent=src_resource.address)
                    changelog_delta, xiva_data = self._generate_changelog_and_xiva_data(uid, resource, new_address)
                    changelog_deltas.append(changelog_delta)
                    tgt_diff.append(xiva_data)

                last_changelog_version = generate_version_number()
                changelog_deltas.append({
                    'uid': uid,
                    'version': last_changelog_version,
                    'zdata': compress_data({
                        'op': 'deleted',
                        'type': 'dir',
                        'key': src_resource.address.path,
                        'fid': src_resource.meta['file_id'],
                        'public': int(bool(src_resource.meta.get('public'))),
                        'visible': int(src_resource.meta.get('visible', 1)),
                    }),
                    'dtime': datetime.datetime.now(),
                })
            else:
                changelog_deltas.append({
                    'uid': uid,
                    'version': new_folder_version,
                    'zdata': compress_data({
                        'op': 'moved',
                        'type': 'dir',
                        'new_key': tgt_address.path,
                        'key': src_resource.address.path,
                        'fid': src_resource.meta['file_id'],
                    }),
                    'dtime':  datetime.datetime.now(),
                })
                should_update_last_quick_move_version = True

            is_quick_move_done = True

            FolderDAO().move(
                uid, DAOPath(src_resource.address.path), DAOPath(tgt_address.path), new_folder_version,
                force_having_yarovaya_mark=force_having_yarovaya_mark
            )
            ChangelogCollection().insert(changelog_deltas)
            if should_update_last_quick_move_version:
                usrctl.set_last_quick_move_version(uid, new_folder_version)
            ChangelogCollection.update_user_disk_version(uid, last_changelog_version)

            resource = factory.get_resource(uid, tgt_address.path)
            self.disk_indexer.push(resource, 'modify', operation='move_resource', update_photoslice=True)

        return is_quick_move_done, new_folder_version, tgt_diff

    @staticmethod
    def _generate_changelog_and_xiva_data(uid, resource, new_address):
        data = UserCollectionZipped.unzip_resource_data(
            resource['type'], resource['data'], resource['zdata']
        )

        changelog_zipped_data = {
            'op': 'new',
            'type': resource['type'],
            'key': new_address.path,
        }
        xiva_data = {
            'op': 'new',
            'key': new_address.path,
            'fid': data['meta'].get('file_id'),
            'resource_type': resource['type'],
        }

        changelog_zipped_data.update(UserDataCollection.get_folder_version_data(data))
        if resource['type'] == 'file':
            changelog_zipped_data.update(UserDataCollection.get_file_version_data(data))
            xiva_data.update({
                'sha256': data['meta']['sha256'],
                'md5': data['meta']['md5'],
                'size': data.get('size'),
            })

        changelog_delta = {
            'uid': uid,
            'version': generate_version_number(),
            'zdata': compress_data(changelog_zipped_data),
            'dtime': datetime.datetime.now(),
        }
        return changelog_delta, xiva_data

    def _copy_and_remove(self, copy_symlinks, exception_keyset, force, overwritten, rm_symlinks, source, src_resource,
                         target, tgt_service, uid, skip_copy=False, check_hids_blockings=True, notify_search=True,
                         force_having_yarovaya_mark=False):
        new_version_copy = None
        user_diff = None
        src_res = None
        if not skip_copy:
            _, _, user_diff, new_version_copy, overwritten, src_res = self.base_copy(
                uid,
                source,
                target,
                force,
                mv=True,
                copy_symlinks=copy_symlinks,
                rm_symlinks=rm_symlinks,
                exception_keyset=exception_keyset,
                check_hids_blockings=check_hids_blockings,
                notify_search=notify_search,
                force_having_yarovaya_mark=force_having_yarovaya_mark,
                set_target_lock=False,
            )
            # Для логгирования событий о удалении каждой подпапки
            # в event_history/event_subscriptions.py#_log_fs_delete_subdirs_if_needed
            if src_res.is_folder:
                src_resource.full_index_map = src_res.full_index_map

        if src_resource.is_shared and tgt_service.name == 'trash':
            tgt_diff = {}
            if user_diff is not None:
                tgt_diff[uid] = user_diff
            owner_tgt_address = Address(src_resource.storage_address.id)
            owner_tgt_address.change_storage('trash')
            owner_tgt_address.drop_path_to_root()
            owner_uid = src_resource.storage_address.uid

            try:
                factory.get_resource(owner_uid, owner_tgt_address)
            except errors.ResourceNotFound:
                pass
            else:
                # меняется имя, если такое уже есть в базе
                owner_tgt_address.id += '_%s' % int(time.time())
                owner_tgt_address.path += '_%s' % int(time.time())
            exception_keyset = exception_keyset + ('file_id',)
            _, _, owner_diff, _, _, _ = self.base_copy(
                owner_uid,
                src_resource.storage_address.id,
                owner_tgt_address.id, force, mv=True,
                copy_symlinks=copy_symlinks,
                rm_symlinks=rm_symlinks,
                exception_keyset=exception_keyset,
                check_hids_blockings=check_hids_blockings,
                notify_search=notify_search,
                force_having_yarovaya_mark=force_having_yarovaya_mark,
                set_target_lock=False,
                trash_append_to_owner_by_guest=True,
            )
            tgt_diff[owner_uid] = owner_diff
        else:
            tgt_diff = user_diff
        self.rm(
            uid, source,
            drop=True,
            notify=False,
            keep_group=False,
            notify_search=False,
            mv=True,
            lock=False,
        )

        return new_version_copy, overwritten, tgt_diff

    def _send_move_event(self, uid, force, src_resource, src_address, tgt_address, source_parent, target_parent,
                         tgt_resource=None, overwritten=False, is_live_photo=False):
        base_event = None
        if self.method == 'move_resource':
            base_event = fs_events.FilesystemMoveEvent(uid=str(uid), force=bool(force), overwritten=overwritten,
                                                       src_resource=src_resource, src_address=src_address,
                                                       tgt_resource=tgt_resource, tgt_address=tgt_address,
                                                       is_live_photo=is_live_photo)
        elif self.method in ('trash_append', 'trash_append_file'):
            base_event = fs_events.FilesystemTrashAppendEvent(uid=str(uid), force=bool(force),
                                                              src_resource=src_resource, src_address=src_address,
                                                              tgt_resource=tgt_resource, tgt_address=tgt_address)
        elif self.method == 'trash_restore':
            base_event = fs_events.FilesystemTrashRestoreEvent(uid=str(uid), force=bool(force),
                                                               type='trash_restore', subtype=tgt_address.storage_name,
                                                               src_resource=src_resource, src_address=src_address,
                                                               tgt_resource=tgt_resource, tgt_address=tgt_address)
        if target_parent and tgt_resource:
            tgt_resource.set_parent(target_parent)

        if source_parent and src_resource:
            src_resource.set_parent(source_parent)

        if base_event is not None:
            base_event.send_self_or_group(source_parent, target_parent)

    def setprop(self, uid, rawaddress, changes, deleted=None, force=False, resource=None, notify_search=True, is_store=False):
        """Изменить/удалить мета-данные

        .. warnings:: Чтобы не менять `mtime` используйте :func:`patch_file`
        """
        if deleted is None:
            deleted = list()
        try:
            for p in ('visible', 'broken'):
                if p in changes:
                    changes[p] = int(changes[p])

            address = Address(rawaddress)
            if resource is None:
                resource = factory.get_resource(uid, address)
            self.check_rights(uid, resource=resource)
            resource.check_rw(operation='setprop')
            resource.set_request(self.request)

            if resource.is_group_internal or resource.is_shared_internal:
                uids = resource.group.all_uids()
                old_versions = resource._service.get_version(uids)
            old_version = resource.version

            diff = []
            if self.method != 'patch_file':
                resource.setprop(changes, deleted, force=force)
            else:
                resource.setprop(changes, deleted, force=force, nomtime=True)
            if resource.version != old_version:
                data = {
                    'op': 'changed',
                    'fid': resource.meta['file_id'],
                    'resource_type': resource.type,
                    'action': 'setprop',
                }
                if resource.external_setprop:
                    data['external_setprop'] = int(resource.external_setprop)
                if resource.is_file:
                    data.update({
                        'md5': resource.meta['md5'],
                        'sha256': resource.meta['sha256'],
                        'size': resource.size,
                    })
                data['key'] = resource.address.path
                new_version = resource.version
                if 'broken' in changes:
                    data['broken'] = 1
                    if len(changes) == 1:
                        new_version = ''
                diff.append(data)
                if resource.is_shared_internal or resource.is_group_internal:
                    self.xiva_helper.add_to_xiva_group(
                        uid,
                        resource.group.gid,
                        diff,
                        old_versions,
                        new_version
                    )
                else:
                    self.xiva_helper.add_to_xiva(
                        uid,
                        diff,
                        old_version,
                        new_version,
                    )
                if notify_search:
                    self.disk_indexer.push(resource, 'modify', operation=self.method, append_djfs_callbacks=is_store)
            return resource
        except errors.ResourceNotFound:
            raise errors.SetpropNotFound(rawaddress)
        except errors.MPFSError:
            raise

    def patch_file(self, *args, **kwargs):
        """Делать тоже, что и :func:`setprop`, но не менять `mtime`
        """
        return self.setprop(*args, **kwargs)

    def add_generated_albums_exclusion(self, uid, path, album_type):
        if album_type not in [atype.value for atype in GeneratedAlbumType]:
            raise AlbumsWrongGeneratedType()

        address = Address(path, uid=uid)
        resource = factory.get_resource(uid, address)

        if resource.is_shared and resource.owner_uid != uid:
            raise share_errors.GroupNoPermit()

        exclusions = set(resource.get_albums_exclusions())
        if album_type in exclusions:
            return exclusions

        if album_type == GeneratedAlbumType.GEO.value:
            djfs_albums_legacy.exclude_photo_from_geo_album(uid, path)

        exclusions.add(album_type)

        self.setprop(uid, address.id, {'albums_exclusions': list(exclusions)})

        return exclusions

    def diff(self, uid, rawaddress, version, allow_quick_move_deltas, load_source_ids=False):
        try:
            address = Address(rawaddress)
            resource = factory.get_resource(uid, address)
            result = resource.history.diff(version, allow_quick_move_deltas)
            if load_source_ids:
                GlobalGalleryController.load_and_set_source_ids_for_resource_docs(result['result'])
            if self.request:
                self.request.form = resource.history.form
        except errors.ResourceNotFound:
            raise errors.DiffNotFound(rawaddress)
        except errors.MPFSError:
            raise
        else:
            return result

    def index(self, uid, rawaddress):
        try:
            address = Address(rawaddress)
            history = History(address.uid)
            return history.index(address)
        except errors.ResourceNotFound:
            raise errors.DiffNotFound(rawaddress)
        except errors.MPFSError:
            raise

    def space(self, uid):
        return self.quota.report(uid)

    def hardlink(self, md5=None, size=None, sha256=None, hid=None):
        if hid:
            return hardlinks.HardLink(hid=hid)
        else:
            return hardlinks.HardLink(md5, size, sha256)

    def hardlink_copy(self, uid, rawaddress, md5, size, sha256,
                      force=False, keep_symlinks=False, changes=None,
                      replace_md5=None, notify=True, notify_search=True,
                      filedata=None, keep_lock=False, oper_type='overwrite', oper_subtype='', suppress_event=False,
                      etime_from_client=None, source_id=None, is_live_photo=False,
                      force_deletion_log_deduplication=False, photoslice_album_type_data=None, replace_hid=None):

        path = Address(rawaddress)
        if path.storage_name == 'photostream' or path.storage_name == PHOTOUNLIM_AREA:
            try:
                self.check_photostream_file_existing(uid, rawaddress, size, md5, sha256, is_live_photo=is_live_photo)
                if (force_deletion_log_deduplication
                        and GlobalGalleryController.is_hid_presented_in_deletion_log(
                            uid, FileChecksums(md5, sha256, size).hid, is_live_photo=is_live_photo)):
                    raise errors.FilePresentedInDeletionLog(
                        "uid=%s hid=%s" % (uid, FileChecksums(md5, sha256, size).hid))
            except errors.LivePhotoUploadAttemptRegularPhotoExists:
                raise
            except errors.FileAlreadyExist:
                if source_id:
                    mpfs_queue.put({
                            'uid': uid,
                            'hid': FileChecksums(md5, sha256, int(size)).hid,
                            'source_ids': [source_id, ],
                            'is_live_photo': is_live_photo,
                        },
                        'add_source_ids')
                raise

        link = hardlinks.HardLink(md5, size, sha256)
        notify_storage_clean_check_stids([link.file_id, link.digest_id, link.pmid] + [stid for _, stid in link.previews.iteritems()])

        file_hid = Binary(hardlinks.common.construct_hid(md5, size, sha256))
        self.check_hids_blockings([file_hid, ])

        if filedata is None:
            filedata = {}

        metadata = filedata.get('meta', {})
        filedata.update({
            'size': link.size,
            'mimetype': link.mimetype,
        })

        metadata.update({
            'file_mid': link.file_id,
            'digest_mid': link.digest_id,
            'drweb': link.drweb,
            'md5': link.md5,
            'sha256': link.sha256,
            'modify_uid': uid
        })

        fields_to_copy = ('pmid', 'etime', 'video_info', 'emid', 'width', 'height', 'angle', 'aesthetics')
        if not FEATURE_TOGGLES_DISABLE_OLD_PREVIEWS:
            fields_to_copy += ('previews',)

        for meta_field in fields_to_copy:
            link_value = getattr(link, meta_field)
            if link_value:
                metadata[meta_field] = link_value

        if path.storage_name == 'photostream':
            rawaddress = self.preprocess_path(uid, rawaddress)
            self.mkdir_photostream_root(uid)
            try:
                rawaddress = self.check_photostream_file_existing(
                    uid, rawaddress, size, md5, sha256, is_live_photo=is_live_photo, replace_hid=replace_hid)
                if (force_deletion_log_deduplication
                        and GlobalGalleryController.is_hid_presented_in_deletion_log(
                            uid, FileChecksums(md5, sha256, size).hid, is_live_photo=is_live_photo)):
                    raise errors.FilePresentedInDeletionLog(
                        "uid=%s hid=%s" % (uid, FileChecksums(md5, sha256, size).hid))
            except errors.LivePhotoUploadAttemptRegularPhotoExists:
                raise
            except errors.FileAlreadyExist:
                if source_id:
                    mpfs_queue.put({
                            'uid': uid,
                            'hid': FileChecksums(md5, sha256, int(size)).hid,
                            'source_ids': [source_id, ],
                            'is_live_photo': is_live_photo,
                        },
                        'add_source_ids')
                raise

        if path.storage_name == 'attach':
            from mpfs.core.user.base import NeedInit, Create
            if NeedInit(uid):
                Create(uid)
            # Находимся в процессе закгрузки аттача, следовательно, должны быть созданы аттачевые домейны и папки,
            # пользователь не будет создан
            if NeedInit(uid, type='attach'):
                Create(uid, type='attach')

            """
            Худший код в истории mpfs :(

            Если hardlink_copy дернули при store (и передали размер, sha и прочее), то
            этот if сработает и выставит файлу все аттачевые атрибуты (имя c дополненным
            таймстемпом и original_name в мету).

            Если hardlink_copy дернули из объекта операции (см. StoreAttach), то у него
            уже есть все эти поля и их не надо проставлять второй раз

            Таск: https://st.yandex-team.ru/CHEMODAN-21391
            """
            if not ('original_name' in filedata and path.name != filedata['original_name']):
                original_name = path.name
                path = path.rename('%s_%s' % (original_name, time.time()))
                rawaddress = path.id
                metadata['original_name'] = original_name

        # проставляем etime, который передал клиент, даже если в базе есть etime
        # https://st.yandex-team.ru/CHEMODAN-41047
        if etime_from_client is not None:
            # Сохраняем etime с клиента только для картинок и видео
            # https://st.yandex-team.ru/CHEMODAN-41657
            mimetype = filedata['mimetype'] if 'mimetype' in filedata else None
            if getGroupByName(path.id, mtype=mimetype) in ('image', 'video',):
                metadata['etime'] = int(etime_from_client)

        filedata['meta'] = metadata

        autouploaded = (path.storage_name == PHOTOSTREAM_AREA or
                        path.storage_name == PHOTOUNLIM_AREA)
        if photoslice_album_type_data:
            filedata['meta']['photoslice_album_type'] = resolve_photoslice_album_type(
                uid, filedata['mimetype'], photoslice_album_type_data, link=link, autouploaded=autouploaded)

        result = self.mkfile(
            uid,
            rawaddress,
            data=filedata,
            keep_symlinks=keep_symlinks,
            notify=notify,
            notify_search=notify_search,
            keep_lock=keep_lock,
            is_store=True
        )
        if source_id and not result.is_shared:
            mpfs_queue.put({
                'uid': uid,
                'hid': FileChecksums(md5, sha256, int(size)).hid,
                'source_ids': [source_id, ],
                'is_live_photo': is_live_photo
            }, 'add_source_ids')

        if changes is not None:
            parent = result.get_parent()
            result = self.setprop(uid, rawaddress, changes, force=True)
            if parent:
                result.set_parent(parent)

        result.set_request(self.request, force=True)

        # suppress_event: логирование события будет отложено, если после хардлинка необходимо опубликовать ресурс
        if self.method in ('store', 'hardlink_copy') and not suppress_event:
            self._send_hardlink_event(result, oper_subtype, oper_type, changes)

        need_regenerate_preview = False
        if link.pmid:
            try:
                need_regenerate_preview = not mulca_service.Mulca().is_file_exist(link.pmid)
            except errors.MulcaNoResponse:
                pass  # не перегенериваем превьюшку в непонятных ситуациях

        if need_regenerate_preview:
            self.async_regenerate_preview_for_files(
                source_file_stid=link.file_id,  # в поле file_id хранится stid
                source_file_name=path.name,
                source_file_size=size,
                source_file_mimetype=link.mimetype,
                old_preview_stid=link.pmid,
                resource_ids=(
                    ResourceId(uid, result.meta['file_id']),
                    ResourceId(link._raw_record.uid, link._raw_record.file_id),
                ),
            )

        return result

    @staticmethod
    def _send_hardlink_event(resource, oper_subtype, oper_type, changes, set_public=False):
        if oper_type in ('overwrite', 'store'):
            oper_subtype = resource.visible_address.storage_name

        parent_addr = resource.address.get_parent()
        parent = factory.get_resource(parent_addr.uid, parent_addr)
        resource.set_parent(parent)

        fs_events.FilesystemHardlinkCopyEvent(tgt_resource=resource, tgt_address=resource.visible_address,
                                              type=oper_type, subtype=oper_subtype,
                                              changes=changes, set_public=set_public,
                                              uid=resource.visible_address.uid, size=safe_to_int(resource.size),
                                              ).send_self_or_group(resource=resource)

    def store(self, uid, raw_address, force=False, changes=None,
              size=0, md5='', sha256='', replace_md5=None,
              client_type='json', user_ip='', use_https=None,
              callback='', connection_id='',
              skip_self_lock=False, skip_check_space=False,
              oper_type='overwrite', oper_subtype='', tld=None,
              live_photo_md5=None, live_photo_sha256=None, live_photo_size=None, etime_from_client=None,
              mtime_from_client=None, ctime_from_client=None, user_agent=None, live_photo_type=None,
              live_photo_operation_id=None, fos_app_version=None, fos_reply_email=None, fos_expire_seconds=None,
              fos_os_version=None, fos_support_text=None, fos_subject=None, fos_recipient_type=None,
              source_id=None, force_deletion_log_deduplication=False, photoslice_album_type_data=None,
              uploader_speed_limit=None, photostream_destination=None):

        """
        Создать операцию загрузки или при возможности сделать хардлинк

        При хардлике возвращается словарь вида: {"status": "hardlinked"}

        `skip_self_lock` используется, чтобы выполнить `store` несмотря на то, что ресурс залочен.
        `skip_check_space` используется, чтобы сохранять файлы при любом свободном месте (для OfficeOnline)

        """
        if not skip_check_space:
            from mpfs.common.util.limits.logic.upload_traffic_limit import check_upload_limits
            check_upload_limits(uid, 0)
            from mpfs.common.util.limits.logic.upload_file_size_limit import check_upload_file_size_limits
            check_upload_file_size_limits(uid, size)

        changes = changes or {}
        replace_hid = None
        if replace_md5 and replace_md5.startswith('md5:'):
            checksum = FileChecksums.from_str(replace_md5)
            if checksum:
                replace_md5 = None
                replace_hid = checksum.hid

        address = Address(raw_address)

        # save client mtime as client etime for photostream uploads
        # https://st.yandex-team.ru/CHEMODAN-43109
        etime_from_client = etime_from_client or ctime_from_client or mtime_from_client

        original_address = deepcopy(address)
        # превращаем /photostream в /photounlim для пользователей с включённым безлимитом
        user = None
        if oper_type in ('overwrite', 'store'):
            if photostream_destination:
                is_forbidden, reason = is_unlim_forbidden_due_to_experiment(uid, address, photostream_destination)
                if is_forbidden:
                    raise errors.ForbiddenVideoUnlim(title=to_json({'reason': reason}))
                if (is_converting_address_to_newunlim_for_uid_needed(address, uid, user_agent)
                        and not is_photounlim_fraudulent_store(address, md5, size)):
                    address.change_storage(PHOTOUNLIM_AREA)
                    raw_address = address.id
            else:
                is_wrong_version = False
                try:
                    parsed_user_agent = UserAgentParser.parse(user_agent)
                    if parsed_user_agent and parsed_user_agent.os and parsed_user_agent.os.lower().startswith('android') and \
                            parsed_user_agent.vsn and (parsed_user_agent.vsn in ["4.42.0-2703", "4.41.2-2652"]):
                        is_wrong_version = True
                except Exception:
                    pass

                if is_wrong_version:
                    if (is_converting_address_to_newunlim_for_uid_needed(address, uid, user_agent)
                            and not is_photounlim_fraudulent_store(address, md5, size)):
                        address.change_storage(PHOTOUNLIM_AREA)
                        raw_address = address.id
                elif (is_converting_address_to_photounlim_for_uid_needed(address, uid, user_agent)
                        and not is_photounlim_fraudulent_store(address, md5, size)):
                    address.change_storage(PHOTOUNLIM_AREA)
                    raw_address = address.id

        original_raw_path = raw_address
        replace_mtime = None
        # Добавляем возможность перезатереть файлы в фотосрезе при передаче replace_md5
        # https://st.yandex-team.ru/CHEMODAN-77309
        if photostream_destination and replace_hid:
            # файл мог быть залит тогда, когда настройки фотоанлима отличались от предыдущих,
            # так что проверяем два назанчения.
            parent_paths = [PHOTOUNLIM_AREA_PATH,
                            self.get_photostream_address(uid).path]
            existing_path, existing_file = FileDAO().find_file_in_folder_by_hid(uid, parent_paths, replace_hid)
            if existing_file:
                existing_raw_path = existing_file.path
                if isinstance(existing_file.modification_time, datetime.datetime):
                    replace_mtime = existing_file.modification_time.get_timestamp()
                # если файл нашёлся в папке Фотокамера, то возвращаем путь /photostream/...
                # для совместимости с последующей обработкой
                address = Address(existing_raw_path, uid=uid)
                raw_address = address.id
                # сбрасываем address в photostream, чтобы правильный operation subtype получить
                if existing_path != PHOTOUNLIM_AREA_PATH:
                    address.change_parent(PHOTOSTREAM_AREA_PATH, old_parent=existing_path)
            else:
                raise errors.PreconditionsFailed

        # мы не пользуемся NeedInit, т.к. не хотим при store ходить по
        # коллекциям и в паспорт.
        if mpfs.engine.process.usrctl().is_user_init(uid):
            # делаем эти проверки только для инициализированных пользователей
            # возможно лучше бы перенести код иницилизации пользователя в mpfs.core.base.store
            # а код ниже выполнять всегда
            self.check_address(raw_address, strict=True)
            self.check_rights(uid, rawaddress=raw_address)
            self.check_lock(raw_address, skip_self_lock)
            self.check_preconditions(uid, raw_address, replace_md5=replace_md5)
            overwrite_file = self.check_overwrite(uid, raw_address, force)
        else:
            overwrite_file = {}

        if not skip_check_space:
            free_space = self.quota.free_with_shared_support(address=address.get_parent())
            self.check_available_space_on_store_by_raw_address(uid, raw_address, required_space=size)
        else:
            free_space = MAX_FILE_SIZE

        # В photounlim файлы не перетираем, а добавляем суффикс
        # https://st.yandex-team.ru/CHEMODAN-39009
        # суффикс не добавляется, если передан replace_md5
        # https://st.yandex-team.ru/CHEMODAN-77309
        if overwrite_file and address.storage_name == PHOTOUNLIM_AREA and not replace_hid:
            address = address.add_suffix('_' + str(int(time.time() * 1000)))
            raw_address = address.id
            overwrite_file = {}

        # Если файл существует - сохраняем его visible для нового файла
        if overwrite_file:
            visible = overwrite_file.get('visible', None)
            file_id = overwrite_file.get('meta', {}).get('file_id', None)
            if force and visible is not None:
                changes['visible'] = visible
                if file_id is not None:
                    changes['file_id'] = file_id

        # Отделяем public от changes
        set_public = None
        if 'public' in changes:
            set_public = int(changes['public']) == 1
            del changes['public']
        elif address.storage_name == 'attach':
            set_public = True

        if not overwrite_file:
            oper_type = 'store'

        if oper_type in ('overwrite', 'store'):
            oper_subtype = address.storage_name

        is_live_photo_store = (
            live_photo_md5 is not None and live_photo_sha256 is not None and live_photo_size is not None
        )
        is_fos_client_store = (
            fos_reply_email is not None and fos_subject is not None and fos_app_version is not None and fos_os_version is not None
        )

        # Сохраняем признак не найденного хардлинка
        hardlink_status = None
        if not is_live_photo_store and ShardedHardLink.is_valid_checksums(md5, sha256, size) and not is_fos_client_store:
            try:
                res = self.hardlink_copy(
                    uid,
                    raw_address,
                    md5,
                    size,
                    sha256,
                    changes=changes,
                    oper_type=oper_type,
                    oper_subtype=oper_subtype,
                    suppress_event=True,
                    etime_from_client=etime_from_client,
                    keep_symlinks=overwrite_file,
                    source_id=source_id,
                    is_live_photo=is_live_photo_store,
                    force_deletion_log_deduplication=force_deletion_log_deduplication,
                    photoslice_album_type_data=photoslice_album_type_data,
                    replace_hid=replace_hid
                )

                if not user:
                    from mpfs.core.user.base import User
                    user = User(uid)
                if client_type == 'desktop':
                    user.set_state(key='file_uploaded', value=1)
                user.set_state(key='first_file', value=1)

                if set_public:
                    self.publicator.set_public(uid, res.address.id, user_ip=user_ip, resource=res,
                                               oper_type=oper_type, oper_subtype=oper_subtype)

                self._send_hardlink_event(res, oper_subtype, oper_type, changes, set_public=set_public)

                self.quota.update_upload_traffic(uid, bytes_uploaded=res.get_size())

                logger.log_stat(uid, oper_type, oper_subtype, int(True), res.get_size(), res.media_type,
                                address.path)
                return {
                    'status': 'hardlinked',
                    '_headers': {
                        'path': res.id,
                        'resource_id': res.resource_id.serialize(),
                    }
                }
            except (errors.StidLockedError, errors.HardLinkNotFound, errors.HardlinkBroken,
                    errors.HardLinkNotReady, errors.HardlinkFileNotInStorage) as e:
                if ShardedHardLink.is_valid_checksums(md5, sha256, size):
                    hardlink_status = (e.__class__.__name__, e.message, e.data)

        from mpfs.core.operations import manager
        odata = {
            'path': raw_address,
            'md5': md5,
            'size': size,
            'sha256': sha256,
            'replace_md5': replace_md5,
            'free_space': free_space,
            'skip_check_space': skip_check_space,
            'callback': callback,
            'changes': changes,
            'connection_id': connection_id,
            'client_type': client_type,
            'set_public': set_public,
            'hardlink_status': hardlink_status,
            'user_ip': user_ip,
        }
        if replace_hid and original_raw_path:
            odata['replace_hid'] = replace_hid
            odata['replace_mtime'] = replace_mtime
            odata['path'] = original_raw_path
            odata['new_path'] = raw_address

        if tld is not None:
            odata['tld'] = tld
        if etime_from_client is not None:
            odata['etime_from_client'] = etime_from_client
        if source_id is not None:
            odata['source_id_to_add'] = source_id
        if uploader_speed_limit:
            odata['upload-max-speed-bps'] = uploader_speed_limit

        if is_live_photo_store:
            odata['live_photo_md5'] = live_photo_md5
            odata['live_photo_sha256'] = live_photo_sha256
            odata['live_photo_size'] = live_photo_size
            odata['live_photo_type'] = live_photo_type
            odata['live_photo_operation_path'] = original_address.id

            user_agent_obj = None
            if user_agent is not None:
                user_agent_obj = UserAgentParser.parse(user_agent)

            robust_store_ios_version = MobileClientVersion.build_from_version(LIVE_PHOTO_ROBUST_STORE_IOS_VERSION)
            if (user_agent_obj is not None and user_agent_obj.is_ios() and
                    user_agent_obj.get_version() >= robust_store_ios_version or live_photo_operation_id is not None):
                odata['live_photo_operation_id'] = live_photo_operation_id
            else:
                odata['id'] = LivePhotoFilesManager.get_live_photo_store_operation_id(
                    uid, original_address.id,
                    md5, sha256, size,
                    live_photo_md5, live_photo_sha256, live_photo_size
                )

        if is_fos_client_store:
            odata['fos_app_version'] = fos_app_version
            odata['fos_reply_email'] = fos_reply_email
            odata['fos_expire_seconds'] = fos_expire_seconds
            odata['fos_os_version'] = fos_os_version
            odata['fos_support_text'] = fos_support_text
            odata['fos_subject'] = fos_subject
            odata['fos_recipient_type'] = fos_recipient_type
        if photoslice_album_type_data:
            odata['photoslice_album_type_data'] = vars(photoslice_album_type_data)

        operation = manager.create_operation(
            uid,
            oper_type,
            oper_subtype,
            odata=odata,
            use_https=use_https,
        )
        return operation

    def deprecated_search(self, uid, rawaddress, request_string):
        address = Address(rawaddress)
        resource = factory.get_resource(uid, address)
        resource.set_request(self.request)
        search_results = factory.search(resource, request_string)

        return {
            'this': resource.info()['this'],
            'search-query': request_string,
            'list': search_results,
        }

    def search(self, uid, rawaddress, request_string, load_source_ids=False, **kwargs):
        """
        Search for request_string.

        Adds 'search_scope' key to meta field of found resources.
        :param uid:
        :param rawaddress:
        :param request_string: Search query string.
        :param kwargs:
        :return:
        """
        address = Address(rawaddress)
        root = factory.get_resource(uid, address)
        root.form_class = MixedForm
        root.set_request(self.request)
        search_results = factory.search(root, request_string, **kwargs)
        count_list_results = kwargs.get('count_lost_results', False)
        return self._process_search_result(search_results, root, load_source_ids, count_list_results)

    def geo_search(self, uid, latitude, longitude, start_date, end_date, distance, count_lost_results):
        """
        Search by date and coordinates.
        """
        address = Address('/disk', uid=uid)
        root = factory.get_resource(uid, address)
        root.form_class = MixedForm
        root.set_request(self.request)
        search_results = factory.geo_search(uid, start_date, end_date, latitude, longitude, distance,
                                            count_lost_results)
        return self._process_search_result(search_results, root, False, count_lost_results)

    def _process_search_result(self, search_results, root, load_source_ids, count_lost_results):
        lost_results_count = 0
        # prepare search results to MPFS
        for item in search_results:
            if not item:
                lost_results_count += 1
            else:
                res = item.resource
                # add search_scope to meta because we have no other convenient method to return search scopes
                res.meta['search_scope'] = item.scope
                root.children_list.append(res)
                append_meta_to_office_files(res, self.request)
        resources_list = [r.resource for r in search_results if r is not None]
        if load_source_ids:
            GlobalGalleryController.load_and_set_source_ids_for_resources(resources_list)
        result = {
            'this': root,
            'search-query': search_results.query or '',
            # FIXME: if we comment out next line nothing will change, because 'list' never called
            'list': resources_list,
        }
        if count_lost_results:
            result['lost_results_count'] = lost_results_count
        return result

    def get_trash_address(self, rawaddress):
        address = Address(rawaddress)
        # вседа меняем имя файла в корзине во избежание коллизий
        address.add_trash_suffix()
        address.change_storage('trash')
        address.drop_path_to_root()
        return address

    def trash_append(self, uid, rawaddress, version=None, lock_data=None, tgt_address=None):
        if tgt_address is None:
            tgt_address = self.get_trash_address(rawaddress)

        resource = factory.get_resource(uid, Address(rawaddress))
        resource.remove_public()
        self.lock_helper.lock(resource, lock_data, operation=self.method)
        self.lock_helper.lock_by_address(tgt_address, lock_data, operation=self.method)

        try:
            result = self.move_resource(
                uid,
                rawaddress,
                tgt_address.id,
                force=True,
                copy_symlinks=True,
                rm_symlinks=False,
                lock=False,
            )
        except Exception:
            self.lock_helper.unlock(resource)
            self.lock_helper.unlock(tgt_address.id)
            raise
        else:
            self.lock_helper.unlock(resource)
            self.lock_helper.unlock(tgt_address.id)

            # Замена имени элемента, под которым значение хранится в trash,
            # на реальное имя, под которым ресурс хранился в user_data
            for rec in self.push_content:
                if rec['action_name'] == 'diff':
                    for each in rec['xiva_data']:
                        if each['op'] == 'new' and each['key'] == tgt_address.path:
                            each['name'] = Address(rawaddress).name
                            break
            return result

    def get_trash_restore_tgt_address(self, uid, rawaddress, new_name=None, new_path=None, force=False):
        source = trash_service.Trash().get_resource(uid, Address(rawaddress))
        src_address = source.address
        tgt_address = Address.Make(src_address.uid, source.original_id)

        if new_name:
            tgt_address = tgt_address.rename(new_name)
        if new_path:
            tgt_address.change_parent(Address.Make(src_address.uid, new_path))

        if not force:
            tgt_address = self.autosuffix_address(tgt_address)
        return tgt_address

    def trash_restore(self, uid, rawaddress, new_name=None, new_path=None, force=False, version=None, lock_data=None, tgt_address=None):
        source = trash_service.Trash().get_resource(uid, Address(rawaddress))
        src_address = source.address

        if tgt_address is None:
            tgt_address = self.get_trash_restore_tgt_address(uid, rawaddress, new_name=new_name, new_path=new_path, force=force)

        _parent_structure = []
        prnt = {}

        def _check_parents(child_address):
            result = None
            parent = child_address.get_parent()
            try:
                parent_resource = factory.get_resource(uid, parent)
            except Exception:
                result = _check_parents(parent) or parent
                self.mkdir(uid, parent.id, notify=True)
                parent_resource = factory.get_resource(uid, parent)
                data = {
                    'op': 'new',
                    'key': parent.path,
                    'fid': parent_resource.meta['file_id'],
                }
                _parent_structure.append(data)
            prnt['p'] = parent_resource
            return result

        tgt_service = factory.get_service(tgt_address)
        old_version = tgt_service.get_version(tgt_address.uid)
        result = _check_parents(tgt_address)
        if result:
            restored_parent = result.get_parent()

            # Interface needs description of the highest restored folder`s parent
            if self.request:
                self.request.form = factory.get_resource(uid, restored_parent).form
        else:
            result = tgt_address
        copy_symlinks = True
        exception_keyset = ()
        if new_name:
            parent_resource = prnt['p']
            if parent_resource.is_shared or parent_resource.is_group:
                original_resource = factory.get_resource(uid, Address.Make(src_address.uid, source.original_id))
            else:
                original_resource = None
            if original_resource:
                symlink = source.meta.get('symlink')
                original_symlink = original_resource.meta.get('symlink')
                if symlink and symlink == original_symlink:
                    exception_keyset = ('published',)
                    copy_symlinks = False
        photounlim_allowed = new_path is None
        self.move_resource(uid, src_address.id, tgt_address.id, force=force,
                           copy_symlinks=copy_symlinks, exception_keyset=exception_keyset, lock_data=lock_data, photounlim_allowed=photounlim_allowed)
        if copy_symlinks:
            resource = factory.get_resource(uid, result)
            resource.restore_public()
        return result.path

    def trash_restore_all(self, uid):
        trash_folder = factory.get_resource(uid, '/trash')
        listing = trash_folder.list()
        new_path = Address.Make(uid, '/disk/trash %s' % time.strftime("%Y.%m.%d %H:%M:%S"))
        self.mkdir(uid, new_path.id)
        for item in listing['list']:
            item_addr = Address.Make(uid, item['id'])
            self.trash_restore(uid, item_addr.id, new_path=new_path.path)

    def trash_append_file(self, uid, rawaddress, version=None):
        """
        Не используется
        """
        address = Address(rawaddress)
        trash_address = Address.Make(address.uid, '/trash/')
        target_address = trash_address.get_child_file(rawaddress)
        self.move_resource(uid, rawaddress, target_address.id, True, copy_symlinks=False)

    def trash_drop_all(self, uid, owner, oid=None, lock_data=None):
        address = Address.Make(owner, '/trash')
        trash_res = factory.get_resource(owner, address)

        for trash_subfolder in trash_res.list()['list']:
            if trash_subfolder.get('type') != 'dir':
                continue
            trash_subfolder_address = Address.Make(owner, trash_subfolder['id'])
            self.trash_drop_element(uid, trash_subfolder_address.id, lock_data)

        self.set_trash_lock(uid, oid, lock_data)

        xiva_job = {
            'xiva': {
                'uid': address.uid,
                'xiva_data': [{'op': 'clean', 'key': '/trash'}, ],
                'old_version': '',
                'new_version': '1',
                'action_name': 'diff',
                'class': DIFF,
            }
        }

        # Инициализируем trash_res еще раз. Необходимо для получения full_tree без каталогов, которые удалили ранее.
        trash_res = factory.get_resource(owner, address)
        service = factory.get_service(address)
        full_index = trash_res.get_full_index()
        del full_index['/trash']
        self.disk_indexer.push_tree(
            owner,
            full_index.values(),
            'delete',
            version=trash_res._service.get_version(owner),
            operation=self.method
        )

        try:
            self.tree_to_post_remove(
                {
                    'this': {'type': 'dir', 'uid': owner},
                    'list': trash_res.tree_index()
                },
                trash_res._service,
                extra_job=xiva_job
            )
            self.counter_helper.commit()
            if service.used(uid) != 0:
                service.update_counter(uid, -1 * service.used(uid))
                self.ask_space_counters_recount(uid)
        except Exception:
            self.ask_space_counters_recount(uid)
            # Логируем ошибку на случай исключения в finally-блоке
            error_log.exception('Failed to clear trash')
            raise
        else:
            TrashCleanerQueueDAO().remove({'uid': uid})

            if self.method == 'trash_drop_all':
                fs_events.FilesystemTrashDropAllEvent(uid=str(uid), owner=owner).send()
        finally:
            try:
                self.unset_trash_lock(uid)
            except SetAutocommitError:
                # Корневую ошибку мы должны были пофиксить - нужно попытаться переснять лок
                self.unset_trash_lock(uid)

        async_remove_versions_by_full_index(full_index)

    def ask_space_counters_recount(self, uid):
        recount.add(uid)

    def trash_drop_element(self, uid, path, lock_data=None):
        if not path.split(':', 1)[-1].startswith('/trash'):
            raise errors.StorageBadRootPath()

        address = Address(path)
        resource = factory.get_resource(uid, address)

        parent_folder = factory.get_resource(uid, address.get_parent())
        resource.set_parent(parent_folder)

        if resource.type == 'dir':
            dir_full_index = resource.get_full_index()

        self.rm(uid, path, lock_data=lock_data)

        if self.method in ('trash_drop_element', 'clean_trash'):
            auto = self.method == 'clean_trash'
            fs_events.FilesystemTrashDropEvent(
                    uid=str(uid),
                    tgt_resource=resource, tgt_address=address,
                    auto=auto).send()

        # Пушим нотифай
        data = {
            'op': 'deleted',
            'key': resource.address.path,
            'fid': resource.meta['file_id'],
            'resource_type': resource.type,
        }
        if resource.type == 'file':
            data['md5'] = resource.meta['md5']
            data['sha256'] = resource.meta['sha256']
            data['size'] = resource.size
        old_version = resource.version or ''
        new_version = parent_folder.version
        self.xiva_helper.add_to_xiva(address.uid, data, old_version, new_version)

        # Пушим индексатор
        if resource.type == 'dir':
            self.disk_indexer.push_tree(
                uid,
                dir_full_index.values(),
                'delete',
                version=resource._service.get_version(uid),
                operation=self.method
            )
        else:
            self.disk_indexer.push(resource, 'delete', operation=self.method)

    def tree_to_post_remove(self, resource, service, symlinks=False, extra_job={}, level=0,
                            remove_top_level_folder=True, set_yarovaya_mark_for_subresources=False):
        """
            create job for each folder on each level in the tree
            remove this folder
        """
        all_files = []
        symlinks_list = []

        if FEATURE_TOGGLES_REMOVE_FOLDER_FIRST:
            self._remove_resource_if_folder(resource, service, True)

        if FEATURE_TOGGLES_SETTING_YAROVAYA_MARK_ENABLED:
            set_yarovaya_mark_for_subresources |= self._is_public_or_shared_or_has_yarovaya_mark_by_resource_dict(
                resource['this'])
            if set_yarovaya_mark_for_subresources:
                resource['this']['data']['meta']['yarovaya_mark'] = set_yarovaya_mark_for_subresources

        if resource['this']['type'] == 'dir':
            for element in resource['list']:
                if element['this']['type'] == 'dir':
                    self.tree_to_post_remove(element, service, symlinks=symlinks, level=level + 1,
                                             set_yarovaya_mark_for_subresources=set_yarovaya_mark_for_subresources)
                else:
                    file_data = {
                        'id': element['this']['key'],
                        'uid': element['this']['uid'],
                        'version': element['this'].get('version'),
                    }
                    if (FEATURE_TOGGLES_SETTING_YAROVAYA_MARK_ENABLED
                            and self._is_public_or_shared_or_has_yarovaya_mark_by_resource_dict(element)):
                        file_data['data'] = {'meta': {'yarovaya_mark': True}}
                    all_files.append(file_data)
                    if symlinks:
                        this_symlink = element['this']['data']['meta'].get('symlink')
                        if this_symlink:
                            symlinks_list.append({
                                'id': element['this']['key'],
                                'uid': element['this']['uid'],
                                'symlink': this_symlink
                            })
        else:
            all_files.append({
                'id': resource['this']['key'],
                'uid': resource['this']['uid'],
                'version': resource['this'].get('version'),
                'data': resource['this']['data'],
            })
        if symlinks:
            this_symlink = resource['this']['data']['meta'].get('symlink')
            if this_symlink:
                symlinks_list.append({
                    'id': resource['this']['key'],
                    'uid': resource['this']['uid'],
                    'symlink': this_symlink,
                })
        if settings.feature_toggles['sync_tree_removing']:
            if all_files:
                self.save_deleted_files(resource['this']['uid'], all_files,
                                        force_having_yarovaya_mark=set_yarovaya_mark_for_subresources)
            if not FEATURE_TOGGLES_REMOVE_FOLDER_FIRST:
                self._remove_resource_if_folder(resource, service, remove_top_level_folder)

            if not level:
                for k, v in extra_job.iteritems():
                    self.job_content[k].append(v)
            if symlinks and symlinks_list:
                self._drop_symlinks_hierarchy(resource['this']['uid'], symlinks_list)
        else:
            if all_files:
                if self.async and not level:
                    self.save_deleted_files(resource['this']['uid'], all_files,
                                            force_having_yarovaya_mark=set_yarovaya_mark_for_subresources)
                    for k, v in extra_job.iteritems():
                        self.job_content[k].append(v)
                else:
                    job_data = {
                        'uid': resource['this']['uid'],
                        'removed': all_files,
                    }
                    if extra_job:
                        job_data['extra'] = extra_job
                    self.job_content['post_remove'].append(job_data)
            elif self.async and not level:
                for k, v in extra_job.iteritems():
                    self.job_content[k].append(v)
            if not FEATURE_TOGGLES_REMOVE_FOLDER_FIRST:
                self._remove_resource_if_folder(resource, service, remove_top_level_folder)

            # Ставим команды на удаление симлинков
            if symlinks and symlinks_list:
                self.job_content['symlinks_remove'].append(dict(
                    uid=resource['this']['uid'],
                    symlinks=symlinks_list
                ))

    @staticmethod
    def _remove_resource_if_folder(resource, service, remove_top_level_folder):
        if resource['this']['type'] == 'dir' and remove_top_level_folder:
            try:
                resource_key = resource['this']['key']
            except Exception:
                pass
            else:
                try:
                    service.remove_single(
                        (resource['this']['uid'], resource_key),
                        resource['this'].get('version'), changelog=False
                    )
                except errors.StorageKeyNotFound:
                    pass

    def inspect(self, uid, rawaddress):
        address = Address(rawaddress)
        return self.quota.inspect(address.uid)

    def restore_deleted(self, uid, rawaddress, new_parent=None, force=True, version=None):
        src_address = Address(rawaddress)
        resource = factory.get_resource(uid, src_address)

        if resource.is_folder:
            tgt_address = Address(rawaddress)
            tgt_address.change_parent(new_parent)
        else:
            re_result = re.match('^(.*?)(:(\d+(\.\d+)?))?$', src_address.path)
            tgt_path = re_result.group(1)
            dstamp = re_result.group(3) or ctimestamp()
            tgt_address = Address('%s:%s' % (src_address.uid, tgt_path))
            tgt_address.change_storage('disk')

        if resource.is_file:
            if new_parent:
                tgt_address.change_parent(new_parent)
            try:
                i = -1
                # Ищем пустое имя, если нашли, то выбросится ResourceNotFound
                while True:
                    factory.get_resource(uid, tgt_address)
                    postfix = ' (%s)' % time.ctime(float(dstamp))
                    if i > 0:
                        postfix = postfix + ' %s' % i
                    tgt_address = Address('%s:%s' % (src_address.uid, tgt_path + postfix))
                    tgt_address.change_storage('disk')
                    if new_parent:
                        tgt_address.change_parent(new_parent)
                    i += 1
            except errors.ResourceNotFound:
                pass
        _parent_structure = []

        def _check_parents(child_address):
            result = None
            parent_address = child_address.get_parent()
            try:
                factory.get_resource(uid, parent_address)
            except Exception:
                result = parent_address
                _check_parents(parent_address)
                parent_resource = self.mkdir(uid, parent_address.id)
                data = {
                    'op': 'new',
                    'key': parent_address.path,
                    'fid': parent_resource['meta']['file_id'],
                    'resource_type': 'dir',
                }
                _parent_structure.append(data)
            return result

        tgt_service = factory.get_service(tgt_address)
        old_version = tgt_service.get_version(uid)
        result = _check_parents(tgt_address)
        if result:
            result = result.get_parent().path

            # Пушим данные в xiva
            self.xiva_helper.add_to_xiva(
                tgt_address.uid,
                _parent_structure,
                old_version,
                tgt_service.get_version(tgt_address.uid)
            )
        self.move_resource(uid, src_address.id, tgt_address.id, True, copy_symlinks=False, rm_symlinks=True, parent_check=False)

        # убираем публичные данные, если есть
        resource = factory.get_resource(uid, tgt_address)
        if resource.is_file and resource.is_smth_public():
            resource.make_private()
            try:
                symlink = Symlink.find(resource.address)
            except errors.SymlinkNotFound:
                pass
            else:
                symlink.update_office_fields({'office_access_state': OfficeAccessStateConst.DISABLED})
            resource.remove_symlink()

        return resource.get_full_index()

    def save_deleted_files(self, uid, removed_files, force_having_yarovaya_mark=False):
        """
        Сохранение данных в hidden.
        """

        def restore_parents(address, restored=set()):
            if not isinstance(address, Address):
                address = Address(address)
            parent_address = address.get_parent()
            if parent_address.id not in restored:
                try:
                    factory.get_resource(uid, parent_address)
                except errors.ResourceNotFound:
                    restored.update(restore_parents(parent_address, restored))
                    try:
                        data = {}
                        if FEATURE_TOGGLES_SETTING_YAROVAYA_MARK_ENABLED and force_having_yarovaya_mark:
                            data = {'meta': {'yarovaya_mark': True}}
                        self.mkdir(uid, parent_address.id, notify_search=False, skip_check_rights=True, data=data)
                    except errors.MkdirFolderAlreadyExist:
                        pass
                    restored.add(parent_address.id)
            return restored

        cur_time = time.time()

        def move_to_hidden(element):
            address = Address.Make(element['uid'], element['id'])
            original_address = deepcopy(address)
            element_service = factory.get_service(address)
            element_data = element.get('data') or element_service.show_single(address, element.get('version', None))['data']
            if FEATURE_TOGGLES_SETTING_YAROVAYA_MARK_ENABLED and force_having_yarovaya_mark:
                element_data['meta']['yarovaya_mark'] = True

            notify_storage_clean_check_resource(element_data['meta'])
            address.change_storage('hidden')
            new_id = address.path + ":" + str(cur_time)
            # В hidden file_id должен быть новый
            # (чтобы не находить ресурс по file_id, который был сгенерирован для ресурса в основных разделах с данными)
            element_data['meta']['file_id'] = Resource.generate_file_id(address.uid, new_id)
            #TODO(kis8ya): убрать в список с понятным именем (список_удаляемых_полей)
            element_data['meta'].pop('office_doc_short_id')

            is_live_photo = element_data['meta'].pop('is_live_photo', None)
            hidden_data.put(address.uid, new_id, element_data)
            if is_live_photo:
                hidden_data_address = Address.Make(address.uid, new_id)
                resource = factory.get_resource(original_address.uid, original_address.path)
                LivePhotoFilesManager.copy_live_video_file(resource.address, hidden_data_address)

            if 'size' in element.get('data', {}):
                # этот код отрабатывает при удалении одного элемента из корзины (trash_drop), удаление самого элемента
                # из корзины происходит выше по стеку
                size = int(element_data['size'])
            else:
                # а эта ветка вызывается при чистке всей корзины (trash_drop_all), удаление элемента элемента из корзины
                # происходит здесь через remove_single чуть ниже
                if is_live_photo:
                    LivePhotoFilesManager.remove_live_video(original_address)

                element_service.remove_single(original_address, element['version'], changelog=False)
                size = int(element_data['size'])
            if settings.feature_toggles['tags']:
                element_service.tags_remove_single(uid, element_data['meta']['file_id'])
            if 'gid' in element_data:
                Counter().add_group(element_data['gid'], -size)
                # Если ресурс из ОП - счетчики обновляем Владельцу
                element_service.update_counter(element_data['owner_uid'], -size)
            else:
                element_service.update_counter(element['uid'], -size)

            if self.method in {'rm', 'mkfile', 'move_resource'} and element_data.get('owner_uid', uid) == uid:
                GlobalGalleryController.try_to_write_to_deletion_log_by_file_doc(uid, element_data)

        # Restore parent' path
        restored_folders = set()
        first_file = removed_files[0]
        first_file_address = Address.Make(first_file['uid'], first_file['id'])
        first_file_address.change_storage('hidden')
        restored_folders.update(restore_parents(first_file_address, restored_folders))

        # Remove children
        for f in removed_files:
            try:
                move_to_hidden(f)
            except errors.NotFound:
                pass
            except Exception, e:
                error_log.error(e)
                error_log.error(traceback.format_exc())
                raise

    @staticmethod
    def _get_collection_for_root_folder(root_folder):
        from mpfs.core.factory import get_collection_for_root_folder  # цикл. импорт
        return get_collection_for_root_folder(root_folder)

    def check_target(self, raw_target, uid=None, force=False):
        """
        Проверка "правильности" ресурса-цели.

        Логика:
            * Должена существовать папка-родитель
            * Если файл существует, то перезаписываем только с флагом "force"
            * На папки force не влияет
            * Незалочен
        """
        tgt_address = Address(raw_target)
        if uid is None:
            uid = tgt_address.uid
        try:
            target_parent = factory.get_resource(uid, tgt_address.get_parent())
        except errors.ResourceNotFound:
            raise errors.CopyParentNotFound()

        try:
            tgt_resource = factory.get_resource(uid, tgt_address)
        except errors.ResourceNotFound:
            pass
        else:
            if tgt_resource.is_folder:
                raise errors.FolderAlreadyExist()
            elif not force:
                raise errors.FileAlreadyExist()
        self.check_address(raw_target, strict=True)
        self.check_lock(raw_target)

    def check_source(self, raw_source, uid=None):
        """
        Проверка "правильности" ресурса-источника.

        Логика:
            * Должен существовать
            * Незалочен
        """
        src_address = Address(raw_source)
        if uid is None:
            uid = src_address.uid
        # Райзит errors.ResourceNotFound, если нет такого ресурса
        factory.get_resource(uid, src_address)
        self.check_address(raw_source, strict=True)
        self.check_lock(raw_source)

    def _filter_folder_data(self, src_address, data):
        # этот if добавлен для задачи https://st.yandex-team.ru/CHEMODAN-27018
        # тут небольшой хак - у папки /disk, которую мы копируем другому пользователю в диск, в мете есть
        # странный пустой словарь status_dict, из-за него потом падает ручка list на родительской папке, у
        # обычной папки такого аттрибута нет и если его убрать из меты, то все ок
        if src_address.path == '/disk' and 'status_dict' in data:
            del data['status_dict']

    def base_copy(self, uid, source, target, force, check_hids_blockings=True, set_target_lock=True, **kw):
        """Копирование ресурса.

        :param set_target_lock: Выславляем в False, если хотим управлять локом цели на более высоком уровне
        """
        mv = kw.get('mv', False)  # этот флаг определяет, будет ли у ресурса фильтроваться file_id (false=будет)
        copy_symlinks = kw.get('copy_symlinks', True)
        rm_symlinks = kw.get('rm_symlinks', False)
        copy_infected_files = kw.get('copy_infected_files', True)
        resource_data = kw.get('resource_data', {})
        exception_keyset = kw.get('exception_keyset', ()) + ('folder_type', )
        src_uid = kw.get('src_uid')
        _changes = []
        src_address = Address(source)
        tgt_address = src_address.clone(target)
        current_timestamp = int(time.time())
        predefined_file_id = kw.get('predefined_file_id')
        notify_search = kw.get('notify_search', True)
        force_having_yarovaya_mark = kw.get('force_having_yarovaya_mark', False)
        trash_append_to_owner_by_guest = kw.get('trash_append_to_owner_by_guest', False)

        try:
            target_parent = factory.get_resource(uid, tgt_address.get_parent())
        except errors.ResourceNotFound:
            raise errors.CopyParentNotFound()
        else:
            tgt_service = factory.get_service(tgt_address)

            try:
                resource_uid = src_uid or uid
                src_resource = factory.get_resource(resource_uid, src_address)
            except errors.ResourceNotFound:
                raise errors.CopyNotFound()
            else:

                try:
                    tgt_resource = factory.get_resource(uid, tgt_address)
                except errors.ResourceNotFound:
                    tgt_resource = None
                else:
                    if FEATURE_TOGGLES_SETTING_YAROVAYA_MARK_ENABLED:
                        force_having_yarovaya_mark |= tgt_resource.should_yarovaya_mark_be_set()
                    if not force:
                        raise errors.CopyTargetExists()

                overwritten = tgt_resource is not None

                # Используется и для логгирования событий о удалении каждой подпапки
                # в event_history/event_subscriptions.py#_log_fs_delete_subdirs_if_needed
                src_resource.get_full_index(root=False)

                if src_resource.is_file:
                    src_hids = [src_resource.hid, ]
                else:
                    src_hids = [v.get('hid') for v in src_resource.full_index_map.itervalues() if
                                v.get('type') == 'file']
                if check_hids_blockings:
                    self.check_hids_blockings(src_hids)

                src_size = int(src_resource.get_size())
                free_space = int(tgt_service.free(tgt_address.uid))
                mrg = merge(src_resource, tgt_address, tgt_resource)
                df_size = mrg.diff_size()

                if src_address.storage_name == PHOTOUNLIM_AREA and tgt_address.storage_name == DISK_AREA:
                    group = None
                    owner_tgt_address = tgt_address
                    operation_type = 'move' if mv else 'copy'
                    if target_parent.is_shared:
                        group = target_parent.group
                        owner_tgt_address = group.get_group_address(tgt_address)
                    self.check_available_space(address=owner_tgt_address, required_space=src_size, group=group,
                                               operation_type=operation_type, operation_user_uid=uid)

                if not mv and src_size > free_space:
                    if (not FEATURE_TOGGLES_CORRECT_SPACE_CHECKS_FOR_SHARED_FOLDERS or
                            not (target_parent.is_shared and ignores_shared_folders_space(uid))):
                        if tgt_resource:
                            if free_space <= df_size or df_size > 0:
                                raise errors.NoFreeSpaceCopyToDisk(
                                    "free: %s expected diff size: %s" % (free_space, df_size))
                        else:
                            raise errors.NoFreeSpaceCopyToDisk()

                src_resource.prepare_transfer()

                # подготавливаем функцию принимающую src файл и dst файл
                versioninig_manupulate = lambda s, d: None
                if mv and src_resource.address.storage_name in ('disk', 'trash'):
                    if target_parent.address.storage_name == 'hidden':
                        # move (disk|trash) -> hidden
                        versioninig_manupulate = lambda _, d: ResourceVersionManager.async_remove_versions(d.resource_id)
                    elif (src_resource.address.storage_name == 'disk' and
                            target_parent.address.storage_name == 'trash'):
                        # move disk -> trash
                        if src_resource.type == 'file' and not src_resource.is_shared:
                            versioninig_manupulate = lambda s, d: ResourceVersionManager.trash_append(uid, s, d)
                    elif (src_resource.address.storage_name == 'trash' and
                            target_parent.address.storage_name == 'disk'):
                        # move trash -> disk
                        if src_resource.type == 'file':
                            versioninig_manupulate = lambda _, d: ResourceVersionManager.restore_from_trash(uid, d)
                    elif src_resource.address.storage_name == target_parent.address.storage_name == 'disk':
                        # move disk -> disk
                        is_owner_changing = src_resource.owner_uid != target_parent.owner_uid
                        if is_owner_changing:
                            versioninig_manupulate = lambda s, d: ResourceVersionManager.move_versions(s.resource_id, d.resource_id)

                # File
                if isinstance(src_resource, base_resources.File):
                    if not (src_resource.is_infected() and not copy_infected_files):
                        mkfile_args = {
                            'data': src_resource.dict(safe=True, unique_fields=mv, exception_keyset=exception_keyset),
                            'copy_symlinks': copy_symlinks,
                            'rm_symlinks': rm_symlinks,
                            'notify': False,
                            'original_resource': src_resource,
                            'notify_search': notify_search,
                        }
                        if mv:
                            mkfile_args['notify_search_type'] = 'update'
                        mkfile_args['data']['meta'].update(resource_data)
                        mkfile_args['data']['mtime'] = current_timestamp
                        mkfile_args['data']['meta']['modify_uid'] = uid
                        if predefined_file_id is not None and not mv:
                            mkfile_args['data']['meta']['file_id'] = predefined_file_id

                        is_live_photo_file = mkfile_args['data']['meta'].pop('is_live_photo', None)
                        mkfile_args['data']['meta'].pop('live_video_id', None)  # слинкуем позже
                        if self.method == 'copy_resource':
                            mkfile_args['data']['meta']['yarovaya_mark'] = False
                        elif FEATURE_TOGGLES_SETTING_YAROVAYA_MARK_ENABLED and force_having_yarovaya_mark:
                            mkfile_args['data']['meta']['yarovaya_mark'] = True

                        created_file = self.mkfile(uid, tgt_address.id, **mkfile_args)

                        if is_live_photo_file:
                            real_tgt_address = target_parent.address.get_child_file(tgt_address.name)
                            additional_exception_keyset = ('file_id',) if not mv else tuple()
                            LivePhotoFilesManager.copy_live_video_file(
                                src_resource.address,
                                real_tgt_address,
                                exception_keyset + additional_exception_keyset,
                            )

                        if (tgt_service.name == TRASH_AREA and not src_resource.is_shared
                                and not trash_append_to_owner_by_guest):
                            GlobalGalleryController.try_to_write_to_deletion_log_by_file_doc(uid, created_file.dict(), force_live_photo_flag=is_live_photo_file)

                        versioninig_manupulate(src_resource, created_file)
                        xiva_data = {
                            'op': 'new',
                            'key': self.mkfile_new_element.id,
                            'md5': self.mkfile_new_element.meta['md5'],
                            'sha256': self.mkfile_new_element.meta['sha256'],
                            'size': self.mkfile_new_element.size,
                            'fid': self.mkfile_new_element.meta['file_id'],
                            'resource_type': 'file',
                        }
                        _changes.append(xiva_data)

                # Folder
                elif isinstance(src_resource, base_resources.Folder):
                    xiva_data = {
                        'op': 'new',
                        'resource_type': 'dir',
                    }
                    try:
                        data = src_resource.dict(safe=True, unique_fields=mv, exception_keyset=exception_keyset)
                        self._filter_folder_data(src_resource.address, data)

                        data['meta'].update(resource_data)
                        data['mtime'] = current_timestamp
                        data['meta']['modify_uid'] = uid
                        if self.method == 'copy_resource':
                            data['meta']['yarovaya_mark'] = False
                        elif FEATURE_TOGGLES_SETTING_YAROVAYA_MARK_ENABLED and force_having_yarovaya_mark:
                            data['meta']['yarovaya_mark'] = True
                        if predefined_file_id is not None and not mv:
                            data['meta']['file_id'] = predefined_file_id

                        self.mkdir(
                            uid,
                            tgt_address.id,
                            data=data,
                            notify=False,
                            original_resource=src_resource,
                            copy_symlinks=copy_symlinks,
                            rm_symlinks=rm_symlinks,
                            keep_lock=not set_target_lock,
                        )
                    except errors.MkdirFolderAlreadyExist:
                        if (FEATURE_TOGGLES_SETTING_YAROVAYA_MARK_ENABLED
                                and (self.mkdir_existing_element.should_yarovaya_mark_be_set()
                                     or data.get('meta', {}).get('yarovaya_mark', False))):
                            self.mkdir_existing_element.set_yarovaya_mark()
                        xiva_data['key'] = tgt_address.path
                        xiva_data['fid'] = self.mkdir_existing_element.meta['file_id']
                    else:
                        xiva_data['key'] = self.mkdir_new_element.id
                        xiva_data['fid'] = self.mkdir_new_element.meta['file_id']
                    if tgt_service.name != 'trash':
                        if src_resource.is_group_root:
                            xiva_data['is_group_root'] = 1
                            xiva_data['shared_rights'] = 'owner'
                        elif src_resource.is_shared_root:
                            xiva_data['shared_rights'] = str(src_resource.link.rights)
                            xiva_data['is_group_root'] = 1
                        elif src_resource.with_shared:
                            xiva_data['with_shared'] = 1
                    _changes.append(xiva_data)
                    removed_children = set()
                    tgt_resource = factory.get_resource(uid, tgt_address)
                    if set_target_lock:
                        self.lock_helper.lock(tgt_resource, operation=self.method)

                    try:
                        for val in sorted(mrg.src_index, key=lambda x: x['id']):
                            if val['id'] in removed_children:
                                if tgt_service.name != 'trash':
                                    _xiva_data = {
                                        'op': 'new',
                                        'key': val['new_path'].path,
                                        'fid': val['meta']['file_id'],
                                    }
                                    if val['type'] == 'file':
                                        _xiva_data['md5'] = val['meta']['md5']
                                        _xiva_data['sha256'] = val['meta']['sha256']
                                        _xiva_data['size'] = val['size']
                                        _xiva_data['resource_type'] = 'file'
                                    else:
                                        _xiva_data['resource_type'] = 'dir'
                                    _changes.append(_xiva_data)

                                continue

                            src_child_address = Address.Make(val['uid'], val['id'])
                            resource_uid = src_uid or uid
                            src_child = factory.get_resource(resource_uid, src_child_address.id)
                            should_child_src_have_yarovaya_mark = (FEATURE_TOGGLES_SETTING_YAROVAYA_MARK_ENABLED
                                                                   and src_child.is_public_or_shared())

                            if src_child.is_infected() and not copy_infected_files:
                                continue

                            src_child.prepare_transfer()
                            try:
                                dst_child = factory.get_resource(uid, val['new_path'])
                            except errors.ResourceNotFound:
                                if isinstance(src_child, base_resources.File):
                                    src_child_data = src_child.dict(safe=True, unique_fields=mv,
                                                               exception_keyset=exception_keyset)
                                    if self.method == 'copy_resource':
                                        src_child_data['meta']['yarovaya_mark'] = False
                                    elif (FEATURE_TOGGLES_SETTING_YAROVAYA_MARK_ENABLED
                                            and (force_having_yarovaya_mark or should_child_src_have_yarovaya_mark)):
                                        src_child_data['meta']['yarovaya_mark'] = True
                                    mkfile_args = {
                                        'data': src_child_data,
                                        'copy_symlinks': copy_symlinks,
                                        'rm_symlinks': rm_symlinks,
                                        'notify': False,
                                        'original_resource': src_child,
                                    }
                                    if mv:
                                        mkfile_args['notify_search_type'] = 'update'

                                    is_live_photo_file = mkfile_args['data']['meta'].pop('is_live_photo', None)
                                    mkfile_args['data']['meta'].pop('live_video_id', None)  # слинкуем позже

                                    created_file = self.mkfile(uid, val['new_path'].id, **mkfile_args)

                                    if is_live_photo_file:
                                        real_tgt_address = created_file.address  # это настоящий путь ресурса
                                        # для файла из ОП - это будет путь у владельца
                                        # (в отличие от val['new_path'].id - это путь у участника)

                                        additional_exception_keyset = ('file_id',) if not mv else tuple()
                                        LivePhotoFilesManager.copy_live_video_file(
                                            src_child.address,
                                            real_tgt_address,
                                            exception_keyset + additional_exception_keyset,
                                        )

                                    if (tgt_service.name == TRASH_AREA and not src_resource.is_shared
                                            and not trash_append_to_owner_by_guest):
                                        GlobalGalleryController.try_to_write_to_deletion_log_by_file_doc(uid, created_file.dict(), force_live_photo_flag=is_live_photo_file)

                                    versioninig_manupulate(src_child, created_file)
                                    _changes.append({
                                        'op': 'new',
                                        'key': self.mkfile_new_element.id,
                                        'md5': self.mkfile_new_element.meta['md5'],
                                        'sha256': self.mkfile_new_element.meta['sha256'],
                                        'size': self.mkfile_new_element.size,
                                        'fid': self.mkfile_new_element.meta['file_id'],
                                        'resource_type': 'file',
                                    })

                                elif isinstance(src_child, base_resources.Folder):
                                    if (src_child.is_shared_root or src_child.is_group_root) and tgt_service.name != 'trash' and mv:
                                        src_child.rename(val['new_path'])
                                        _xiva_data = {
                                            'op': 'new',
                                            'key': val['new_path'].path,
                                            'is_group_root': 1,
                                            'fid': src_child.meta['file_id'],
                                            'resource_type': 'dir',
                                        }
                                        if src_child.is_group_root:
                                            _xiva_data['shared_rights'] = 'owner'
                                        elif src_child.is_shared_root:
                                            _xiva_data['shared_rights'] = str(src_child.link.rights)
                                        _changes.append(_xiva_data)
                                        removed_children.update(ifilter(
                                            lambda _id: _id not in removed_children and is_subpath(_id, src_child.id),
                                            imap(lambda val: val['id'], mrg.src_index)))
                                        self.disk_indexer.push(src_child, 'delete', operation=self.method)
                                        for item in ifilter(
                                            lambda val: val['id'] not in removed_children and is_subpath(val['id'], src_child.id),
                                                mrg.src_index
                                        ):
                                            # FIXME: Зачем мы это делаем N раз в цикле?
                                            self.disk_indexer.push(src_child, 'delete', operation=self.method)
                                            removed_children.add(item)
                                    elif src_child.is_shared_root and tgt_service.name == 'trash' and mv:
                                        src_child.rm()
                                        removed_children.update(ifilter(
                                            lambda _id: _id not in removed_children and is_subpath(_id, src_child.id),
                                            imap(lambda val: val['id'], mrg.src_index)))
                                    elif src_child.is_shared and tgt_service.name == 'trash' and mv:
                                        if src_resource.is_shared and not src_resource.is_shared_root:
                                            src_child_data = src_child.dict(safe=True, unique_fields=mv,
                                                                            exception_keyset=exception_keyset)
                                            if self.method == 'copy_resource':
                                                src_child_data['meta']['yarovaya_mark'] = False
                                            elif (FEATURE_TOGGLES_SETTING_YAROVAYA_MARK_ENABLED
                                                    and (force_having_yarovaya_mark or should_child_src_have_yarovaya_mark)):
                                                src_child_data['meta']['yarovaya_mark'] = True
                                            self.mkdir(
                                                uid,
                                                val['new_path'].id,
                                                data=src_child_data,
                                                notify=False,
                                                original_resource=src_child,
                                                copy_symlinks=copy_symlinks,
                                                rm_symlinks=rm_symlinks,
                                            )
                                            _xiva_data = {
                                                'op': 'new',
                                                'key': self.mkdir_new_element.id,
                                                'fid': self.mkdir_new_element.meta['file_id'],
                                            }
                                        else:
                                            _xiva_data = {
                                                'op': 'new',
                                                'key': val['new_path'].path,
                                                'fid': val['meta']['file_id'],
                                            }
                                        _xiva_data['resource_type'] = 'dir'
                                        _changes.append(_xiva_data)
                                    elif src_child.with_shared and mv and tgt_service.name != 'trash':
                                        _xiva_data = {
                                            'op': 'new',
                                            'key': val['new_path'].path,
                                            'with_shared': 1,
                                            'fid': val['meta']['file_id'],
                                            'resource_type': 'dir',
                                        }
                                        _changes.append(_xiva_data)
                                        src_child_data = src_child.dict(safe=True, unique_fields=mv, exception_keyset=exception_keyset)
                                        if self.method == 'copy_resource':
                                            src_child_data['meta']['yarovaya_mark'] = False
                                        elif (FEATURE_TOGGLES_SETTING_YAROVAYA_MARK_ENABLED
                                                and force_having_yarovaya_mark or should_child_src_have_yarovaya_mark):
                                            src_child_data['meta']['yarovaya_mark'] = True
                                        self.mkdir(
                                            uid,
                                            val['new_path'].id,
                                            data=src_child_data,
                                            notify=False,
                                            original_resource=src_child,
                                            copy_symlinks=copy_symlinks,
                                            rm_symlinks=rm_symlinks
                                        )
                                    else:
                                        if src_child.is_shared and src_resource.is_shared_root and tgt_service.name != 'trash' and mv:
                                            _xiva_data = {
                                                'op': 'new',
                                                'key': val['new_path'].path,
                                                'fid': val['meta']['file_id'],
                                            }
                                        else:
                                            src_child_data = src_child.dict(safe=True, unique_fields=mv, exception_keyset=exception_keyset)
                                            if self.method == 'copy_resource':
                                                src_child_data['meta']['yarovaya_mark'] = False
                                            elif (FEATURE_TOGGLES_SETTING_YAROVAYA_MARK_ENABLED
                                                    and (force_having_yarovaya_mark or should_child_src_have_yarovaya_mark)):
                                                src_child_data['meta']['yarovaya_mark'] = True
                                            self.mkdir(
                                                uid,
                                                val['new_path'].id,
                                                data=src_child_data,
                                                notify=False,
                                                original_resource=src_child,
                                                copy_symlinks=copy_symlinks,
                                                rm_symlinks=rm_symlinks
                                            )
                                            _xiva_data = {
                                                'op': 'new',
                                                'key': self.mkdir_new_element.id,
                                                'fid': self.mkdir_new_element.meta['file_id'],
                                            }
                                        _xiva_data['resource_type'] = 'dir'
                                        _changes.append(_xiva_data)
                            else:
                                if isinstance(dst_child, base_resources.File):
                                    data = src_child.dict(safe=True, unique_fields=mv, exception_keyset=exception_keyset)
                                    data['mtime'] = current_timestamp
                                    if self.method == 'copy_resource':
                                        data['meta']['yarovaya_mark'] = False
                                    elif (FEATURE_TOGGLES_SETTING_YAROVAYA_MARK_ENABLED
                                            and force_having_yarovaya_mark or should_child_src_have_yarovaya_mark):
                                        data['meta']['yarovaya_mark'] = True
                                    created_file = self.mkfile(
                                        uid,
                                        val['new_path'].id,
                                        data=data,
                                        copy_symlinks=copy_symlinks,
                                        rm_symlinks=rm_symlinks,
                                        notify=False,
                                        original_resource=src_child,
                                    )
                                    if (tgt_service.name == TRASH_AREA and not src_resource.is_shared
                                            and not trash_append_to_owner_by_guest):
                                        GlobalGalleryController.try_to_write_to_deletion_log_by_file_doc(uid, created_file.dict())
                                    _changes.append({
                                        'op': 'changed',
                                        'key': self.mkfile_new_element.id,
                                        'md5': self.mkfile_new_element.meta['md5'],
                                        'sha256': self.mkfile_new_element.meta['sha256'],
                                        'size': self.mkfile_new_element.size,
                                        'fid': self.mkfile_new_element.meta['file_id'],
                                        'resource_type': 'file',
                                    })
                                elif isinstance(dst_child, base_resources.Folder):
                                    dst_child.setprop({'mtime': current_timestamp}, [])
                            if self.method == 'trash_append':
                                try:
                                    resource = factory.get_resource(uid, val['new_path'])
                                    resource.remove_public()
                                except errors.ResourceNotFound:
                                    pass
                            elif self.method == 'trash_restore' and copy_symlinks and not rm_symlinks:
                                try:
                                    resource = factory.get_resource(uid, val['new_path'])
                                    resource.restore_public()
                                except errors.ResourceNotFound:
                                    pass

                    except Exception:
                        try:
                            resource = factory.get_resource(uid, tgt_address)
                            self.lock_helper.unlock(resource)
                        except Exception:
                            pass
                        raise

                else:
                    raise errors.MPFSError()
                dst_size = df_size or src_size
                resource = factory.get_resource(uid, tgt_address)
                self.lock_helper.unlock(resource)

                # Возвращаем src_resource чтобы в move_resource не выполнять get_full_index второй раз,
                # а использовать полученное тут значение. Это проще чем принимать его оттуда здесь
                return src_size, dst_size, _changes, tgt_service.get_version(uid), overwritten, src_resource

    @staticmethod
    def _prepare_for_postgres(filter_file_id, src_resource, src_address, tgt_address):
        from mpfs.core.filesystem.resources.share import SharedResource  # cycle import

        tgt_uid = src_address.uid
        try:
            parent_path = tgt_address.get_parent()
            parent_folder = factory.get_resource(tgt_address.uid, parent_path)
            if isinstance(parent_folder, SharedResource):
                tgt_uid = parent_folder.owner_uid
        except errors.ResourceNotFound:
            pass

        if (filter_file_id and src_address.uid == tgt_address.uid and
                mpfs.engine.process.usrctl().is_collection_in_postgres(tgt_uid, 'user_data')):
            if 'file_id' in src_resource.meta:  # зачем тут проверка спросите вы? а вот кто-то зовет эту функцию для папки /disk и у нее нет file_id
                session = Session.create_from_uid(src_resource.address.uid)
                session.execute(SQL_SET_FILE_ID_TO_NULL_BY_PATH, {'uid': src_resource.address.uid,
                                                                  'path': src_resource.address.path})

    def get_downloads_address(self, uid, locale=None):
        return self.get_sysdir_address(uid, 'downloads', locale=locale)

    @classmethod
    def get_photostream_address(cls, uid, locale=None):
        return cls.get_sysdir_address(uid, 'photostream', locale=locale)

    def mkdir_photostream_root(self, uid, locale=None):
        return self.mksysdir(uid, type='photostream', locale=locale)

    @staticmethod
    def get_sysdir_address(uid, folder_type, locale=None):
        from mpfs.core.user.constants import DEFAULT_FOLDERS
        from mpfs.core.user.base import User
        if locale is None:
            locale = User(uid).get_supported_locale()
        folders = DEFAULT_FOLDERS[folder_type]
        if locale in folders:
            path = folders[locale]
        else:
            path = folders['ru']

        return Address.Make(uid, path)

    def chksysdir(self, uid, type, locale=None):
        address = self.get_sysdir_address(uid, type, locale)
        try:
            resource = factory.get_resource(uid, address.id)
        except errors.ResourceNotFound:
            return False
        else:
            return resource

    def mksysdir(self, uid, type, locale=None):
        folder = None
        resource = self.chksysdir(uid, type, locale=locale)
        address = self.get_sysdir_address(uid, type, locale=locale)

        try:
            if not resource:
                self.mkdir(uid, address.id, notify=True, sysdir=True)
                self._send_mksysdir_event(uid, type)
            else:
                if not resource.is_folder or resource.is_shared_root:
                    iter_name, sucess_moved = 1, False
                    while not sucess_moved:
                        try:
                            path = address.path
                            if iter_name:
                                path = '%s %s' % (path, iter_name)
                            dst = Address('%s:%s' % (uid, path))
                            sucess_moved = self.move_resource(
                                uid,
                                address.id,
                                dst.id,
                                force=False
                            )
                        except errors.CopyTargetExists:
                            iter_name += 1
                    self.mkdir(uid, address.id, notify=True, sysdir=True)
                    self._send_mksysdir_event(uid, type)
                else:
                    folder = resource
        except errors.MkdirFolderAlreadyExist:
            # https://st.yandex-team.ru/CHEMODAN-27995
            # ничего не делаем, код дальше достанет этот ресурс и вернет его в этом случае
            pass

        if not (folder and folder.is_photostream()):
            folder = factory.get_resource(uid, address.id)
            self.setprop(uid, address.id, {'folder_type': type}, force=True, resource=folder)
            if type == 'photostream':
                from mpfs.core.user.base import User
                User(uid).set_state('photostream_used', 1)

        # https://jira.yandex-team.ru/browse/CHEMODAN-18157
        # Костыльеро для ярушки на время
        if type in SYSTEM_ATTACH_SYS_FOLDERS:
            self.setprop(uid, address.id, {'visible': 0}, force=True, resource=folder)

        folder.set_request(self.request, force=True)

        return folder.dict()

    def _send_mksysdir_event(self, uid, type):
        if self.method == 'mksysdir':
            fs_events.FilesystemCreateSystemDirectoryEvent(uid=str(uid), type=type).send()

    def check_overwrite(self, uid, address, force):
        address_obj = Address(address)
        try:
            if factory.is_unique(address_obj.id):
                parent_address = address_obj.get_parent()
                try:
                    parent_resource = factory.get_resource(uid, parent_address)
                except errors.ResourceNotFound:
                    raise errors.FolderNotFound(parent_address.id)
                if parent_resource.type == 'file':
                    raise errors.NotFolder()

                resource = factory.get_resource(uid, address_obj, request=self.request)
                resource.set_request(self.request)
                resource.load_views_counter()
                if resource.type == 'dir':
                    raise errors.MkdirFolderAlreadyExist(address_obj.id)
                if not force:
                    raise errors.StoreAlreadyExists(address_obj.id, headers={
                        'path': resource.id,
                        'resource_id': resource.resource_id.serialize()
                    })
                return resource.dict()
        except errors.ResourceNotFound, rnf:
            if isinstance(rnf, errors.FolderNotFound):
                raise errors.StoreParentNotFound(address_obj.id)
            return None

    @staticmethod
    def resource(uid, rawaddress, check_parent=True):
        """
        Получение ресурса
        """
        try:
            address = Address(rawaddress)
            if check_parent:
                parent = factory.get_resource(uid, address.get_parent())
            else:
                parent = None

            resource = factory.get_resource(uid, address)
            if parent:
                resource.set_parent(parent)

            return resource
        except errors.AddressError:
            raise errors.InfoPathError(rawaddress)

    def get_symlink_resource(self, uid, rawaddress, allow_deleted=False):
        """
        Получение ресурса по его симлинку
        """
        try:
            symlink_addr = SymlinkAddress(rawaddress)
            symlink = Symlink(symlink_addr)

            resource_id_uid = symlink.get_resource_id_uid()
            file_id = symlink.get_file_id()
            owner_uid = symlink.get_uid()
            if symlink.is_expired() and not allow_deleted:
                raise errors.ResourceNotFound('expired')
            if is_quick_move_enabled(owner_uid) and file_id is not None:
                if not resource_id_uid:
                    resource_id_uid = owner_uid
                resource = factory.get_not_removed_resource_by_file_id(resource_id_uid, file_id, is_for_public=True)
            else:
                address = Address(symlink.target_addr())
                resource = factory.get_resource(uid, address)
                if resource.is_trash or resource.is_hidden:
                    raise errors.ResourceNotFound(rawaddress)

            resource.set_request(self.request)
            return resource
        except errors.SymlinkNotFound:
            raise errors.ResourceNotFound(rawaddress)
        except errors.AddressError:
            raise errors.InfoPathError(rawaddress)

    def symlink(self, uid, rawaddress):
        """
        Создание ссылки на ресурс
        """
        # FIXME: Похоже на то, что этот метод не используется больше.
        address = Address(rawaddress)
        tgt_resource = factory.get_resource(uid, address)

        symlink = Symlink.Create(address, resource_id=tgt_resource.resource_id)
        tgt_resource.set_symlink(symlink)
        return symlink

    def symlink_create(self, *args, **kwargs):
        """
        Создание ссылки на ресурс без сохранения в ресурсе
        """
        tgt_resource = kwargs.get('resource')
        if not tgt_resource:
            uid, rawaddress = args[:2]
            address = Address(rawaddress)
            tgt_resource = factory.get_resource(uid, address)
        if tgt_resource.get_symlink() is not None:
            try:
                symlink = Symlink(SymlinkAddress(tgt_resource.get_symlink()))
                return symlink
            except Exception:
                pass

        kwargs['public_uid'] = tgt_resource.committer
        kwargs['resource_id'] = tgt_resource.resource_id

        symlink = Symlink.Create(tgt_resource.storage_address, **kwargs)
        return symlink

    def remove_symlink(self, uid, rawaddress, resource=None):
        """
        Удалить один симлинк
        """
        if not resource:
            address = Address(rawaddress)
            resource = factory.get_resource(uid, address)
        if resource.get_symlink() is not None:
            symlink = Symlink(SymlinkAddress(resource.get_symlink()))
            symlink.update_office_fields({'office_access_state': OfficeAccessStateConst.DISABLED})
            symlink.office_access_state = OfficeAccessStateConst.DISABLED
            symlink.delete()
            resource.remove_symlink()

    def _drop_symlinks_hierarchy(self, uid, tree):
        """
        Обойти дерево ресурсов и удалить симлинки, если есть
        """
        for v in tree:
            symlink_addr = SymlinkAddress(v.get('symlink'))
            try:
                symlink = Symlink(symlink_addr)
                symlink.update_office_fields({'office_access_state': OfficeAccessStateConst.DISABLED})
                symlink.delete()
            except errors.SymlinkNotFound:
                pass

            res_addr = Address.Make(v.get('uid'), v.get('id'))
            try:
                resource = factory.get_resource(uid, res_addr)
                if resource.get_symlink():
                    try:
                        resource.remove_symlink()
                        resource.make_private()
                    except errors.StorageKeyNotFound:
                        pass
            except errors.ResourceNotFound:
                pass

    def _update_symlinks_hierarchy(self, uid, tree):
        """
        Обойти дерево ресурсов и проапдетить симлинки, если есть
        """
        for v in tree:
            symlink_addr = SymlinkAddress(v.get('symlink'))
            symlink = Symlink(symlink_addr)
            resource_addr = Address.Make(v.get('uid'), v.get('id'))
            symlink.set_target(resource_addr)

    def make_folder_content_private(self, uid, address):
        try:
            folder = factory.get_resource(uid, address)
        except errors.ResourceNotFound:
            log.info('Task make_folder_content_private was not completed: ResourceNotFound uid=%s, address=%s' %
                     (uid, address))
            return

        folder.load_full_index_items()

        def process_folder(folder):
            for child_file in folder.children_items['files']:
                if not child_file.get_symlink():
                    child_file.make_private()
            for child_folder in folder.children_items['folders']:
                process_folder(child_folder)

        process_folder(folder)

    @classmethod
    def autosuffix_address(cls, address):
        """
        Если ресурс по переданному адресу существует, ищет такое n,
        при котором по адресу "адрес (n)" ресурс не существует
        """
        return cls.autosuffix_addresses([address])[0]

    @classmethod
    def autosuffix_addresses(cls, addresses):
        """
        как autosuffix_address, но учитывает другие файлы из списка
        """
        new_addresses = []
        seen_addresses = set()
        for address in addresses:
            new_address = address
            iteration = 0
            while True:
                conflict = ((new_address.uid, new_address.id) in seen_addresses or
                            cls.exists(new_address.uid, new_address.id))
                seen_addresses.add((new_address.uid, new_address.id))
                if conflict:
                    iteration += 1
                    new_address = address.add_suffix(' (%s)' % iteration)
                else:
                    new_addresses.append(new_address)
                    break
        return new_addresses

    # Теги CHEMODAN-14569
    def elements_for_tag(self, uid, rawaddress, tags):
        address = Address(rawaddress)
        service = factory.get_service(address)
        return service.elements_for_tag(uid, address.path, tags)

    def tags_for_element(self, uid, rawaddress):
        resource = factory.get_resource(uid, Address(rawaddress))
        return resource.get_tags()

    def tags_in_folder_list(self, uid, rawaddress):
        folder = factory.get_resource(uid, Address(rawaddress))
        return folder.get_children_tags_list()

    def tags_in_folder_tree(self, uid, rawaddress):
        folder = factory.get_resource(uid, Address(rawaddress))
        return folder.get_children_tags_tree()

    def elements_in_folder_list(self, uid, rawaddress, data):
        folder = factory.get_resource(uid, Address(rawaddress))
        return folder.get_children_list_by_tags(data)

    def elements_in_folder_tree(self, uid, rawaddress, data):
        folder = factory.get_resource(uid, Address(rawaddress))
        return folder.get_children_tree_by_tags(data)

    def tags_set(self, uid, rawaddress, scope, data):
        resource = factory.get_resource(uid, rawaddress)
        resource.set_tags(scope, data)

    def tags_photo_timeline(self, uid, rawaddress, system_tags, user_tags):
        address = Address(rawaddress)
        service = factory.get_service(address)
        return service.tags_photo_timeline(uid, system_tags, user_tags)

    def tags_photo_list(self, uid, rawaddress, filters, system_tags, user_tags):
        folder = factory.get_resource(uid, Address(rawaddress))
        return folder.tags_photo_list(filters, system_tags, user_tags)

    def tags_set_photo_all(self, uid, rawaddress):
        address = Address(rawaddress)
        service = factory.get_service(address)
        return service.tags_set_photo_all(uid)

    def tags_set_photo_file(self, uid, rawaddress, etime=None, file_data={}):
        if not file_data:
            file_data = self.info(uid, rawaddress)['this']
        if not etime:
            etime = file_data['meta'].get('etime')
        emid, exif = self.extract_exif(file_data['meta']['file_mid'])
        service = factory.get_service(Address(rawaddress))
        camera = {}
        geo = {}
        if 'Model' in exif:
            camera['model'] = exif['Model']
            if 'Lens' in exif:
                camera['lens'] = exif['Lens']
        if 'GPSPosition' in exif:
            gps_decimal = convert_gps_deg_dec(exif['GPSPosition'])
            geo['position'] = gps_decimal
            geo.update(geocoder_service.Geocoder().get_place(gps_decimal))
        service.tags_set_photo_file(uid, file_data['meta']['file_id'], etime, camera, geo)
        return emid, exif

    def extract_exif(self, file_mid):
        emid, exif = kladun_service.Kladun().extract_exif(file_mid)
        return emid, exif
    # теги завершились

    def clean_trash(self, uid, period=30, dryrun=False, batch_size=5000):
        """
        Аргументы
            uid - идентификатор пользователя
            period - сколько дней разрешаем хранить файлы
            dryrun - включает режим подсчёта без реального удаления
        Результат
            (deleted_resources, total_size, skipped)
            список удалённых элементов, суммарный их размер, кол-во пропущенных
        """

        log.debug("%s CLEAN_TRASH uid: %s dryrun: %s period: %s %s" % ('='*10, uid, dryrun, period, '='*10))

        # проверка превышения лимита на количество ресурсов в папке trash
        autoclean_resource_limit = settings.system['system']['trash_autoclean_resource_limit']
        if autoclean_resource_limit >= 0:
            count_resources = FolderDAO().get_resource_count(uid, '/trash')
            if count_resources > autoclean_resource_limit:
                log.info("uid: %s : exceeding clean limit, all resources skipped %s" % (uid, count_resources))
                return [], 0, count_resources

        bound_date = datetime.datetime.now() + datetime.timedelta(days=-int(period))
        timestamp = int(time.mktime(bound_date.timetuple()))

        skipped, to = 0, batch_size

        if is_new_fs_spec_required('trash'):
            query = {
                'uid': uid,
                'parent': "/trash",
            }
        else:
            query = {
                'uid': uid,
                'parent': id_for_key(uid, "/trash")
            }

        total_size = 0
        deleted_resources = []
        db = CollectionRoutedDatabase()

        resources_count = db['trash'].find(query).count()

        while True:
            for value in db['trash'].find(query, skip=skipped, limit=(to - skipped)):
                try:
                    path = value['key']
                    resource = factory.get_resource(uid, path)

                    if resource.meta['append_time'] < timestamp:
                        if dryrun:
                            if resource.is_file:
                                total_size += resource.size
                            else:
                                total_size += self.quota.get_size_for_tree(uid, path)
                        else:
                            self.trash_drop_element(uid, resource.address.id)
                        deleted_resources.append(resource.address.path)
                    else:
                        skipped += 1
                except:
                    error_log.error("Error processing resource: %s" % str(value))
                    error_log.error(traceback.format_exc())
                    skipped += 1

            if to > resources_count:
                break

            to += batch_size

        if not dryrun:
            total_size = int(math.fabs(Counter()._data[uid]['disk']))
            Counter().commit()

        log.info("uid: %s : %s resources in trash removed. Total size: %s. Skipped: %s" % (
            uid, len(deleted_resources), total_size, skipped)
        )

        return deleted_resources, total_size, skipped

    def add_to_trash_cleaner_queue(self, uid):
        date = int(datetime.datetime.now().strftime('%Y%m%d'))
        TrashCleanerQueueDAO().save({'uid': uid, 'date': date})

    def regenerate_preview_for_all_resources(self, address):
        """Перегенерить превьюшку ресурсу и проставить её всем ресурсам в БД, у которых есть старая превьюшка"""
        self._regenerate_stid_for_all_resources(address, 'pmid')

    def async_regenerate_preview_for_files(self, source_file_stid, source_file_name, source_file_size,
                                           source_file_mimetype, old_preview_stid, resource_ids):

        mpfs_queue.put({
            'source_file_stid': source_file_stid,
            'source_file_name': source_file_name,
            'source_file_size': source_file_size,
            'source_file_mimetype': source_file_mimetype,
            'old_preview_stid': old_preview_stid,
            'raw_resource_ids': [resource_id.serialize() for resource_id in resource_ids],
        }, 'regenerate_preview_for_files')

    def regenerate_preview_for_files(self, source_file_stid, source_file_name, source_file_size, source_file_mimetype,
                                     old_preview_stid, resource_ids):
        """
        Метод апдейтящий превью stid и атрибуты возвращаемые кладуном при перегенерации превью

        :param source_file_stid: stid файла из которого нужно сгенерировать превью
        :param source_file_name: имя файла из которого нужно сгенерировать превью
        :param source_file_size: размер файла из которого нужно сгенерировать превью
        :param source_file_mimetype: mimetype файла из которого нужно сгенерировать превью
        :param old_preview_stid: старый превью stid
        :param resource_ids: iterable с ResourceId файлов для которых нужно заменить превью и атрибуты возвращаемые
        кладуном при перегенерации превью
        :return: None
        """

        regenerate_preview_result = previewer_service.Previewer().regenerate_preview(
            name=source_file_name,
            size=source_file_size,
            mimetype=source_file_mimetype,
            file_mid=source_file_stid,
        )

        stids_for_clean = [
            DeletedStid(stid=old_preview_stid, stid_type='pmid', stid_source=DeletedStidSources.REGENERATE_PREVIEW)
        ]

        updated = FileDAO().update_preview_on_all_pg_shards(old_preview_stid, regenerate_preview_result)

        for resource_id in resource_ids:
            if mpfs.engine.process.usrctl().is_user_in_postgres(resource_id.uid):
                continue

            try:
                resource = factory.get_resource_by_resource_id(resource_id.uid, resource_id)
            except errors.ResourceNotFound:
                continue

            if 'pmid' not in resource.meta:
                error_log.error(
                    'Resource %s passed to regenerate_preview_for_files has not field "pmid" in meta' % resource_id)
                continue

            if resource.meta['pmid'] != old_preview_stid:
                continue

            resource.meta['pmid'] = regenerate_preview_result.pmid

            if regenerate_preview_result.original_width is not None \
                    and regenerate_preview_result.original_height is not None:
                resource.meta['width'] = regenerate_preview_result.original_width
                resource.meta['height'] = regenerate_preview_result.original_height

            if regenerate_preview_result.rotate_angle is not None:
                resource.meta['angle'] = regenerate_preview_result.rotate_angle

            if regenerate_preview_result.has_video_info():
                resource.meta['video_info'] = regenerate_preview_result.video_info

            resource.save(skip_parents_check=True)
            updated = True

        if not updated:
            stids_for_clean.append(
                DeletedStid(
                    stid=regenerate_preview_result.pmid,
                    stid_type='pmid',
                    stid_source=DeletedStidSources.REGENERATE_PREVIEW,
                )
            )
        else:
            log.info('Preview regenerated for stid %s' % source_file_stid)

        DeletedStid.controller.bulk_create(stids_for_clean, get_size_from_storage=True)


    def regenerate_digest_for_all_resources(self, address):
        """Перегенерить digest ресурсу и проставить его всем ресурсам в БД, у которых есть старый digest"""
        self._regenerate_stid_for_all_resources(address, 'digest_mid')

    def _regenerate_stid_for_all_resources(self, address, stid_meta_field_name):
        if stid_meta_field_name not in ['pmid', 'digest_mid']:
            raise ValueError()
        if not isinstance(address, Address):
            raise TypeError()

        resource = factory.get_resource(address.uid, normalize_unicode(address.path))
        old_stid = resource.meta.get(stid_meta_field_name)
        if stid_meta_field_name == 'pmid':
            resource.regenerate_preview()
        else:
            resource.regenerate_digest()
        new_stid = resource.meta.get(stid_meta_field_name)

        if not old_stid:
            return

        for resource in factory.iter_resources_by_stids([old_stid]):
            resource.meta[stid_meta_field_name] = new_stid
            resource.save(skip_parents_check=True)

    def _is_public_or_shared_or_has_yarovaya_mark_by_resource_dict(self, resource_dict):
        if not (isinstance(resource_dict.get('data'), dict) and isinstance(resource_dict['data'].get('meta'), dict)):
            return False
        meta = resource_dict['data']['meta']
        return (any(meta.get(k) is not None for k
                    in ('symlink', 'public', 'public_hash', 'short_url', 'short_url_named', 'group'))
                or bool(meta.get('yarovaya_mark', False)))


class merge(object):
    def __init__(self, src, dst_address, dst):
        self.src = src
        self.dst = dst

        def new_id(item):
            item['new_path'] = change_parent(
                Address.Make(item['uid'], item['id'], type=item['type']),
                dst_address.id,
                src.id
            )
            return item

        self.src_index = map(
            new_id,
            sorted(
                src.get_full_index(root=False).itervalues(),
                key=operator.itemgetter('id')
            )
        )

    def diff_size(self):
        diff_size = 0
        if self.dst:
            if self.dst.is_shared:
                dst_children = {}
            else:
                dst_children = dict(
                    (it['key'], int(it['size'])) for it in filter(
                        lambda i: i['type'] == 'file',
                        self.dst.index()[1:]
                    )
                )
            for v in self.src_index:
                if v['type'] == 'file':
                    dst_file_size = dst_children.get(v['new_path'].path, 0)
                    diff_size += int(v['size']) - dst_file_size
        return diff_size
