# -*- coding: utf-8 -*-
from multiprocessing.pool import ThreadPool

from mpfs.common.util import logger
from mpfs.config import settings
from mpfs.core.controllers import BaseModelController
from mpfs.core.filesystem.cleaner.models import DeletedStid, DeletedStidRetry, StorageCleanCheckStid
from mpfs.core.metastorage.control import deleted_stids, deleted_stids_retry_remove, storage_clean_check_stids
from mpfs.core.services.mulca_service import Mulca

deleted_stids_log = logger.get('mpfs.deleted-stids')
MULCA_THREADPOOL_SIZE = 10


def get_size_from_storage_if_needed(deleted_stid):
    try:
        if deleted_stid.size is None:
            deleted_stid.size = Mulca().get_file_size(deleted_stid.stid)
    except Exception:
        pass
    return deleted_stid


class DeletedStidsController(BaseModelController):
    model = DeletedStid
    collection = deleted_stids

    def bulk_create(self, objs, get_size_from_storage=False, **kwargs):
        """
        Фэйковое создание обьектов с записью в лог для последущей аггрегации на YT
        :param objs: Список объектов для создания.
        :param get_size_from_storage: Забирать ли размер стида из стораджа, если его нет
        :param **kwargs: для совместимости с базовым методом
        """
        if get_size_from_storage:
            pool = ThreadPool(processes=MULCA_THREADPOOL_SIZE)
            objs = pool.imap_unordered(get_size_from_storage_if_needed, objs)
            pool.close()
            pool.join()

        for deleted_stid in objs:
            deleted_stids_log.info(deleted_stid.as_tskv())

    def real_bulk_create(self, objs, continue_on_error=True):
        """Создание обьектов в коллекции контроллера"""
        from mpfs.common.errors import StorageError
        try:
            return super(DeletedStidsController, self).bulk_create(objs, continue_on_error=True)
        except StorageError:
            # так игнорируем DuplicateKeyError
            return []


class DeletedStidsRetryController(BaseModelController):
    model = DeletedStidRetry
    collection = deleted_stids_retry_remove

    def bulk_create(self, objs, continue_on_error=True):
        from mpfs.common.errors import StorageError
        try:
            return super(DeletedStidsRetryController, self).bulk_create(objs, continue_on_error=True)
        except StorageError:
            # так игнорируем DuplicateKeyError
            return []


class StorageCleanCheckStidsController(BaseModelController):
    model = StorageCleanCheckStid
    collection = storage_clean_check_stids
