# -*- coding: utf-8 -*-
import time
import random
import datetime

import mpfs.engine.process
from mpfs.common.util import chunks2

from mpfs.common.util.rps_limiter import InMemoryRPSLimiter
from mpfs.config import settings
from mpfs.core.filesystem.cleaner.models import DeletedStid, DeletedStidSources
from mpfs.core.filesystem.dao.legacy import CollectionRoutedDatabase
from mpfs.core.filesystem.dao.resource import AdditionalDataDAO
from mpfs.core.services.queller_service import QuellerTask
from mpfs.core.metastorage.control import support_prohibited_cleaning_users
from mpfs.metastorage.postgres.query_executer import PGQueryExecuter

logger = mpfs.engine.process.get_default_log()

HIDDEN_DATA_CLEANER_ENABLED = settings.hidden_data_cleaner['enabled']
HIDDEN_DATA_CLEANER_WORKER_CLEAN_DELAY = settings.hidden_data_cleaner['worker']['clean_delay']
HIDDEN_DATA_CLEANER_WORKER_STIDS_IGNORE_LIST = settings.hidden_data_cleaner['worker']['stids_ignore_list']
HIDDEN_DATA_CLEANER_WORKER_MIN_FILE_SIZE = settings.hidden_data_cleaner['worker']['min_file_size']
HIDDEN_DATA_CLEANER_WORKER_EXECUTING_TIMEOUT = settings.hidden_data_cleaner['worker']['executing_timeout']
HIDDEN_DATA_CLEANER_WORKER_FETCH_BATCH_SIZE = settings.hidden_data_cleaner['worker']['fetch_batch_size']
HIDDEN_DATA_CLEANER_WORKER_PROCESS_BATCH_SIZE = settings.hidden_data_cleaner['worker']['process_batch_size']
HIDDEN_DATA_CLEANER_WORKER_RPS_LIMIT = settings.hidden_data_cleaner['worker']['rps_limit']
POSTGRES_ENABLE_STORAGE_CLEANING = settings.postgres['enable_storage_cleaning']


class HiddenDataCleanerToggle(object):
    """
    Рубильник чистки hidden_data
    """
    config_toggle = HIDDEN_DATA_CLEANER_ENABLED

    @classmethod
    def is_enable(cls):
        return cls.config_toggle


class CleanTimeout(Exception):
    pass


class CleanNonHidden(Exception):
    pass


class ShardRoutedHiddenDAO(object):
    def __init__(self, shard_name):
        self.shard_name = shard_name
        self.dao = CollectionRoutedDatabase()['hidden_data']

    def find_files_for_cleaning(self, min_file_size, max_datetime, filter_uids=None, limit=None):
        return self.dao.find_files_for_cleaning(self.shard_name, min_file_size, max_datetime, filter_uids, limit)

    def remove_files_by_uid_id(self, uid_id_list):
        return self.dao.remove_files_by_uid_id(self.shard_name, uid_id_list)

    def remove_hanging_storage_files_by_ids(self, storage_ids):
        return self.dao.remove_hanging_storage_files_by_ids(self.shard_name, storage_ids)

    def unset_live_photo_flag_by_uid_id(self, uid_id_list):
        return self.dao.unset_live_photo_flag_by_uid_id(self.shard_name, uid_id_list)


class ShardRoutedAdditionalDAO(object):
    def __init__(self, shard_name):
        self.shard_name = shard_name
        self.dao = AdditionalDataDAO()

    def find_files_by_uid_id_on_shard(self, uid_id_list):
        return self.dao.find_files_by_uid_id_on_shard(self.shard_name, uid_id_list)

    def get_video_ids_by_file_ids_on_shard(self, uid_id_list):
        return self.dao.get_video_ids_by_file_ids_on_shard(self.shard_name, uid_id_list)

    def remove_links_by_uid_id_on_shard(self, uid_id_list):
        return self.dao.remove_links_by_uid_id_on_shard(self.shard_name, uid_id_list)

    def remove_files_by_uid_id_on_shard(self, uid_id_list):
        return self.dao.remove_files_by_uid_id_on_shard(self.shard_name, uid_id_list)


class HiddenDataCleanerWorker(QuellerTask):
    """
    Воркер чистки коллекции `hidden_data`

    Получет на вход имя шарда и чистит ему коллекцию `hidden_data`
    Обрабатываем документы по следующим правилам:
        * Документ должен отлежаться `CLEAN_DELAY` дней
        * Из файлов достаем stid-ы и кладем в `deleted_stids`, затем удаляем
    # TODO Чистка папок
    """
    toggle = HiddenDataCleanerToggle

    stids_ignore_set = set(HIDDEN_DATA_CLEANER_WORKER_STIDS_IGNORE_LIST)
    CLEAN_DELAY = HIDDEN_DATA_CLEANER_WORKER_CLEAN_DELAY
    MIN_FILE_SIZE = HIDDEN_DATA_CLEANER_WORKER_MIN_FILE_SIZE
    EXECUTING_TIMEOUT = HIDDEN_DATA_CLEANER_WORKER_EXECUTING_TIMEOUT
    FETCH_BATCH_SIZE = HIDDEN_DATA_CLEANER_WORKER_FETCH_BATCH_SIZE
    PROCESS_BATCH_SIZE = HIDDEN_DATA_CLEANER_WORKER_PROCESS_BATCH_SIZE

    def __init__(self):
        self._sharded_hidden_dao = None
        self._sharded_additional_dao = None

        self._start_ts = None
        self._shard_name = None

        self._rps_limiter = InMemoryRPSLimiter(HIDDEN_DATA_CLEANER_WORKER_RPS_LIMIT)
        QuellerTask.__init__(self)

    @classmethod
    def put(cls, shard):
        return cls._put({'shard': shard})

    def run(self, *args, **kwargs):
        self._shard_name = kwargs['shard']
        self._start_ts = time.time()
        if not self.toggle.is_enable():
            logger.info('Disabled.')
            return
        if not self._shard_name:
            logger.info('No shard name.')
            return

        try:
            self._run()
        except CleanTimeout:
            exec_stat = 'Timeout'
        else:
            exec_stat = 'Done'

        logger.info(('"%(name)s" did the job. '
                     'shard: "%(shard_name)s"; processing_time: "%(processing_time)0.3f"; stat: "%(stat)s".'
                     '') % {'shard_name': self._shard_name,
                            'name': self.__class__.__name__,
                            'stat': exec_stat,
                            'processing_time': time.time() - self._start_ts})

    def _delete_files(self, file_items):
        for item in file_items:
            if not item.path.startswith('/hidden/'):
                raise CleanNonHidden()

        photo_ids = []
        for item in file_items:
            if item.is_live_photo:
                photo_ids.append((item.uid, item.fid))

        if photo_ids:
            video_ids = self._sharded_additional_dao.get_video_ids_by_file_ids_on_shard(photo_ids)
            for video_item in self._sharded_additional_dao.find_files_by_uid_id_on_shard(video_ids):
                if not video_item.path.startswith('/additional/'):
                    continue
                file_items.append(video_item)

            # удаляем флаг is_live_photo, потому что в базе есть триггер с проверкой process_live_photo_link_action
            self._rps_limiter.block_until_allowed(len(photo_ids))
            self._sharded_hidden_dao.unset_live_photo_flag_by_uid_id(photo_ids)
            # удаляем все линки от live video
            self._rps_limiter.block_until_allowed(len(photo_ids + video_ids))
            self._sharded_additional_dao.remove_links_by_uid_id_on_shard(photo_ids + video_ids)

        # удаляем документы
        file_ids = [(i.uid, i.fid) for i in file_items]
        if file_ids:
            self._rps_limiter.block_until_allowed(len(file_ids))
            self._sharded_hidden_dao.remove_files_by_uid_id(file_ids)

        return file_items

    def _delete_storage_files(self, file_items):
        if not file_items:
            return []

        return self._sharded_hidden_dao.remove_hanging_storage_files_by_ids(set(i.hid for i in file_items))

    def _save_deleted_stids(self, file_items, deleted_storage_ids):
        deleted_storage_ids = set(deleted_storage_ids)
        deleted_stids_objs = []
        for item in file_items:
            if item.hid not in deleted_storage_ids:
                continue
            stids = (
                (item.file_stid, 'file_mid'),
                (item.digest_stid, 'digest_mid'),
                (item.preview_stid, 'pmid'),
            )

            for (stid, stid_type) in stids:
                if stid is None:
                    continue
                if stid in self.stids_ignore_set:
                    continue

                deleted_stids_objs.append(
                    DeletedStid(
                        stid=stid,
                        stid_type=stid_type,
                        hid=item.hid,
                        size=item.size if stid_type == 'file_mid' else None,
                        stid_source=DeletedStidSources.HIDDEN_DATA_CLEAN,
                    )
                )
        DeletedStid.controller.bulk_create(deleted_stids_objs, get_size_from_storage=True)

    def _run(self):
        bound_dt = datetime.datetime.now() + datetime.timedelta(days=-self.CLEAN_DELAY)
        prohibited_cleaning_users_uids = {
            o['uid'] for o in support_prohibited_cleaning_users.list_all() if o.get('uid')
        }
        logger.info('Prohibited cleaning uids: %s.' % prohibited_cleaning_users_uids)

        self._sharded_hidden_dao = ShardRoutedHiddenDAO(self._shard_name)
        self._sharded_additional_dao = ShardRoutedAdditionalDAO(self._shard_name)

        if self._shard_name.startswith('disk'):
            raise NotImplementedError()

        while True:
            files_to_clean = self._sharded_hidden_dao.find_files_for_cleaning(self.MIN_FILE_SIZE, bound_dt,
                                                                              prohibited_cleaning_users_uids,
                                                                              self.FETCH_BATCH_SIZE)
            files_count = 0
            for file_items in chunks2(files_to_clean, self.PROCESS_BATCH_SIZE):
                file_items = self._delete_files(file_items)
                deleted_storage_ids = self._delete_storage_files(file_items)
                self._save_deleted_stids(file_items, deleted_storage_ids)
                files_count += len(file_items)
                if time.time() - self._start_ts > self.EXECUTING_TIMEOUT:
                    raise CleanTimeout()

            has_files_to_clean = files_count >= self.FETCH_BATCH_SIZE
            if not has_files_to_clean:
                break


class HiddenDataCleanerManager(object):
    """
    Менеджер чистки `hidden_data`

    Алгоритм:
        0. Проверяет, что нет активных воркеров чистки hidden_data
        1. Из конфига читает во сколько воркеров можно чистить `hidden_data`
        2. Смотрит в системной БД какие у нас есть шарды
        3. Разбивает массив шардов на число групп == значению из конфига(п.1)
        4. Запускает воркеров
    """
    toggle = HiddenDataCleanerToggle

    def run(self):
        if not self.toggle.is_enable():
            logger.info('Disabled. Exit.')
            return
        if HiddenDataCleanerWorker.not_finished_len() > 0:
            logger.info('Find active "%s". Exit.' % HiddenDataCleanerWorker)
            return

        shard_names = mpfs.engine.process.dbctl().mapper.rspool.get_all_shards_names()
        if POSTGRES_ENABLE_STORAGE_CLEANING:
            shard_names += PGQueryExecuter().get_all_shard_ids()
        random.shuffle(shard_names)
        for shard in shard_names:
            if not shard:
                continue
            HiddenDataCleanerWorker().put(shard)
            logger.info('%s create tasks: %r' % (self.__class__.__name__, shard))
