# -*- coding: utf-8 -*-
import mpfs.engine.process

from mpfs.config import settings
from mpfs.core.filesystem.cleaner.models import DeletedStid
from mpfs.core.filesystem.cleaner.worker import StorageCleanerWorker
from mpfs.core.services.bazinga_service import BazingaInterface
from mpfs.common.util import chunks2


STORAGE_CLEANER_ENABLED = settings.storage_cleaner['enabled']

STORAGE_CLEANER_MANAGER_BATCH_SIZE = settings.storage_cleaner['manager']['batch_size']
STORAGE_CLEANER_MANAGER_STIDS_PER_WORKER = settings.storage_cleaner['manager']['stids_per_worker']


class StorageCleanerManager(object):
    """Менеджер чистки стораджа

    Механизм работы такой:
        manager = BazingaManagerCleaner()
        manager.run()
    """
    ENABLED = STORAGE_CLEANER_ENABLED
    BATCH_SIZE = STORAGE_CLEANER_MANAGER_BATCH_SIZE
    STIDS_PER_WORKER = STORAGE_CLEANER_MANAGER_STIDS_PER_WORKER
    worker_class = StorageCleanerWorker

    def run(self):
        log = mpfs.engine.process.get_default_log()
        if not self.ENABLED:
            log.info("Disabled. Check config.")
            return

        bazinga_active_tasks_num = self.worker_class.not_finished_len()
        active_task_limit = self._get_active_task_limit()
        if bazinga_active_tasks_num > active_task_limit:
            log.info("Too many active tasks %i (limit: %i).", bazinga_active_tasks_num, active_task_limit)
            return

        spawned_workers_num = self._run()
        log.info("Spawned %i workres.", spawned_workers_num)

    def _get_active_task_limit(self):
        deleted_stids_len = DeletedStid.controller.count()
        if deleted_stids_len <= self.BATCH_SIZE:
            # если в `delete_stids` осталось мало stid-ов, то ждем пока все таски добегут
            return 0
        # если больше 99% предыдущей пачки обработано, можно обрабатывать следующую пачку
        return int(0.01 * self.BATCH_SIZE / self.STIDS_PER_WORKER)

    def _run(self):
        stids = (i.stid for i in DeletedStid.controller.order_by('-size')[:self.BATCH_SIZE])
        grouped_stids = chunks2(stids, chunk_size=self.STIDS_PER_WORKER)
        tasks = (self.worker_class(s) for s in grouped_stids)
        return len(BazingaInterface().bulk_create_tasks(tasks))
