import time
from datetime import datetime, timedelta

import click
import yt.wrapper as yt
from nile.api.v1 import clusters
from qb2.api.v1 import extractors as se
from qb2.api.v1 import filters as qf

import mpfs.common.errors as errors
import mpfs.engine.process
from mpfs.common.util import chunks2
from mpfs.common.util.rps_limiter import InMemoryRPSLimiter
from mpfs.config import settings
from mpfs.core.filesystem.cleaner.models import DeletedStid
from mpfs.core.mrstat.stat_utils import set_yt_proxy
from mpfs.core.services.juggler import JEvent, JStatus

log = mpfs.engine.process.get_default_log()

YT_MERGE = settings.hidden_data_cleaner['yt_merge']
YT_MERGE_ENABLED = YT_MERGE['enabled']
YT_MERGE_SLEEP_SECS = YT_MERGE['sleep_secs']
YT_MERGE_RETRY_LIMIT = YT_MERGE['retry_limit']
YT_MERGE_MIN_SIZE = YT_MERGE['min_size']

CREATE_TASKS_FROM_YT = settings.hidden_data_cleaner['create_tasks_from_yt']
CREATE_TASKS_FROM_YT_ENABLED = CREATE_TASKS_FROM_YT['enabled']
CREATE_TASKS_FROM_YT_BATCH_SIZE = CREATE_TASKS_FROM_YT['yt']['batch_size']
CREATE_TASKS_FROM_YT_ERASE_ENABLED = CREATE_TASKS_FROM_YT['yt']['erase_enabled']
CREATE_TASKS_FROM_YT_DELETED_STIDS_COUNT_LIMIT = CREATE_TASKS_FROM_YT['deleted_stids']['count_limit']
CREATE_TASKS_FROM_YT_DELETED_STIDS_ADD_LIMIT = CREATE_TASKS_FROM_YT['deleted_stids']['add_limit']
CREATE_TASKS_FROM_YT_DELETED_STIDS_RPS_LIMIT = CREATE_TASKS_FROM_YT['deleted_stids']['rps_limit']
CREATE_TASKS_FROM_YT_DELETED_STIDS_CHUNK_SIZE = CREATE_TASKS_FROM_YT['deleted_stids']['chunk_size']


class DeletedStidsDayMergeSortEvent(JEvent):
    host = 'disk_pworker_devops'
    service = 'DeletedStidsDayMergeSort'


def day_merge_sort(day):
    if not YT_MERGE_ENABLED:
        click.echo('day_merge_sort disabled')
        return

    job_root = '//home/mpfs-stat/storage'
    table_name = 'stids_cleaning_candidates'
    result_table = job_root + '/' + table_name
    result_table_small = result_table + '_small'

    cluster = clusters.Hahn(token=settings.mrstat['yt_token']).env(templates=dict(job_root=job_root))
    job = cluster.job()

    day = day or str((datetime.now() - timedelta(1)).date())
    src_log = job.table('//home/logfeller/logs/ydisk-mpfs-deleted-stids-log/1d/' + day)

    daily_stids_small, daily_stids = src_log.qb2(
        log='generic-yson-log',
        fields=[se.log_fields('ctime', 'stid', 'stid_type', 'hid', 'stid_source'), se.integer_log_field('size')]
    ).split(
        qf.compare('size', '>', value=YT_MERGE_MIN_SIZE),
    )

    last_error = None
    for _ in range(YT_MERGE_RETRY_LIMIT):
        try:
            with cluster.driver.client.Transaction():
                # Wait for lock 40 minutes
                cluster.driver.client.lock(result_table, waitable=True, wait_for=40 * 60 * 1000)
                daily_stids.concat(job.table(result_table)).unique('stid').sort('size').put(result_table)
                daily_stids_small.concat(job.table(result_table_small, ignore_missing=True)).put(result_table_small)
                job.run()
        except Exception as e:
            log.exception(e)
            last_error = e
            time.sleep(YT_MERGE_SLEEP_SECS)
        else:
            juggler_event = DeletedStidsDayMergeSortEvent(JStatus.OK, description='OK')
            break
    else:
        juggler_event = DeletedStidsDayMergeSortEvent(JStatus.CRIT, description=last_error.message)

    try:
        juggler_event.send()
    except errors.APIError:
        pass


def create_tasks_from_yt():
    if not CREATE_TASKS_FROM_YT_ENABLED:
        click.echo('create_tasks_from_yt disabled')
        return

    set_yt_proxy()
    table_name = '//home/mpfs-stat/storage/stids_cleaning_candidates'

    deleted_stids_controller = DeletedStid.controller
    collection_size = deleted_stids_controller.count()
    if collection_size > CREATE_TASKS_FROM_YT_DELETED_STIDS_COUNT_LIMIT:
        log.info('DeletedStidsCollection is too big: %s' % collection_size)
        return

    rps_limiter = InMemoryRPSLimiter(CREATE_TASKS_FROM_YT_DELETED_STIDS_RPS_LIMIT)
    tasks_created = 0
    while tasks_created < CREATE_TASKS_FROM_YT_DELETED_STIDS_ADD_LIMIT:
        with yt.Transaction():
            # Wait for lock 5 minutes
            yt.lock(table_name, waitable=True, wait_for=5 * 60 * 1000)
            last_index = yt.row_count(table_name)
            start_index = 0
            if last_index > CREATE_TASKS_FROM_YT_BATCH_SIZE:
                start_index = last_index - CREATE_TASKS_FROM_YT_BATCH_SIZE

            table_path = yt.TablePath(table_name, start_index=start_index)
            records = yt.read_table(table_path, format=yt.JsonFormat())
            for records_chunk in chunks2(records, chunk_size=CREATE_TASKS_FROM_YT_DELETED_STIDS_CHUNK_SIZE):
                stids_chunk = [DeletedStid(**r) for r in records_chunk if r['stid']]
                try:
                    rps_limiter.block_until_allowed(len(stids_chunk))
                    tasks_created += len(deleted_stids_controller.real_bulk_create(stids_chunk))
                except Exception as e:
                    log.exception(e)
                    break
            else:
                if CREATE_TASKS_FROM_YT_ERASE_ENABLED:
                    yt.run_erase(table_path)

            if start_index == 0:
                break
