# -*- coding: utf-8 -*-

from bson import ObjectId

from mpfs.dao.base import BaseDAO, BaseDAOItem, PostgresBaseDAOImplementation
from mpfs.dao.fields import ChangelogOpTypeField, ChangelogSharedTypeField, CompressedJsonField, \
    IntegerField, ObjectIdField, ResourceTypeField, StringField, UidField, UuidField, MSKDateTimeField
from mpfs.metastorage.mongo.util import decompress_data, compress_data
from mpfs.metastorage.postgres.schema import changelog


class ChangelogDAOItem(BaseDAOItem):
    mongo_collection_name = 'changelog'
    postgres_table_obj = changelog
    is_sharded = True

    id = ObjectIdField(mongo_path='_id', pg_path=changelog.c.id)
    uid = UidField(mongo_path='uid', pg_path=changelog.c.uid)
    path = StringField(mongo_path='key', pg_path=changelog.c.path)
    type = ResourceTypeField(mongo_path='type', pg_path=changelog.c.type)
    op = ChangelogOpTypeField(mongo_path='op', pg_path=changelog.c.op)
    version = IntegerField(mongo_path='version', pg_path=changelog.c.version)

    zdata = CompressedJsonField(mongo_path='zdata', pg_path=changelog.c.zdata, default_value=None)

    gid = UuidField(mongo_path='gid', pg_path=changelog.c.gid, default_value=None)
    group_path = StringField(mongo_path='group_path', pg_path=changelog.c.group_path, default_value=None)
    dtime = MSKDateTimeField(mongo_path='dtime', pg_path=changelog.c.dtime, default_value=None)

    shared = ChangelogSharedTypeField(mongo_path='shared', pg_path=changelog.c.shared, default_value=None)
    rights = IntegerField(mongo_path='rights', pg_path=changelog.c.rights, default_value=None)

    exclude_keys_after_conversion_to_mongo = {
        'zdata': None,
        'gid': None,
        'group_path': None,
        'dtime': None,
        'shared': None,
        'rights': None,
    }

    def _mongo_getter(self, key):
        # В монго записи разных типов сейчас хранятся по-разному:
        #     значения key, type, op могут лежать либо в zdata, либо в корне объекта.
        # Чтобы пока не учить фреймворк искать по нескольким путям, решил сделать костыль тут
        # Из zdata они убираются, чтобы не хранились в двух местах и невозможно было получить неконсистентное состояние
        if key in ('key', 'type', 'op'):
            if key in self._mongo_dict:
                return super(ChangelogDAOItem, self)._mongo_getter(key)
            else:
                return super(ChangelogDAOItem, self)._mongo_getter('zdata.' + key)
        if key == 'zdata':
            zdata = super(ChangelogDAOItem, self)._mongo_getter(key)
            zdata = decompress_data(zdata)
            zdata.pop('key', None)
            zdata.pop('type', None)
            zdata.pop('op', None)
            zdata = compress_data(zdata)
            return zdata
        return super(ChangelogDAOItem, self)._mongo_getter(key)


class ChangelogDAO(BaseDAO):
    dao_item_cls = ChangelogDAOItem

    def __init__(self):
        super(ChangelogDAO, self).__init__()
        self._pg_impl = PostgresChangelogDAOImplementation(self.dao_item_cls)


class PostgresChangelogDAOImplementation(PostgresBaseDAOImplementation):
    def insert(self, doc_or_docs, manipulate=True, continue_on_error=False, **kwargs):
        if not isinstance(doc_or_docs, list):
            if '_id' not in doc_or_docs:
                doc_or_docs['_id'] = ObjectId()
        else:
            for doc in doc_or_docs:
                if '_id' not in doc:
                    doc['_id'] = ObjectId()

        return super(PostgresChangelogDAOImplementation, self).insert(doc_or_docs, manipulate, continue_on_error, **kwargs)
