# -*- coding: utf-8 -*-
from sqlalchemy.engine import RowProxy

from mpfs.common.util import from_json
from mpfs.common.util.filetypes import builtin_extensions
from mpfs.metastorage.mongo.binary import Binary
from mpfs.metastorage.mongo.util import decompress_data
from mpfs.metastorage.postgres.schema import MediaType, AntiVirusScanStatus


class CustomSetpropFieldsMixin(object):
    """    
    Миксин для костыльного решения - хранения всех значений в zdata.setprop, которые не попали в отдельные филды,
    в отдельном поле класса _custom_setprop_fields, которое затем либо просто добавляется в монгу в zdata.setprop,
    либо записывается в отдельный jsonb в постгресе.
    Это могут быть произвольные филды, которые можно установить через ручку setprop.
    Класс необходим только для FileDAOItem и FolderDAOItem.
    """

    def __init__(self):
        from mpfs.core.filesystem.dao.file import FileDAOItem  # цикл. импорт
        from mpfs.core.filesystem.dao.folder import FolderDAOItem  # цикл. импорт
        
        assert isinstance(self, (FileDAOItem, FolderDAOItem))
        assert 'custom_setprop_fields' in self._fields
        assert self.postgres_table_obj is not None

        super(CustomSetpropFieldsMixin, self).__init__()
        self._custom_setprop_fields = None

    @classmethod
    def create_from_mongo_dict(cls, data, validate=False):
        if 'zdata' in data:
            data['zdata'] = decompress_data(data['zdata'])
        setprop_fields = data.get('zdata', {}).get('setprop', {})

        custom_setprop_fields = {}
        for key, value in setprop_fields.iteritems():
            has_custom_field = False
            for field_name, field_obj in cls._fields.iteritems():
                if field_obj.mongo_path == 'zdata.setprop.' + key:
                    has_custom_field = True
                    break
            if not has_custom_field:
                custom_setprop_fields[key] = value

        for key in custom_setprop_fields.iterkeys():
            data['zdata']['setprop'].pop(key)

        instance = super(CustomSetpropFieldsMixin, cls).create_from_mongo_dict(data, validate=validate)
        if custom_setprop_fields:
            instance._custom_setprop_fields = custom_setprop_fields

        return instance

    @classmethod
    def create_from_pg_data(cls, data):
        if isinstance(data, RowProxy):
            pg_data = cls._convert_row_proxy_to_pg_data(data)
        else:
            pg_data = data

        custom_setprop_fields = None
        table = cls.postgres_table_obj
        if table.c.custom_setprop_fields in pg_data and pg_data[table.c.custom_setprop_fields] is not None:
            custom_setprop_fields = cls._fields['custom_setprop_fields'].from_postgres(
                pg_data[table.c.custom_setprop_fields])
            pg_data[table.c.custom_setprop_fields] = None

        instance = super(CustomSetpropFieldsMixin, cls).create_from_pg_data(pg_data)
        if custom_setprop_fields:
            instance._custom_setprop_fields = custom_setprop_fields

        return instance

    def _common_getter(self, field_obj):
        if field_obj.mongo_path == 'custom_setprop_fields':
            return self._custom_setprop_fields
        return super(CustomSetpropFieldsMixin, self)._common_getter(field_obj)

    def get_postgres_representation(self, skip_missing_fields=False):
        pg_data = super(CustomSetpropFieldsMixin, self).get_postgres_representation(skip_missing_fields)
        table = self.postgres_table_obj
        if self._custom_setprop_fields:
            pg_data[table.c.custom_setprop_fields] = self._fields['custom_setprop_fields'].to_postgres(
                self._custom_setprop_fields)
        return pg_data

    def get_mongo_representation(self, skip_missing_fields=False):
        # костыль, чтобы в словаре возвращался еще и тип, в BaseDAOItem это сделать нельзя, там общая логика
        mongo_dict = super(CustomSetpropFieldsMixin, self).get_mongo_representation(skip_missing_fields)
        mongo_dict.pop('custom_setprop_fields', None)
        if self._custom_setprop_fields:
            if 'zdata' not in mongo_dict:
                mongo_dict['zdata'] = {}
            if 'setprop' not in mongo_dict['zdata']:
                mongo_dict['zdata']['setprop'] = {}
            for key, value in self._custom_setprop_fields.iteritems():
                mongo_dict['zdata']['setprop'][key] = value
        return mongo_dict


class OptimizedResourceConvertionMixin(object):
    """
    Миксин для переопределения метода конвертации запросов из постгреса в монговое представление.
    Нужен для файлов и папок.
    """

    def __init__(self):
        from mpfs.core.filesystem.dao.file import FileDAOItem  # цикл. импорт
        from mpfs.core.filesystem.dao.folder import FolderDAOItem  # цикл. импорт

        assert isinstance(self, (FileDAOItem, FolderDAOItem))
        assert 'custom_setprop_fields' in self._fields
        assert self.postgres_table_obj is not None

        super(OptimizedResourceConvertionMixin, self).__init__()

    @classmethod
    def create_from_pg_data(cls, data):
        flat_dict = {}

        for field_name, field_obj in cls._fields.iteritems():
            try:
                pg_field_name = field_obj.pg_path
                mongo_value = data[pg_field_name]
            except KeyError:
                continue
            except Exception as e:
                raise type(e)(e.message + ' (field \'%s\')' % field_name)

            # Скопипащено из кода преобразования филдов для ускорения
            if field_name == 'custom_setprop_fields':
                continue
            elif field_name == 'mediatype' and mongo_value is not None:
                mongo_value = builtin_extensions[MediaType(mongo_value).value]
            elif field_name == 'antivirus_status':
                value = AntiVirusScanStatus(mongo_value)
                if value == AntiVirusScanStatus.clean:
                    mongo_value = 1
                elif value == AntiVirusScanStatus.infected:
                    mongo_value = 2
                else:
                    mongo_value = 4
            elif field_name == 'video_info' and mongo_value is not None:
                mongo_value = from_json(mongo_value)
            elif field_name == 'hid':
                mongo_value = Binary(str(mongo_value))

            if field_name in ('file_stid', 'preview_stid', 'digest_stid'):
                if field_obj.mongo_path not in flat_dict:
                    flat_dict[field_obj.mongo_path] = []
                if field_name == 'preview_stid' and mongo_value is None:
                    continue
                flat_dict[field_obj.mongo_path].append(
                    {'stid': mongo_value, 'type': field_obj.mongo_item_parser.stid_type})
            else:
                flat_dict[field_obj.mongo_path] = mongo_value

        mongo_dict = {}
        if cls.exclude_keys_after_conversion_to_mongo:
            for key, value in flat_dict.iteritems():
                if key in cls.exclude_keys_after_conversion_to_mongo and \
                        cls.exclude_keys_after_conversion_to_mongo[key] == value:
                    continue

                key_parts = key.split('.')
                path_to_dict, dict_key = key_parts[:-1], key_parts[-1]

                cur_dict = mongo_dict

                for p in path_to_dict:
                    if p not in cur_dict:
                        cur_dict[p] = {}
                    cur_dict = cur_dict[p]

                cur_dict[dict_key] = value

        instance = cls()
        instance._mongo_dict = mongo_dict
        return instance
