# -*- coding: utf-8 -*-

from mpfs.dao.base import BaseDAO, BaseDAOItem, PostgresBaseDAOImplementation
from mpfs.dao.session import Session
from mpfs.dao.fields import JsonField, IntegerField, StringField, UidField, Md5Field
from mpfs.metastorage.postgres.schema import disk_info
from mpfs.metastorage.postgres.queries import SQL_DISK_INFO_INCEREMNT_DATA, SQL_DISK_INFO_INCEREMNT_DATA_BYTES, \
    SQL_SELECT_DISK_INFO_BY_UID_AND_PATH


class DiskInfoDAOItem(BaseDAOItem):
    is_sharded = True

    mongo_collection_name = 'disk_info'
    postgres_table_obj = disk_info
    id = Md5Field(mongo_path='_id', pg_path=disk_info.c.id)
    uid = UidField(mongo_path='uid', pg_path=disk_info.c.uid)
    parent = Md5Field(mongo_path='parent', pg_path=disk_info.c.parent, default_value=None)
    version = IntegerField(mongo_path='version', pg_path=disk_info.c.version, default_value=0)
    path = StringField(mongo_path='key', pg_path=disk_info.c.path, default_value=None)
    type = StringField(mongo_path='type', pg_path=disk_info.c.type, default_value=None)
    data = JsonField(mongo_path='data', pg_path=disk_info.c.data, default_value=None)

    exclude_keys_after_conversion_to_mongo = {
        'parent': None,
        'version': 0,
        'path': None,
        'type': None,
        'data': None,
    }

    @classmethod
    def get_postgres_primary_key(cls):
        return 'id'


class DiskInfoDAO(BaseDAO):
    dao_item_cls = DiskInfoDAOItem

    def __init__(self):
        super(DiskInfoDAO, self).__init__()
        self._pg_impl = PostgresDiskInfoDAOImplementation(self.dao_item_cls)

    def get_key(self, uid, key):
        return self._pg_impl.get_key(uid, key)


class PostgresDiskInfoDAOImplementation(PostgresBaseDAOImplementation):
    def get_key(self, uid, key):
        pg_uid = self.dao_item_cls.get_field_pg_representation('uid', uid)
        pg_path = self.dao_item_cls.get_field_pg_representation('path', key)
        session = Session.create_from_uid(uid)

        cursor = session.execute(SQL_SELECT_DISK_INFO_BY_UID_AND_PATH, {'uid': pg_uid,
                                                                        'path': pg_path})
        doc = cursor.fetchone()

        return self.doc_to_item(doc)

    def update(self, spec, document, upsert=False, multi=False, **kwargs):
        if document.keys() == ['$inc']:
            id_ = spec['_id']
            uid = int(spec['uid'])
            doc_inc_keys = document['$inc'].viewkeys()
            spec_keys = spec.viewkeys()

            if doc_inc_keys == {'data'} and spec_keys == {'_id', 'uid'}:
                value = int(document['$inc']['data'])
                session = Session.create_from_uid(uid)
                with session.begin():
                    result = session.execute(SQL_DISK_INFO_INCEREMNT_DATA, {'id': id_, 'uid': uid, 'value': value})
                return self._pg_update_result_to_mongo(result, upsert)

            if doc_inc_keys == {'data.bytes'} and spec_keys == {'_id', 'uid'}:
                value = int(document['$inc']['data.bytes'])
                session = Session.create_from_uid(uid)
                with session.begin():
                    result = session.execute(SQL_DISK_INFO_INCEREMNT_DATA_BYTES, {'id': id_, 'uid': uid, 'value': value})
                return self._pg_update_result_to_mongo(result)

        return super(PostgresDiskInfoDAOImplementation, self).update(spec, document, upsert, multi, **kwargs)
