# -*- coding: utf-8 -*-
import itertools
import os
import uuid

from collections import OrderedDict
from mpfs.config import settings
from mpfs.engine.process import get_default_log
from mpfs.common.util import to_json
from mpfs.core.filesystem.dao.common import CustomSetpropFieldsMixin, OptimizedResourceConvertionMixin
from mpfs.dao.base import (
    BaseDAOItem,
    FakeColumn,
    BaseDAO,
    PostgresBaseDAOImplementation,
)
from mpfs.dao.fields import (IntegerAsBoolField, DateTimeField, IntegerField, HidField, MediaTypeField, StringField,
                             UidField, AntiVirusStatusField, JsonField, FidField, Md5Field, FileIdField, Sha256Field,
                             ByteStringField, StidField, PathField, BoolField, RealField, EnumField, StringArrayField)
from mpfs.dao.session import Session
from mpfs.metastorage.postgres.schema import (
    files,
    storage_files,
    AntiVirusScanStatus,
    PhotosliceAlbumType,
    OfficeActionState,
)
from mpfs.metastorage.postgres.queries import (
    SQL_GET_FILE_STID_BY_STORAGE_ID,
    SQL_UPDATE_PREVIEW,
    SQL_FILE_BY_FILE_ID_WITH_OLDEST_VERSION_NOT_REMOVED,
    SQL_DOES_FILE_EXIST_BY_UID_HID_LIVE_PHOTO,
    SQL_DOES_FILE_EXIST_BY_UID_HID,
    SQL_REMOVE_HANGING_STORAGE_FILES, SQL_FILES_UPDATE_OFFICE_FIELDS,
    SQL_FILES_BY_UID_HID,
)


SERVICES_MULCA_EMPTY_FILE_HID = settings.services['mulca']['empty_file_hid']


class StidsListParser(object):
    """
    Класс для доставания стида по типу из монгового списка стидов и формирования такого списка.
    В монге список стидов выглядит так:
    [
        {'type': 'file_mid', 'stid': '<value_1>'},
        {'type': 'pmid', 'stid': '<value_2>'},
        {'type': 'digest_mid', 'stid': '<value_3>'}
    ]
    """

    def __init__(self, stid_type, is_optional=False):
        self.stid_type = stid_type
        self.is_optional = is_optional

    def parse(self, stids):
        """Костыль для доставания нужного стида из списка.
        """
        for s in stids:
            if s['type'] == self.stid_type:
                return s['stid']

        if self.is_optional:
            return None

        raise KeyError('stid of type `%s` not found' % self.stid_type)

    def format(self, stid):
        """Костыль для формирования списка из стида.
        """
        if self.is_optional and stid is None:
            return []

        return [{
            'type': self.stid_type,
            'stid': stid
        }]


class FileDAOItem(CustomSetpropFieldsMixin, OptimizedResourceConvertionMixin, BaseDAOItem):
    """Структура, описывающая данные файла вне зависимости от БД. Содержит только значения, не содержит логику.
    """

    fid = FidField(mongo_path='_id', pg_path=files.c.fid)
    parent_fid = FidField(mongo_path='parent', pg_path=files.c.parent_fid, default_value=None)
    path = PathField(mongo_path='key', pg_path=FakeColumn('path'))  # в постгресе явно возвращаем параметр из запроса в path
    uid = UidField(mongo_path='uid', pg_path=files.c.uid)
    modify_uid = UidField(mongo_path='data.modify_uid', pg_path=files.c.modify_uid, default_value=None)

    hid = HidField(mongo_path='hid', pg_path=storage_files.c.storage_id)
    size = IntegerField(mongo_path='data.size', pg_path=storage_files.c.size)
    md5 = Md5Field(mongo_path='zdata.meta.md5', pg_path=storage_files.c.md5_sum)
    sha256 = Sha256Field(mongo_path='zdata.meta.sha256', pg_path=storage_files.c.sha256_sum)

    file_id = FileIdField(mongo_path='data.file_id', pg_path=files.c.id, default_value=None)
    version = IntegerField(mongo_path='version', pg_path=files.c.version)

    file_stid = StidField(mongo_path='data.stids', mongo_item_parser=StidsListParser('file_mid'),
                          pg_path=storage_files.c.stid)
    preview_stid = StidField(mongo_path='data.stids', mongo_item_parser=StidsListParser('pmid', is_optional=True),
                             pg_path=storage_files.c.preview_stid, default_value=None)
    digest_stid = StidField(mongo_path='data.stids', mongo_item_parser=StidsListParser('digest_mid'),
                            pg_path=storage_files.c.digest_stid)

    upload_time = DateTimeField(mongo_path='data.utime', pg_path=files.c.date_uploaded)
    creation_time = DateTimeField(mongo_path='zdata.meta.ctime', pg_path=files.c.date_created)
    modification_time = DateTimeField(mongo_path='data.mtime', pg_path=files.c.date_modified)
    exif_time = DateTimeField(mongo_path='data.etime', pg_path=files.c.date_exif, default_value=None)
    trash_append_time = DateTimeField(mongo_path='zdata.setprop.append_time', pg_path=files.c.date_removed,
                                      default_value=None)
    trash_clean_time = DateTimeField(mongo_path='dtime', pg_path=files.c.date_hidden_data,
                                     default_value=None)

    original_path = PathField(mongo_path='data.original_id', pg_path=files.c.path_before_remove,
                              default_value=None)

    is_visible = IntegerAsBoolField(mongo_path='data.visible', pg_path=files.c.visible)
    antivirus_status = AntiVirusStatusField(mongo_path='zdata.meta.drweb', pg_path=storage_files.c.av_scan_status,
                                            default_value=AntiVirusScanStatus.clean)
    created_from = StringField(mongo_path='zdata.meta.source', pg_path=files.c.source, default_value=None)

    mimetype = StringField(mongo_path='data.mimetype', pg_path=files.c.mime_type, default_value=None)
    mediatype = MediaTypeField(mongo_path='data.mt', pg_path=files.c.media_type, default_value=None)

    is_public = IntegerAsBoolField(mongo_path='data.public', pg_path=files.c.public, default_value=False)
    is_blocked = IntegerAsBoolField(mongo_path='data.blocked', pg_path=files.c.blocked, default_value=False)
    was_published = IntegerAsBoolField(mongo_path='zdata.setprop.published', pg_path=files.c.published,
                                       default_value=False)
    public_hash = ByteStringField(mongo_path='zdata.pub.public_hash', pg_path=files.c.public_hash, default_value=None)
    short_url = StringField(mongo_path='zdata.pub.short_url', pg_path=files.c.short_url, default_value=None)
    symlink = StringField(mongo_path='zdata.pub.symlink', pg_path=files.c.symlink, default_value=None)
    office_access_state = EnumField(mongo_path='data.office_access_state',
                                    pg_path=files.c.office_access_state,
                                    default_value=None,
                                    enum_class=OfficeActionState)
    office_doc_short_id = StringField(mongo_path='data.office_doc_short_id',
                                      pg_path=files.c.office_doc_short_id,
                                      default_value=None)
    video_info = JsonField(mongo_path='zdata.setprop.video_info', pg_path=storage_files.c.video_data,
                           default_value=None)
    folder_url = StringField(mongo_path='zdata.setprop.folder_url', pg_path=files.c.folder_url,
                             default_value=None)

    download_counter = IntegerField(mongo_path='data.download_counter', pg_path=files.c.download_counter,
                                    default_value=None)
    custom_properties = StringField(mongo_path='zdata.setprop.custom_properties', pg_path=files.c.custom_properties,
                                    default_value=None)
    source_uid = UidField(mongo_path='zdata.setprop.source_uid', pg_path=files.c.source_uid,
                          default_value=None)

    custom_setprop_fields = JsonField(mongo_path='custom_setprop_fields',
                                      pg_path=files.c.custom_setprop_fields, default_value=None)

    width = IntegerField(mongo_path='data.width', pg_path=storage_files.c.width, default_value=None)
    height = IntegerField(mongo_path='data.height', pg_path=storage_files.c.height, default_value=None)
    angle = IntegerField(mongo_path='data.angle', pg_path=storage_files.c.angle, default_value=None)

    is_live_photo = BoolField(mongo_path='data.is_live_photo', pg_path=files.c.is_live_photo, default_value=None)
    yarovaya_mark = BoolField(mongo_path='data.yarovaya_mark', pg_path=files.c.yarovaya_mark, default_value=None)

    aesthetics = RealField(mongo_path='data.aesthetics', pg_path=files.c.ext_aesthetics, default_value=None)
    photoslice_album_type = EnumField(mongo_path='data.photoslice_album_type', pg_path=files.c.photoslice_album_type, default_value=None, enum_class=PhotosliceAlbumType)
    albums_exclusions = StringArrayField(mongo_path='data.albums_exclusions',
                                         pg_path=files.c.albums_exclusions,
                                         default_value=None)
    ext_coordinates = StringField(mongo_path='data.ext_coordinates', pg_path=files.c.ext_coordinates, default_value=None)
    exclude_keys_after_conversion_to_mongo = {
        'dtime': None,
        'data.public': 0,
        'data.original_id': None,
        'data.modify_uid': None,
        'data.file_id': None,
        'data.utime': None,
        'data.mtime': None,
        'data.etime': None,
        'data.download_counter': None,
        'data.blocked': 0,
        'zdata.meta.ctime': None,
        'zdata.meta.source': None,
        'zdata.pub.public_hash': None,
        'zdata.pub.symlink': None,
        'zdata.pub.short_url': None,
        'zdata.setprop.append_time': None,
        'zdata.setprop.folder_url': None,
        'zdata.setprop.custom_properties': None,
        'zdata.setprop.source_uid': None,
        'zdata.setprop.published': False,
        'zdata.setprop.video_info': None,
        'data.width': None,
        'data.height': None,
        'data.angle': None,
        'data.is_live_photo': None,
        'data.yarovaya_mark': None,
        'data.aesthetics': None,
        'data.albums_exclusions': None,
        'data.ext_coordinates': None,
    }

    validation_ignored_mongo_dict_fields = ('type', 'path', 'zdata.setprop.id', 'zdata.setprop.original_parent_id',
                                            'zdata.setprop.total_results_count')

    postgres_table_obj = files

    columns_map = {c.name: c for c in itertools.chain(files.columns, storage_files.columns)}
    columns_map['path'] = FakeColumn('path')

    @classmethod
    def convert_mongo_key_to_postgres(cls, key):
        # костыль, переопределено для сортировки, если передан параметр name (а в монге его нет), то мы его не найдем
        # в этом классе. Но! в постгресе этот параметр есть, поэтому можно по нему сортировать и отдаем его как есть
        if key == 'name':
            return 'name'
        return super(FileDAOItem, cls).convert_mongo_key_to_postgres(key)

    def get_mongo_representation(self, skip_missing_fields=False):
        # костыль, чтобы в словаре возвращался еще и тип, в BaseDAOItem это сделать нельзя, там общая логика
        mongo_dict = super(FileDAOItem, self).get_mongo_representation(skip_missing_fields)
        mongo_dict['type'] = 'file'
        return mongo_dict

    def is_storage_part_equal(self, other):
        storage_file_field_names = [name for name, field in self._fields.iteritems()
                                    if field.pg_path.name in storage_files.columns]
        for name in storage_file_field_names:
            self_value = getattr(self, name, None)
            other_value = getattr(other, name, None)
            if self_value != other_value:
                return False
        return True


class FileDAO(BaseDAO):
    """
    Мета-DAO для файлов. Работает с абстракцией файла и двумя таблицами - files и storage_files.
    """
    dao_item_cls = FileDAOItem

    def __init__(self, session=None):
        super(FileDAO, self).__init__(session)
        self._pg_impl = PostgresFileDAOImplementation(self.dao_item_cls)

    def create(self, file_item):
        """
        Вставка объекта файла в постгрес.
        Вставляет в одной транзацкции в две таблицы. Если такой файл в storage_files уже есть, то не обновляет его, а
        делает вставку только в files.
        :type file_item: FileDAOItem
        """
        self.check_hid(file_item)

        pg_dict = file_item.get_postgres_representation(skip_missing_fields=True)

        metadata_files_values = OrderedDict()
        storage_files_values = OrderedDict()
        path = pg_dict.pop(FakeColumn('path'))  # избавляемся от path, его в постгрес все равно не кладем
        metadata_files_values['name'] = path.rsplit('/', 1)[1]  # вместо этого кладем name - формируем его из path
        for coll, val in pg_dict.iteritems():
            if coll.table == files:
                metadata_files_values[coll.name] = val
            else:
                storage_files_values[coll.name] = val

        # добавляем для линковки записей между собой
        metadata_files_values['storage_id'] = storage_files_values['storage_id']

        storage_files_query = 'INSERT INTO disk.storage_files (%s) VALUES (%s) ON CONFLICT DO NOTHING' % \
                              (','.join(storage_files_values.iterkeys()),
                               ','.join([':' + name for name in storage_files_values.iterkeys()]))
        metadata_files_query = 'INSERT INTO disk.files (%s) VALUES (%s)' % \
                               (','.join(metadata_files_values.iterkeys()),
                                ','.join([':' + name for name in metadata_files_values.iterkeys()]))

        if self.session is not None:
            storage_file_insert_result = self.session.execute(storage_files_query, storage_files_values)
            self.session.execute(metadata_files_query, metadata_files_values)
        else:
            session = Session.create_from_uid(file_item.uid)
            with session.begin():
                storage_file_insert_result = session.execute(storage_files_query, storage_files_values)
                session.execute(metadata_files_query, metadata_files_values)

        if storage_file_insert_result.rowcount == 0:
            storage_id = storage_files_values['storage_id']
            file_stid = storage_files_values['stid']
            result_proxy = self.session.execute(SQL_GET_FILE_STID_BY_STORAGE_ID, {'storage_id': storage_id})
            db_stid = result_proxy.fetchone()[0]
            if db_stid != file_stid:
                storage_file_log_fields = to_json({
                    'hid': file_item.hid,
                    'preview_stid': file_item.preview_stid,
                    'file_stid': file_item.file_stid,
                    'digest_stid': file_item.digest_stid,
                    'size': file_item.size,
                    'md5': file_item.md5,
                    'sha256': file_item.sha256
                })
                get_default_log().info("Find extra stid. storage_id: %s, db_stid: %s, new_storage_file: %s.", storage_id, db_stid, storage_file_log_fields)

    def update_storage_part(self, file_item):
        pg_dict = file_item.get_postgres_representation(skip_missing_fields=True)

        pg_dict.pop(FakeColumn('path'))
        storage_files_values = OrderedDict()
        for coll, val in pg_dict.iteritems():
            if coll.table == storage_files:
                storage_files_values[coll.name] = val

        storage_files_query = 'UPDATE disk.storage_files SET %s WHERE storage_id=:storage_id' % \
                              ','.join([name + '=:' + name for name in storage_files_values.iterkeys()])

        if self.session is not None:
            self.session.execute(storage_files_query, storage_files_values)
        else:
            session = Session.create_from_uid(file_item.uid)
            session.execute(storage_files_query, storage_files_values)

    def remove_hanging_storage_files(self, storage_ids, exclude_empty_file=True):
        if self.session is None:
            raise ValueError('Session is not specified')

        if not storage_ids:
            return

        if not isinstance(storage_ids, (tuple, list)):
            storage_ids = tuple(storage_ids)

        empty_file_hid = SERVICES_MULCA_EMPTY_FILE_HID
        if isinstance(storage_ids[0], uuid.UUID):
            empty_file_hid = uuid.UUID(empty_file_hid)

        if exclude_empty_file and empty_file_hid in storage_ids:
            # фильтруем пустой файл здесь из-за неоптимального query plan'а на postgres'е:
            # https://st.yandex-team.ru/CHEMODAN-63556
            storage_ids = tuple(hid for hid in storage_ids if hid != empty_file_hid)
        else:
            storage_ids = tuple(storage_ids)

        if not storage_ids:
            return

        return self.session.execute(SQL_REMOVE_HANGING_STORAGE_FILES, {'storage_ids': storage_ids})

    @staticmethod
    def check_hid(file_item):
        from mpfs.core.filesystem.hardlinks.common import construct_hid
        calculated_hid = construct_hid(file_item.md5, file_item.size, file_item.sha256)
        if file_item.hid != calculated_hid:
            raise ValueError('File hid `%s` for path `%s` is not equal to calculated hash from md5, sha256 and size' %
                             (file_item.hid, file_item.path))

    def remove_all_by_uid(self, uid):
        session = self.session
        if session is None:
            session = Session.create_from_uid(uid)

        with session.begin():
            session.execute('DELETE FROM disk.additional_file_links WHERE uid=:uid', {'uid': uid})
            session.execute('DELETE FROM disk.files WHERE uid=:uid', {'uid': uid})

    def update_preview_on_all_pg_shards(self, old_preview_stid, regenerate_preview_result):
        """
        Функция апдейтящая stid превью и атрибуты возращаемые кладуном при перегенерации превью на всех шардах PG

        :param old_preview_stid: старый превью stid
        :param regenerate_preview_result: результат перегенерации превью из кладуна, инстанс RegeneratePreviewResult
        :return: boolean, заменили ли превью хотябы на одном шарде
        """

        sessions = Session.create_for_all_shards()

        query_params = {
            'old_preview_stid': FileDAOItem._fields['preview_stid'].to_postgres(old_preview_stid)
        }

        pg_data = {
            'preview_stid': FileDAOItem._fields['preview_stid'].to_postgres(regenerate_preview_result.pmid)
        }

        if regenerate_preview_result.original_height is not None and regenerate_preview_result.original_width is not None:
            pg_data['width'] = FileDAOItem._fields['width'].to_postgres(regenerate_preview_result.original_width)
            pg_data['height'] = FileDAOItem._fields['height'].to_postgres(regenerate_preview_result.original_height)

        if regenerate_preview_result.has_video_info():
            pg_data['video_data'] = FileDAOItem._fields['video_info'].to_postgres(regenerate_preview_result.video_info)

        if regenerate_preview_result.rotate_angle is not None:
            pg_data['angle'] = FileDAOItem._fields['angle'].to_postgres(regenerate_preview_result.rotate_angle)

        fields_for_update_placeholder = ','.join('%s=:%s' % (f, f) for f in pg_data.iterkeys())

        query = SQL_UPDATE_PREVIEW % fields_for_update_placeholder
        query_params.update(pg_data)

        updated = False
        for session in sessions:
            cursor = session.execute(query, query_params)
            if not updated:
                updated = cursor.fetchone() is not None
            session.close()

        return updated

    def find_by_file_id_with_oldest_version_not_removed(self, uid, file_id):
        session = self.get_session(uid)

        result = session.execute(SQL_FILE_BY_FILE_ID_WITH_OLDEST_VERSION_NOT_REMOVED, {
            'uid': FileDAOItem.convert_mongo_value_to_postgres_for_key('uid', uid)[1],
            'file_id': FileDAOItem.convert_mongo_value_to_postgres_for_key('data.file_id', file_id)[1],
        })
        data = result.fetchone()

        if data:
            return FileDAOItem.create_from_pg_data(data)
        return None

    def does_exist_by_uid_hid(self, uid, hid, is_live_photo=False):
        return self._get_impl_by_uid(uid).does_exist_by_uid_hid(uid, hid, is_live_photo=is_live_photo)

    def update_office_fields_from_link_data(self, uid, file_id):
        pg_file_id = FileDAOItem.convert_mongo_value_to_postgres_for_key('data.file_id', file_id)[1]
        pg_uid = FileDAOItem.convert_mongo_value_to_postgres_for_key('uid', uid)[1]

        session = Session.create_from_uid(uid)
        session.execute(SQL_FILES_UPDATE_OFFICE_FIELDS, {'uid': pg_uid, 'file_id': pg_file_id})

    def find_file_in_folder_by_hid(self, uid, parent_paths, hid):
        session = self.get_session(uid)

        for parent_path in parent_paths:
            params = {
                'uid': FileDAOItem.convert_mongo_value_to_postgres_for_key('uid', uid)[1],
                'root_folder': parent_path,
                'hid': FileDAOItem.convert_mongo_value_to_postgres_for_key('hid', hid)[1]
            }

            data = session.execute(SQL_FILES_BY_UID_HID, params).fetchone()
            if data:
                result = FileDAOItem.create_from_pg_data(data)
                return parent_path, result
        return None, None


class PostgresFileDAOImplementation(PostgresBaseDAOImplementation):

    def does_exist_by_uid_hid(self, uid, hid, is_live_photo=False):
        params = {
            'uid': self.get_field_repr('uid', uid),
            'storage_id': self.get_field_repr('hid', hid),
            'is_live_photo': is_live_photo,
        }
        session = Session.create_from_uid(params['uid'])
        query = (SQL_DOES_FILE_EXIST_BY_UID_HID_LIVE_PHOTO if is_live_photo
                 else SQL_DOES_FILE_EXIST_BY_UID_HID)
        return session.execute(query, params).fetchone() is not None
