# -*- coding: utf-8 -*-

from mpfs.dao.base import BaseDAO, BaseDAOItem, MongoBaseDAOImplementation, PostgresBaseDAOImplementation, \
    QueryWithParams
from mpfs.dao.session import Session
from mpfs.dao.cursor import PostgresCursor
from mpfs.dao.fields import UTCDateTimeField, JsonField, Md5Field, StringField, UidField
from mpfs.metastorage.postgres.schema import filesystem_locks
from mpfs.metastorage.postgres.queries import SQL_FILESYSTEM_LOCK_UPDATE, SQL_FILESYSTEM_LOCK_WITH_DATA_UPDATE


class FilesystemLockDAOItem(BaseDAOItem):
    mongo_collection_name = 'filesystem_locks'
    postgres_table_obj = filesystem_locks
    is_sharded = True

    id = Md5Field(mongo_path='_id', pg_path=filesystem_locks.c.id)
    uid = UidField(mongo_path='uid', pg_path=filesystem_locks.c.uid)
    path = StringField(mongo_path='key', pg_path=filesystem_locks.c.path)
    dtime = UTCDateTimeField(mongo_path='dtime', pg_path=filesystem_locks.c.dtime, default_value=None)
    data = JsonField(mongo_path='data', pg_path=filesystem_locks.c.data, default_value=None)

    @classmethod
    def get_postgres_primary_key(cls):
        return 'id'


class FilesystemLockDAO(BaseDAO):
    dao_item_cls = FilesystemLockDAOItem

    def __init__(self):
        super(FilesystemLockDAO, self).__init__()
        self._mongo_impl = MongoFilesystemLockDAOImplementation(self.dao_item_cls)
        self._pg_impl = PostgresFilesystemLockDAOImplementation(self.dao_item_cls)

    def find_and_modify(self, find, modify, fsync=False, w=1):
        impl = self._get_impl(find)
        return impl.find_and_modify(find, modify, fsync=fsync, w=w)


class MongoFilesystemLockDAOImplementation(MongoBaseDAOImplementation):
    def find_and_modify(self, find, modify, fsync=False, w=1):
        return self._mongo_helper.get_collection(self.dao_item_cls.mongo_collection_name).find_and_modify(find, modify, fsync=fsync, w=w)


class PostgresFilesystemLockDAOImplementation(PostgresBaseDAOImplementation):
    def find_and_modify(self, find, modify, fsync=False, w=1):
        if (set(find.keys()) == {'_id', 'uid', 'key'} and
                modify.keys() == ['$set'] and modify['$set'].keys() == ['dtime']):
            uid = int(find['uid'])
            path = find['key']
            _, id_ = FilesystemLockDAOItem.convert_mongo_value_to_postgres_for_key('_id', find['_id'])
            _, dtime = FilesystemLockDAOItem.convert_mongo_value_to_postgres_for_key('dtime', modify['$set']['dtime'])

            session = Session.create_from_uid(uid)
            query = QueryWithParams(SQL_FILESYSTEM_LOCK_UPDATE, {'id': id_, 'uid': uid, 'path': path, 'dtime': dtime})
            return list(PostgresCursor(session, query, self.dao_item_cls))
        elif (set(find.keys()) == {'_id', 'uid', 'key'} and
                modify.keys() == ['$set'] and set(modify['$set'].keys()) == {'dtime', 'data'}):
            uid = int(find['uid'])
            path = find['key']
            _, data = FilesystemLockDAOItem.convert_mongo_value_to_postgres_for_key('data', modify['$set']['data'])
            _, id_ = FilesystemLockDAOItem.convert_mongo_value_to_postgres_for_key('_id', find['_id'])
            _, dtime = FilesystemLockDAOItem.convert_mongo_value_to_postgres_for_key('dtime', modify['$set']['dtime'])

            session = Session.create_from_uid(uid)
            query = QueryWithParams(SQL_FILESYSTEM_LOCK_WITH_DATA_UPDATE,
                                    {'id': id_, 'uid': uid, 'path': path, 'dtime': dtime, 'data': data})
            return list(PostgresCursor(session, query, self.dao_item_cls))
        raise NotImplementedError('Not implemented for find "%s" and modify "%s"' % (find, modify))
