# -*- coding: utf-8 -*-
from collections import OrderedDict

from mpfs.common.util import hashed
from mpfs.config import settings
from mpfs.core.address import Address
from mpfs.core.filesystem.dao.common import CustomSetpropFieldsMixin, OptimizedResourceConvertionMixin
from mpfs.core.filesystem.resources.base import Resource
from mpfs.core.user.constants import PHOTOUNLIM_AREA_PATH
from mpfs.dao.base import BaseDAOItem, BaseDAO, FakeColumn, DAOPath, BaseDAOImplementation, QueryWithParams, MongoHelper
from mpfs.dao.cursor import PostgresCursor
from mpfs.dao.session import Session
from mpfs.dao.fields import (IntegerAsBoolField, DateTimeField, IntegerField, StringField, UidField, FidField,
                             FileIdField, ByteStringField, JsonField, PathField, BoolField)
from mpfs.metastorage.mongo.util import parent_for_key, name_for_key
from mpfs.metastorage.postgres.queries import (SQL_UPDATE_FOLDER_NAME_AND_VERSION,
                                               SQL_UPDATE_FOLDER_PARENT_NAME_AND_VERSION,
                                               SQL_HAS_PARENTS_WITH_YAROVAYA_MARK_BY_PATH,
                                               SQL_UPDATE_FOLDER_PARENT_NAME_AND_VERSION_WITH_YAROVAYA_MARK_SET,
                                               SQL_FOLDER_BY_FILE_ID_WITH_OLDEST_VERSION_NOT_REMOVED,
                                               SQL_COUNT_RESOURCES_BY_UID,
                                               SQL_COUNT_SUBFOLDERS_BY_PATH, SQL_COUNT_SUBFILES_BY_PATH,
                                               SQL_GET_ANOTHER_MEDIA_TYPE_FILE_FROM_FOLDER,
                                               SQL_FOLDERS_BY_UID_DETERMINED,
                                               SQL_FILES_BY_UID_DETERMINED)
from mpfs.metastorage.postgres.schema import folders
from mpfs.metastorage.postgres.query_executer import ReadPreference as PGReadPreference

FEATURE_TOGGLES_SETTING_YAROVAYA_MARK_ENABLED = settings.feature_toggles['setting_yarovaya_mark_enabled']


class FolderDAOItem(CustomSetpropFieldsMixin, OptimizedResourceConvertionMixin, BaseDAOItem):
    """Структура, описывающая данные папки вне зависимости от БД.
    Содержит только значения,не содержит логику.
    """
    fid = FidField(mongo_path='_id', pg_path=folders.c.fid)
    parent_fid = FidField(mongo_path='parent', pg_path=folders.c.parent_fid, default_value=None)
    path = PathField(mongo_path='key', pg_path='path')  # в постгресе явно возвращаем параметр из запроса в path
    uid = UidField(mongo_path='uid', pg_path=folders.c.uid)
    modify_uid = UidField(mongo_path='data.modify_uid', pg_path=folders.c.modify_uid, default_value=None)

    file_id = FileIdField(mongo_path='data.file_id', pg_path=folders.c.id, default_value=None)
    version = IntegerField(mongo_path='version', pg_path=folders.c.version)

    creation_time = DateTimeField(mongo_path='zdata.meta.ctime', pg_path=folders.c.date_created, default_value=None)
    upload_time = DateTimeField(mongo_path='data.utime', pg_path=folders.c.date_uploaded, default_value=None)
    modification_time = DateTimeField(mongo_path='data.mtime', pg_path=folders.c.date_modified, default_value=None)
    trash_append_time = DateTimeField(mongo_path='zdata.setprop.append_time', pg_path=folders.c.date_removed,
                                      default_value=None)
    trash_clean_time = DateTimeField(mongo_path='dtime', pg_path=folders.c.date_hidden_data,
                                     default_value=None)

    original_path = PathField(mongo_path='data.original_id', pg_path=folders.c.path_before_remove,
                              default_value=None)

    is_visible = IntegerAsBoolField(mongo_path='data.visible', pg_path=folders.c.visible, default_value=True)
    is_public = IntegerAsBoolField(mongo_path='data.public', pg_path=folders.c.public, default_value=False)
    is_blocked = IntegerAsBoolField(mongo_path='data.blocked', pg_path=folders.c.blocked, default_value=False)
    was_published = IntegerAsBoolField(mongo_path='zdata.setprop.published', pg_path=folders.c.published,
                                       default_value=False)

    public_hash = ByteStringField(mongo_path='zdata.pub.public_hash', pg_path=folders.c.public_hash, default_value=None)
    short_url = StringField(mongo_path='zdata.pub.short_url', pg_path=folders.c.short_url, default_value=None)
    symlink = StringField(mongo_path='zdata.pub.symlink', pg_path=folders.c.symlink, default_value=None)
    folder_type = StringField(mongo_path='zdata.setprop.folder_type', pg_path=folders.c.folder_type, default_value=None)

    folder_url = StringField(mongo_path='zdata.setprop.folder_url', pg_path=folders.c.folder_url,
                             default_value=None)
    download_counter = IntegerField(mongo_path='data.download_counter', pg_path=folders.c.download_counter,
                                    default_value=None)
    custom_properties = StringField(mongo_path='zdata.setprop.custom_properties', pg_path=folders.c.custom_properties,
                                    default_value=None)

    yarovaya_mark = BoolField(mongo_path='data.yarovaya_mark', pg_path=folders.c.yarovaya_mark, default_value=None)

    custom_setprop_fields = JsonField(mongo_path='custom_setprop_fields',
                                      pg_path=folders.c.custom_setprop_fields, default_value=None)

    exclude_keys_after_conversion_to_mongo = {
        'parent': None,
        'dtime': None,
        'version': 0,
        # если ты хочешь раскомментить это поле - 100 раз подумай, ибо тогда в diff'ах у тебя будет у всех папок
        # visible: 0 и это сломает сохранение в офлайн всем мобильным клиентам:
        # 'data.visible': 1,
        'data.public': 0,
        'data.original_id': None,
        'data.modify_uid': None,
        'data.file_id': None,
        'data.utime': None,
        'data.mtime': None,
        'data.download_counter': None,
        'data.blocked': 0,
        'data.yarovaya_mark': None,
        'zdata.meta.ctime': None,
        'zdata.pub.public_hash': None,
        'zdata.pub.symlink': None,
        'zdata.pub.short_url': None,
        'zdata.setprop.folder_type': None,
        'zdata.setprop.append_time': None,
        'zdata.setprop.folder_url': None,
        'zdata.setprop.custom_properties': None,
        'zdata.setprop.published': False,
    }

    validation_ignored_mongo_dict_fields = ('type', 'path', 'zdata.setprop.id', 'zdata.setprop.original_parent_id',
                                            'zdata.setprop.total_results_count')
    mongo_collection_name = 'user_data'
    postgres_table_obj = folders

    columns_map = {c.name: c for c in folders.columns}
    columns_map['path'] = FakeColumn('path')

    def get_mongo_representation(self, skip_missing_fields=False):
        # костыль, чтобы в словаре возвращался еще и тип, в BaseDAOItem это сделать нельзя, там общая логика
        mongo_dict = super(FolderDAOItem, self).get_mongo_representation(skip_missing_fields)
        mongo_dict['type'] = 'dir'

        if 'data' not in mongo_dict:
            mongo_dict['data'] = {}  # обратная совместимость со старым кодом

        # костыль, чтобы фотоанлимная папка корректно доставалась с resource_id https://st.yandex-team.ru/CHEMODAN-38844
        if mongo_dict.get('key', None) == PHOTOUNLIM_AREA_PATH and not mongo_dict.get('zdata'):
            mongo_dict['zdata'] = {}

        path = mongo_dict.get('key', '')
        if self._is_root_or_root_child(path):
            mongo_dict.get('data', {}).pop('visible')  # костыль для обратной миграции, чтобы не тащить то, чего не было

        return mongo_dict

    def get_postgres_representation(self, skip_missing_fields=False):
        pg_data = super(FolderDAOItem, self).get_postgres_representation(skip_missing_fields)

        # костыль, чтобы генерировать file_id у корневых папок для постгреса, потому что там есть констреинт на file_id
        path = pg_data.get('path', '')
        if pg_data[folders.c.id] is None and (self._is_root_or_root_child(path) or self._is_folder_in_hidden(path)):
            # если у папки нет file_id и это корневая папка / или ее ребенок, например /disk, /hidden и т.д.,
            generated_file_id = Resource.generate_file_id(str(pg_data[folders.c.uid]), path)
            _, pg_file_id = self.convert_mongo_value_to_postgres_for_key('data.file_id', generated_file_id)
            pg_data[folders.c.id] = pg_file_id

        return pg_data

    @classmethod
    def convert_mongo_key_to_postgres(cls, key):
        # костыль, переопределено для сортировки, если передан параметр name (а в монге его нет), то мы его не найдем
        # в этом классе. Но! в постгресе этот параметр есть, поэтому можно по нему сортировать и отдаем его как есть
        if key == 'name':
            return 'name'
        return super(FolderDAOItem, cls).convert_mongo_key_to_postgres(key)

    @staticmethod
    def _is_root_or_root_child(path):
        return path == '/' or '/' not in path[1:]

    @staticmethod
    def _is_folder_in_hidden(path):
        return path.startswith('/hidden/')


class FolderDAO(BaseDAO):
    """
    Базовое DAO для таблицы folders
    """
    dao_item_cls = FolderDAOItem
    mongo_coll_name = 'user_data'

    def __init__(self, session=None):
        super(FolderDAO, self).__init__(session)
        self._mongo_impl = MongoFolderDAOImplementation(self.mongo_coll_name)
        self._pg_impl = PostgresFolderDAOImplementation(self.mongo_coll_name)

    def create(self, folder_item):
        """
        Вставка объекта папки в постгрес.
        Пока что без батчей, потому что есть папки с разными наборами входных данных (из-за чего пользуемся флагом
        skip_missing_fields в функции get_postgres_representation. Мы можем их группировать, но пока в этом нет нужды.

        :type folder_item: FolderDAOItem
        """
        pg_dict = folder_item.get_postgres_representation(skip_missing_fields=True)
        values = self._get_postgres_insert_values(pg_dict)

        query = 'INSERT INTO disk.folders (%s) VALUES (%s) RETURNING fid' % (
            ','.join(values.iterkeys()),
            ','.join([':' + name for name in values.iterkeys()])
        )

        if self.session is not None:
            item = self.session.execute(query, values).fetchone()
            return item[folders.c.fid]
        else:
            session = Session.create_from_uid(folder_item.uid)
            with session.begin():
                item = session.execute(query, values).fetchone()
                return item[folders.c.fid]

    def update_item_values(self, folder_item):
        pg_dict = folder_item.get_postgres_representation(skip_missing_fields=True)
        values = self._get_postgres_insert_values(pg_dict)

        query = 'UPDATE disk.folders SET %s WHERE uid=:uid AND fid=:fid' % \
            ','.join([name + '=:' + name for name in values.iterkeys()])

        if self.session is not None:
            self.session.execute(query, values)
        else:
            session = Session.create_from_uid(folder_item.uid)
            session.execute(query, values)

    def remove_all_by_uid(self, uid):
        session = self.session
        if session is None:
            session = Session.create_from_uid(uid)

        with session.begin():
            session.execute('DELETE FROM disk.folders WHERE uid=:uid', {'uid': uid})

    def move(self, uid, src_path, dst_path, new_version, force_having_yarovaya_mark=False):
        if not isinstance(src_path, DAOPath) or not isinstance(dst_path, DAOPath):
            raise TypeError('src_path or dst_path is not DAOPath type')

        session = self.session
        if session is None:
            session = Session.create_from_uid(uid)

        if src_path.get_parent_path() == dst_path.get_parent_path():
            session.execute(
                SQL_UPDATE_FOLDER_NAME_AND_VERSION,
                {'uid': uid, 'path': src_path.get_value(), 'new_name': dst_path.get_name(), 'version': new_version}
            )
        else:
            pg_args = {'uid': uid, 'src_path': src_path.get_value(),
                       'target_parent_path': dst_path.get_parent_path().get_value(),
                       'target_name': dst_path.get_name(), 'version': new_version}
            if FEATURE_TOGGLES_SETTING_YAROVAYA_MARK_ENABLED and force_having_yarovaya_mark:
                query = SQL_UPDATE_FOLDER_PARENT_NAME_AND_VERSION_WITH_YAROVAYA_MARK_SET
            else:
                query = SQL_UPDATE_FOLDER_PARENT_NAME_AND_VERSION
            session.execute(query, pg_args)

    def has_parents_with_yarovaya_mark(self, address):
        impl = self._get_impl_by_uid(address.uid)
        return impl.has_parents_with_yarovaya_mark(address)

    def count_subfolders(self, address):
        return self._get_impl_by_uid(address.uid).count_subfolders(address)

    def count_subfiles(self, address):
        return self._get_impl_by_uid(address.uid).count_subfiles(address)

    def has_another_media_type_files(self, address, media_types):
        return self._get_impl_by_uid(address.uid).has_another_media_type_files(address, media_types)

    def _get_postgres_insert_values(self, pg_dict):
        values = OrderedDict()
        path = pg_dict.pop(FakeColumn('path'), None)  # избавляемся от path, его в постгрес все равно не кладем
        values['name'] = path.rsplit('/', 1)[1]  # вместо этого кладем name - формируем его из path

        for coll, val in pg_dict.iteritems():
            values[coll.name] = val

        return values

    def find_by_file_id_with_oldest_version_not_removed(self, uid, file_id):
        session = self.get_session(uid)

        result = session.execute(SQL_FOLDER_BY_FILE_ID_WITH_OLDEST_VERSION_NOT_REMOVED, {
            'uid': FolderDAOItem.convert_mongo_value_to_postgres_for_key('uid', uid)[1],
            'file_id': FolderDAOItem.convert_mongo_value_to_postgres_for_key('data.file_id', file_id)[1],
        })
        data = result.fetchone()

        if data:
            return FolderDAOItem.create_from_pg_data(data)
        return None

    def get_resource_count(self, uid, path):
        session = self.session
        if session is None:
            session = Session.create_from_uid(uid)

        result = session.execute(SQL_COUNT_RESOURCES_BY_UID, {
            'uid': uid, 'root_folder': path
        })
        data = result.fetchone()

        return data['sum'] - 1

    def get_files_determined(self, uid, path, limit):
        session = self.session
        if session is None:
            session = Session.create_from_uid(uid)

        result = session.execute(SQL_FILES_BY_UID_DETERMINED, {
            'uid': uid, 'root_folder': Address(path, uid=uid).path, 'limit': limit
        })

        return [{'fid': fid, 'path': path} for fid, path in result.fetchall()]

    def get_folders_determined(self, uid, path, limit):
        session = self.session
        if session is None:
            session = Session.create_from_uid(uid)

        result = session.execute(SQL_FOLDERS_BY_UID_DETERMINED, {
            'uid': uid, 'root_folder': Address(path, uid=uid).path, 'limit': limit
        })

        return [{'fid': fid, 'path': path} for fid, path in result.fetchall()]



class PostgresFolderDAOImplementation(BaseDAOImplementation):
    def __init__(self, collection_name):
        self.collection_name = collection_name

    def find(self, spec=None, fields=None, skip=0, limit=0, sort=None, **kwargs):
        raise NotImplementedError()

    def find_on_shard(self, spec=None, fields=None, skip=0, limit=0, sort=None, **kwargs):
        raise NotImplementedError()

    def insert(self, doc_or_docs, manipulate=True, continue_on_error=False, **kwargs):
        raise NotImplementedError()

    def remove(self, spec_or_id=None, multi=True, **kwargs):
        raise NotImplementedError()

    def update(self, spec, document, upsert=False, multi=False, **kwargs):
        raise NotImplementedError()

    def has_parents_with_yarovaya_mark(self, address):
        params = {
            'uid': int(address.uid),
            'all_parent_names': list(reversed([x.name for x in address.get_parents()][:-1] + ['']))
        }
        session = Session.create_from_uid(address.uid)
        return session.execute(SQL_HAS_PARENTS_WITH_YAROVAYA_MARK_BY_PATH, params).fetchone() is not None

    def _count_sub_resources(self, address, sql):
        params = {
            'uid': int(address.uid),
            'path': address.path
        }
        session = Session.create_from_uid(address.uid, read_preference=PGReadPreference.secondary_preferred)
        return session.execute(sql, params).fetchone()[0]

    def count_subfolders(self, address):
        return self._count_sub_resources(address, SQL_COUNT_SUBFOLDERS_BY_PATH)

    def count_subfiles(self, address):
        return self._count_sub_resources(address, SQL_COUNT_SUBFILES_BY_PATH)

    def has_another_media_type_files(self, address, media_types):
        assert isinstance(media_types, tuple)
        params = {
            'uid': int(address.uid),
            'path': address.path,
            'media_types': media_types
        }
        session = Session.create_from_uid(address.uid, read_preference=PGReadPreference.secondary_preferred)
        result = session.execute(SQL_GET_ANOTHER_MEDIA_TYPE_FILE_FROM_FOLDER, params).fetchone()
        return result is not None


class MongoFolderDAOImplementation(BaseDAOImplementation):
    def __init__(self, collection_name):
        self._mongo_helper = MongoHelper()
        self.collection_name = collection_name

    def find(self, spec=None, fields=None, skip=0, limit=0, sort=None, **kwargs):
        raise NotImplementedError()

    def find_on_shard(self, spec=None, fields=None, skip=0, limit=0, sort=None, **kwargs):
        raise NotImplementedError()

    def insert(self, doc_or_docs, manipulate=True, continue_on_error=False, **kwargs):
        raise NotImplementedError()

    def remove(self, spec_or_id=None, multi=True, **kwargs):
        raise NotImplementedError()

    def update(self, spec, document, upsert=False, multi=False, **kwargs):
        raise NotImplementedError()

    def has_parents_with_yarovaya_mark(self, address):
        collection = self._mongo_helper.get_collection_by_address(address.uid, address)
        all_parent_ids = [hashed(':'.join([a.uid, a.path])) for a in address.get_parents()]
        spec = {
            'uid': address.uid,
            'data.yarovaya_mark': True,
            '_id': {'$in': all_parent_ids},
        }
        return collection.find_one(spec) is not None

    def count_subfolders(self, address):
        raise NotImplementedError()

    def count_subfiles(self, address):
        raise NotImplementedError()

    def has_another_media_type_files(self, address, media_types):
        raise NotImplementedError()
