# -*- coding: utf-8 -*-
import datetime
import time

from mpfs.dao.base import BaseDAOItem, BaseDAO, PostgresBaseDAOImplementation, MongoBaseDAOImplementation
from mpfs.dao.fields import UidField, IntegerField, DateTimeField
from mpfs.dao.migration.migration import BaseCommonCollectionMigration
from mpfs.metastorage.postgres.queries import SQL_ADD_USER_TO_RECOUNT
from mpfs.metastorage.postgres.schema import recount


class RecountDAOItem(BaseDAOItem):
    mongo_collection_name = 'recount'
    postgres_table_obj = recount
    is_sharded = False

    @classmethod
    def get_postgres_primary_key(cls):
        return 'uid'

    uid = UidField(mongo_path='_id', pg_path=recount.c.uid)
    ctime = DateTimeField(mongo_path='ctime', pg_path=recount.c.ctime, default_value=None)
    mtime = DateTimeField(mongo_path='mtime', pg_path=recount.c.mtime, default_value=None)
    count = IntegerField(mongo_path='count', pg_path=recount.c.count, default_value=1)


class RecountDAO(BaseDAO):
    dao_item_cls = RecountDAOItem

    def __init__(self):
        super(RecountDAO, self).__init__()
        self._mongo_impl = MongoRecountDAOImplementation(self.dao_item_cls)
        self._pg_impl = PostgresRecountDAOImplementation(self.dao_item_cls)

    def add(self, uid):
        """
        Добавить пользователя в коллекцию на пересчёт счётчиков места
        """
        return self._get_impl(None).add(uid)


class MongoRecountDAOImplementation(MongoBaseDAOImplementation):
    def add(self, uid):
        current_time = int(time.time())
        doc = {
            '$setOnInsert': {'ctime': current_time},
            '$set': {'mtime': current_time},
            '$inc': {'count': 1}
        }
        return super(MongoRecountDAOImplementation, self).update({'_id': uid}, doc, upsert=True)


class PostgresRecountDAOImplementation(PostgresBaseDAOImplementation):
    def add(self, uid):
        session = self._get_session()
        with session.begin():
            return session.execute(SQL_ADD_USER_TO_RECOUNT, {'uid': uid, 'timestamp': datetime.datetime.now()})


class RecountMigration(BaseCommonCollectionMigration):
    dao_cls = RecountDAO
