# -*- coding: utf-8 -*-
import re
import copy
import string
import time
import itertools

from abc import abstractmethod
from collections import OrderedDict, defaultdict
from multiprocessing import TimeoutError
from multiprocessing.pool import ThreadPool
from operator import itemgetter

from pymongo import ReadPreference, ASCENDING, DESCENDING

import mpfs.engine.process

from mpfs.config import settings
from mpfs.common.util import merge2
from mpfs.core.address import Address
from mpfs.core.filesystem.cleaner.models import DeletedStid, DeletedStidSources
from mpfs.core.filesystem.dao.file import FileDAOItem, FileDAO
from mpfs.core.filesystem.dao.folder import FolderDAOItem, FolderDAO
from mpfs.core.user.constants import PHOTOUNLIM_AREA_PATH
from mpfs.common.errors import LivePhotoMultipleFound, LivePhotoNotFound
from mpfs.dao.base import BaseDAO, BaseDAOImplementation, MongoHelper, convert_mongo_read_preference, \
    QueryWithParams, FakeColumn
from mpfs.dao.fields import FieldValidationError
from mpfs.dao.session import Session
from mpfs.metastorage.mongo.binary import Binary
from mpfs.metastorage.postgres.exceptions import QueryCanceledError, EofDetectedError
from mpfs.metastorage.postgres.query_executer import ReadPreference as PGReadPreference
from mpfs.dao.cursor import PostgresEmptyCursor, PostgresCursorChain, PostgresCursor
from mpfs.dao.query_converter import MongoQueryConverter
from mpfs.metastorage.mongo.util import hashed, id_for_key, manual_route, parent_for_key, name_for_key
from mpfs.metastorage.postgres.queries import (SQL_FILE_BY_PATH, SQL_FOLDER_BY_PATH, SQL_FILES_BY_PARENT_PATH,
                                               SQL_FOLDERS_BY_PARENT_PATH, SQL_LAST_FILES_FOR_SUBTREE,
                                               SQL_LAST_FILES_BY_UID,
                                               SQL_FILES_BY_FILE_ID,
                                               SQL_FOLDERS_BY_FILE_ID, SQL_FILES_BY_PATHS, SQL_FOLDERS_BY_PATHS,
                                               SQL_FILES_BY_HID, SQL_DELETE_FILE_BY_PATH, SQL_DELETE_FOLDER_BY_PATH,
                                               SQL_DELETE_FILES_BY_PATHS, SQL_DELETE_FOLDERS_BY_PATHS,
                                               SQL_DELETE_FILE_BY_PATH_AND_VERSION,
                                               SQL_DELETE_FOLDER_BY_PATH_AND_VERSION, SQL_REMOVE_FILE_ID_BY_PATH,
                                               SQL_DELETE_FILE_BY_FID, SQL_DELETE_STORAGE_FILE_BY_STORAGE_ID,
                                               SQL_DELETE_FOLDER_BY_FID, SQL_FILE_BY_PATH_AND_VERSION,
                                               SQL_FOLDER_BY_PATH_AND_VERSION, SQL_FOLDERS_BY_UID, SQL_FILES_BY_UID,
                                               SQL_SNAPSHOT_FOLDERS_CHUNK, SQL_SNAPSHOT_FILES_CHUNK,
                                               SQL_FILES_BY_PARENT_PATH_AND_MEDIATYPE_AND_MTIME,
                                               SQL_FILES_BY_UID_AND_MEDIATYPE, SQL_FILES_INCREMENT_DOWNLOAD_COUNTER,
                                               SQL_FOLDERS_INCREMENT_DOWNLOAD_COUNTER, SQL_FILES_WITH_SELECT_TEMPLATE,
                                               SQL_FOLDERS_WITH_SELECT_TEMPLATE, SQL_FILES_BY_STID,
                                               SQL_SEARCH_FILES_BY_NAME, SQL_SEARCH_FOLDERS_BY_NAME,
                                               SQL_STIDS_BY_STIDS_IN_FILES, SQL_FILES_BY_SIZE_AND_DTIME,
                                               SQL_REMOVE_FILES_BY_UID_FID,
                                               SQL_FILES_BY_COLLECTION, SQL_FILES_BY_STIDS, SQL_FIND_FILES_BY_UID_FID,
                                               SQL_ONE_FILE_BY_HID, SQL_ONE_FILE_BY_UID_AND_HID,
                                               SQL_ONE_FILE_BY_HID_FOR_ANOTHER_UID, SQL_FIND_STORAGE_FILE_BY_STORAGE_ID,
                                               SQL_FOLDERS_BY_COLLECTION, SQL_FILES_BY_SIZE_AND_DTIME_AND_NOT_IN_UIDS,
                                               SQL_DELETE_FILES_BY_UID, SQL_DELETE_FOLDERS_BY_UID,
                                               SQL_FIND_BY_UID_AND_HIDS, SQL_FILES_BY_UID_MTIME_COLLECTION,
                                               SQL_FILES_BY_PARENT_PATH_AND_NAMES, SQL_FOLDERS_BY_PARENT_PATH_AND_NAMES,
                                               SqlTemplatedQuery, SQL_INSERT_ADDITIONAL_STORAGE_FILE_LINK,
                                               SQL_SELECT_FILE_BY_PATH_FOR_UPDATE, SQL_SET_LIVE_PHOTO_FILE_FLAG,
                                               SQL_FILES_BY_HID_FOR_UID, SQL_FILE_COUNT_BY_HID_FOR_UID,
                                               SQL_GET_ADDITIONAL_FILE_BY_MAIN_FILE_PATH_AND_TYPE,
                                               SQL_REMOVE_ADDITIONAL_STORAGE_FILE_LINK,
                                               SQL_GET_ADDITIONAL_FILE_UID_FID_BY_MAIN_UID_FID,
                                               SQL_GET_FILES_BY_UID_FID, SQL_REMOVE_ADDITIONAL_FILE_LINK_BY_UID_FID,
                                               SQL_FILES_BY_UID_HID, SQL_UNSET_LIVE_PHOTO_FILE_FLAG,
                                               SQL_UNSET_LIVE_PHOTO_FILE_FLAGS, SQL_FILES_BY_UID_OFFICE_DOC_SHORT_ID)
from mpfs.metastorage.postgres.schema import files, storage_files, AdditionalFileTypes
from mpfs.metastorage.postgres.query_executer import PGQueryExecuter


POSTGRES_HARDLINKS_THREADPOOL_SIZE = settings.postgres['hardlinks']['threadpool_size']
POSTGRES_HARDLINKS_QUERY_TIMEOUT = settings.postgres['hardlinks']['query_timeout']


log = mpfs.engine.process.get_default_log()
error_log = mpfs.engine.process.get_error_log()


class ResourceDAO(BaseDAO):
    """Базовый класс для ресурса. Deprecated. Используется старой бизнес-логикой, так как там разделение на папки и
    файлы происходит на уровне сервисов (disk_service и т.д.)
    """
    mongo_coll_name = 'user_data'

    def __init__(self):
        super(ResourceDAO, self).__init__()
        self._mongo_impl = MongoResourceDAOImplementation(self.mongo_coll_name)
        self._pg_impl = PostgresResourceDAOImplementation(self.mongo_coll_name)

    def _get_impl(self, item):
        if isinstance(item, dict):
            uid = item['uid']
        elif isinstance(item, basestring):
            uid = item
        else:
            raise NotImplementedError()
        if mpfs.engine.process.usrctl().is_collection_in_postgres(uid, self.mongo_coll_name):
            return self._pg_impl
        return self._mongo_impl

    def get_by_path(self, uid, path, read_preference=ReadPreference.PRIMARY):
        impl = self._get_impl(uid)
        return impl.get_by_path(uid, path, read_preference=read_preference)

    def aggregate(self, query):
        uid = query[0]['$match']['uid']
        impl = self._get_impl(uid)
        return impl.aggregate(query)

    def find_snapshot_initial_chunk(self, uid):
        impl = self._get_impl(uid)
        return impl.find_snapshot_initial_chunk(uid)

    def find_snapshot_chunk(self, uid, file_id, last_ids, last_file_type, limit):
        impl = self._get_impl(uid)
        return impl.find_snapshot_chunk(uid, file_id, last_ids, last_file_type, limit)

    def find_stids_on_shard(self, stids, shard_name, limit=None):
        impl = self._get_impl_by_shard(shard_name)
        return impl.find_stids_on_shard(stids, shard_name, limit=limit)

    def get_last_files(self, uid, limit=10):
        """Получить последние файлы для всего диска"""
        return self._get_impl(uid).get_last_files(uid, limit=limit)

    def get_last_files_for_subtree(self, uid, path, limit=10):
        """Получить последние файлы из определенной папки и её детей"""
        return self._get_impl(uid).get_last_files_for_subtree(uid, path, limit=limit)

    def find_one_file_on_shard_by_hid(self, hid, shard_name):
        impl = self._get_impl_by_shard(shard_name)
        return impl.find_one_file_on_shard_by_hid(hid, shard_name)

    def find_one_file_item_on_shard_by_hid(self, hid, shard_name):
        if not shard_name:
            raise ValueError('Shard name is not specified')
        impl = self._get_impl_by_shard(shard_name)
        return impl.find_one_file_item_on_shard_by_hid(hid, shard_name)

    def find_one_file_item_on_all_pg_shards_by_hid(self, hid):
        return self._pg_impl.find_one_file_item_on_all_shards_by_hid(hid)

    def find_one_file_by_uid_and_hid(self, uid, hid):
        """Получить 1 файл пользователя по hid-у

        Ищет только в user_data (/disk)
        """
        return self._get_impl(uid).find_one_file_by_uid_and_hid(uid, hid)

    def find_by_uid_and_hids(self, uid, hids, limit=100, offset=0):
        return self._get_impl(uid).find_by_uid_and_hids(uid, hids, limit, offset)

    def is_one_owner(self, uid, hid):
        """Проверяет является ли uid единоличным владельцем hid-а

        Ищет только в user_data (/disk)
        """
        return self._mongo_impl.is_one_owner(uid, hid) and self._pg_impl.is_one_owner(uid, hid)

    def link_live_photo_files(self, uid, photo_path, video_path):
        impl = self._get_impl(uid)
        return impl.link_live_photo_files(uid, photo_path, video_path)

    def unlink_live_photo_files(self, uid, photo_path, video_path):
        impl = self._get_impl(uid)
        return impl.unlink_live_photo_files(uid, photo_path, video_path)

    def get_live_photo_file(self, uid, photo_hid):
        impl = self._get_impl(uid)
        return impl.get_live_photo_file(uid, photo_hid)

    def count_live_photo_files(self, uid, photo_hid):
        impl = self._get_impl(uid)
        return impl.count_live_photo_files(uid, photo_hid)


class ResourceDAOImplementation(BaseDAOImplementation):
    @abstractmethod
    def get_by_path(self, uid, path, read_preference=ReadPreference.PRIMARY):
        raise NotImplementedError()

    @abstractmethod
    def aggregate(self, query):
        raise NotImplementedError()


class PostgresResourceDAOImplementation(ResourceDAOImplementation):
    def __init__(self, collection_name):
        self.collection_name = collection_name

    def find(self, spec=None, fields=None, skip=0, limit=0, sort=None, **kwargs):
        uid = spec.get('uid')
        if uid is None:
            raise NotImplementedError()
        session = Session.create_from_uid(uid)
        return self._find(session, spec=spec, fields=fields, skip=skip, limit=limit, sort=sort, **kwargs)

    def find_on_shard(self, spec=None, fields=None, skip=0, limit=0, sort=None, shard_name=None, **kwargs):
        session = Session.create_from_shard_id(shard_name)
        return self._find(session, spec=spec, fields=fields, skip=skip, limit=limit, sort=sort, **kwargs)

    def _find(self, session, spec=None, fields=None, skip=0, limit=0, sort=None, **kwargs):
        spec_keys = set()
        files_query, folders_query = None, None

        spec = self._preprocess_spec(spec)
        if spec is not None:
            spec_keys = set(spec.keys())

        if spec_keys == {'uid', 'parent', 'data.mt', 'data.mtime'}:
            # кастомный код для ленты, возвращает только файлы
            # если надо будет обобщить, то знай - тут кастомный код, не общий, можешь крушить-ломать
            uid, parent_path, mt, mtime = spec['uid'], spec['parent'], spec['data.mt'], spec['data.mtime']
            if len(mt) != 1 or mt.keys()[0] != '$in':
                raise NotImplementedError()
            mt_values = mt['$in']

            if set(mtime.keys()) not in ({'$gte'}, {'$lte'}, {'$gte', '$lte'}):
                raise NotImplementedError()
            mtime_lte_value = mtime.get('$lte')
            mtime_gte_value = mtime.get('$gte')

            if parent_path.endswith('/'):
                parent_path = parent_path[:-1]  # отрезаем последний слеш (даже у корневой папки '/')

            mt_values = tuple(FileDAOItem.convert_mongo_value_to_postgres_for_key('data.mt', v)[1] for v in mt_values)
            # tuple - это важно тут, чтобы в запрос передался не ARRAY, а тупл, ARRAY с IN не умеет

            query = SQL_FILES_BY_PARENT_PATH_AND_MEDIATYPE_AND_MTIME
            params = {'uid': uid, 'parent_path': parent_path, 'media_type_list': mt_values}

            if mtime_gte_value is not None:
                query += ' AND f.date_modified >= :date_modified_gte'
                _, mtime_gte_value = FileDAOItem.convert_mongo_value_to_postgres_for_key('data.mtime', mtime_gte_value)
                params['date_modified_gte'] = mtime_gte_value
            if mtime_lte_value is not None:
                query += ' AND f.date_modified <= :date_modified_lte'
                _, mtime_lte_value = FileDAOItem.convert_mongo_value_to_postgres_for_key('data.mtime', mtime_lte_value)
                params['date_modified_lte'] = mtime_lte_value

            files_query = QueryWithParams(query, params)
        elif spec_keys >= {'uid', 'parent'}:
            uid, parent_path = spec.pop('uid'), spec.pop('parent')
            if parent_path.endswith('/'):
                parent_path = parent_path[:-1]  # отрезаем последний слеш (даже у корневой папки '/')

            res_type = None
            if 'type' in spec:
                res_type = spec.pop('type')

            spec = self._filter_mongo_specific_extra_parameters(spec)

            if res_type is None or res_type == 'file':
                sql_query = SQL_FILES_BY_PARENT_PATH
                query_params = {'uid': uid, 'parent_path': parent_path}
                if spec:
                    where_sql, params = MongoQueryConverter(FileDAOItem)._get_where_key_values(spec)
                    sql_query += ' AND ' + where_sql
                    query_params.update(params)
                files_query = QueryWithParams(sql_query, query_params)
            # тут кастомная обработка случая, когда parent_path == '/' - это надо, потому что в постгресе папки
            # disk, trash, hidden_data лежат в одной таблице, поэтому если запросить все, что лежит в / - вернутся все
            # папки, а это явно не то, чего мы хотим, мы хотим получить одну подпапку, поэтому запрос с парентом / мы
            # заменяем на запрос подпапки в зависимости от имени коллекции
            if res_type is None or res_type == 'dir':
                if parent_path == '':
                    path = self._get_folder_for_collection(self.collection_name)
                    folders_query = QueryWithParams(SQL_FOLDER_BY_PATH, {'uid': uid, 'path': path})
                else:
                    folders_query = QueryWithParams(SQL_FOLDERS_BY_PARENT_PATH,
                                                    {'uid': uid, 'parent_path': parent_path})

                if spec:
                    if self._spec_has_file_specific_filter(spec):
                        # если есть ключ в спеке, который есть только у файлов, то запрос на папки просто убираем, так
                        # как папки под этот фильтр точно не попадают
                        folders_query = None
                    else:
                        where_sql, params = MongoQueryConverter(FolderDAOItem)._get_where_key_values(spec)
                        folders_query.query += ' AND ' + where_sql
                        folders_query.params.update(params)
        elif spec_keys == {'uid', 'path'} or spec_keys == {'uid', 'path', 'key'}:
            uid, path = spec['uid'], spec['path']
            if isinstance(path, basestring):
                files_query = QueryWithParams(SQL_FILE_BY_PATH, {'uid': uid, 'path': path})
                folders_query = QueryWithParams(SQL_FOLDER_BY_PATH, {'uid': uid, 'path': path})
            else:
                paths = self._get_list_from_in_clause(path)

                parents = set()
                for path in paths:
                    parents.add(parent_for_key(path))

                if len(parents) < len(paths) / 2:
                    parent_to_children = defaultdict(list)
                    for path in paths:
                        parent_to_children[parent_for_key(path)].append(name_for_key(path))

                    query_params = {'uid': uid}
                    counter = 0
                    folders_subqueries, files_subqueries = [], []
                    for parent, children in parent_to_children.iteritems():
                        new_parent_path_param = 'parent_path_%d' % counter
                        new_names_param = 'names_%d' % counter
                        folders_subqueries.append(
                            SQL_FOLDERS_BY_PARENT_PATH_AND_NAMES.replace_param('parent_path', new_parent_path_param)
                            .replace_param('names', new_names_param)
                        )
                        files_subqueries.append(
                            SQL_FILES_BY_PARENT_PATH_AND_NAMES.replace_param('parent_path', new_parent_path_param)
                            .replace_param('names', new_names_param)
                        )
                        query_params.update({
                            new_parent_path_param: parent,
                            new_names_param: tuple(children)
                        })
                        counter += 1

                    files_query = QueryWithParams(SqlTemplatedQuery.join(files_subqueries), query_params)
                    folders_query = QueryWithParams(SqlTemplatedQuery.join(folders_subqueries), query_params)
                else:
                    files_query = QueryWithParams(SQL_FILES_BY_PATHS, {'uid': uid, 'paths': paths})
                    folders_query = QueryWithParams(SQL_FOLDERS_BY_PATHS, {'uid': uid, 'paths': paths})
        elif spec_keys == {'uid', 'path', 'version'}:
            uid, path, version = spec['uid'], spec['path'], spec['version']
            if isinstance(path, basestring):
                files_query = QueryWithParams(SQL_FILE_BY_PATH_AND_VERSION,
                                              {'uid': uid, 'path': path, 'version': version})
                folders_query = QueryWithParams(SQL_FOLDER_BY_PATH_AND_VERSION,
                                                {'uid': uid, 'path': path, 'version': version})
            else:
                raise NotImplementedError()
        elif spec_keys == {'uid', 'data.file_id'}:
            uid, file_id = spec['uid'], spec['data.file_id']
            root_folder = self._get_folder_for_collection(self.collection_name)
            file_ids = self._get_list_from_in_clause(file_id)

            converted_file_ids = []
            for f in file_ids:
                try:
                    converted_file_id = FileDAOItem.convert_mongo_value_to_postgres_for_key('data.file_id', f)[1]
                    converted_file_ids.append(converted_file_id)
                except FieldValidationError:
                    pass  # не ищем по полям, не прошедшим валидацию, чтобы они не сломали запрос

            if converted_file_ids:
                file_ids = tuple(converted_file_ids)
                files_query = QueryWithParams(SQL_FILES_BY_FILE_ID,
                                              {'uid': uid, 'file_ids': file_ids, 'root_folder': root_folder})
                folders_query = QueryWithParams(SQL_FOLDERS_BY_FILE_ID,
                                                {'uid': uid, 'file_ids': file_ids, 'root_folder': root_folder})
        elif spec_keys == {'uid', 'type'}:
            uid, res_type = spec['uid'], spec['type']
            root_folder = self._get_folder_for_collection(self.collection_name)
            if res_type == 'dir':
                folders_query = QueryWithParams(SQL_FOLDERS_BY_UID, {'uid': uid, 'root_folder': root_folder})
            elif res_type == 'file':
                files_query = QueryWithParams(SQL_FILES_BY_UID, {'uid': uid, 'root_folder': root_folder})
            else:
                raise NotImplementedError()
        elif spec_keys == {'uid', 'data.mt'} and isinstance(spec['data.mt'], dict) and len(spec['data.mt']) == 1 and \
                spec['data.mt'].keys()[0] == '$in':
            # search reindex with mediatype -> files only
            uid, mt_values = spec['uid'], spec['data.mt']['$in']

            mt_values = tuple(FileDAOItem.convert_mongo_value_to_postgres_for_key('data.mt', v)[1] for v in mt_values)
            root_folder = self._get_folder_for_collection(self.collection_name)
            files_query = QueryWithParams(SQL_FILES_BY_UID_AND_MEDIATYPE,
                                          {'uid': uid, 'root_folder': root_folder,
                                           'media_type_tuple': tuple(mt_values)})
        elif spec_keys == {'uid', '$or'} and len(spec['$or']) == 2 and \
                'key' in spec['$or'][0] and 'key' in spec['$or'][1] and \
                spec['$or'][0]['key'].keys() == spec['$or'][1]['key'].keys() == ['$regex']:
            # поиск для саппортилки
            search_name = spec['$or'][0]['key']['$regex'][6:-6]
            uid = spec['uid']
            root_folder = self._get_folder_for_collection(self.collection_name)
            folders_query = QueryWithParams(SQL_SEARCH_FOLDERS_BY_NAME, {'uid': uid,
                                                                         'root_folder': root_folder,
                                                                         'name': search_name})
            files_query = QueryWithParams(SQL_SEARCH_FILES_BY_NAME, {'uid': uid,
                                                                     'root_folder': root_folder,
                                                                     'name': search_name})
        elif spec_keys == {'uid'}:
            uid = spec['uid']
            root_folder = self._get_folder_for_collection(self.collection_name)
            folders_query = QueryWithParams(SQL_FOLDERS_BY_UID, {'uid': uid, 'root_folder': root_folder})
            files_query = QueryWithParams(SQL_FILES_BY_UID, {'uid': uid, 'root_folder': root_folder})
        elif spec_keys == {'hid'} or (spec_keys == {'hid', 'data.stids'} and spec['data.stids'] == {'$exists': 1}):
            # получение хардлинка на определенном шарде
            root_path = self._get_folder_for_collection(self.collection_name)  # ищем хардлинк в определенной коллекции
            mongo_hid = spec['hid']
            pg_hid = FileDAOItem.convert_mongo_value_to_postgres_for_key('hid', mongo_hid)[1]
            files_query = QueryWithParams(SQL_FILES_BY_HID, {'hid': pg_hid, 'root_folder': root_path})
            folders_query = None
        elif spec_keys == {'uid', 'hid'}:
            # получение хардлинка на определенном шарде у определенного пользователя
            root_path = self._get_folder_for_collection(self.collection_name)  # ищем хардлинк в определенной коллекции
            uid = spec['uid']
            mongo_hid = spec['hid']
            pg_hid = FileDAOItem.convert_mongo_value_to_postgres_for_key('hid', mongo_hid)[1]
            files_query = QueryWithParams(SQL_FILES_BY_UID_HID, {'uid': uid, 'hid': pg_hid, 'root_folder': root_path})
            folders_query = None
        elif spec_keys == {'uid', 'data.office_doc_short_id'}:
            # получение ресурса по офисному индексированному полю
            root_path = self._get_folder_for_collection(self.collection_name)
            uid = spec['uid']
            mongo_office_doc_short_id = spec['data.office_doc_short_id']
            pg_office_doc_short_id = FileDAOItem.convert_mongo_value_to_postgres_for_key(
                'data.office_doc_short_id',
                mongo_office_doc_short_id
            )[1]
            files_query = QueryWithParams(SQL_FILES_BY_UID_OFFICE_DOC_SHORT_ID,
                                          {'uid': uid,
                                           'office_doc_short_id': pg_office_doc_short_id,
                                           'root_folder': root_path})
            folders_query = None
        elif spec_keys == {'data.stids.stid'}:
            mongo_stid = spec['data.stids.stid']
            pg_stid = FileDAOItem.convert_mongo_value_to_postgres_for_key('data.stids', mongo_stid)[1]
            files_query = QueryWithParams(SQL_FILES_BY_STID, {'stid': pg_stid})
            folders_query = None
        elif spec_keys == {'data.stids'} and isinstance(spec['data.stids'], dict) and \
                len(spec['data.stids']) == 1 and spec['data.stids'].keys() == ['$elemMatch'] and \
                spec['data.stids']['$elemMatch'].get('stid', {}).get('$in', None) is not None:
            stids = tuple(spec['data.stids']['$elemMatch']['stid']['$in'])
            root_folder = self._get_folder_for_collection(self.collection_name)
            files_query = QueryWithParams(SQL_FILES_BY_STIDS, {'stids': stids, 'root_folder': root_folder})
            folders_query = None
        elif spec_keys == {'uid', 'type', 'data.mtime'} and spec['type'] == 'file' and \
                isinstance(spec['data.mtime'], dict) and len(spec['data.mtime']) == 1 and \
                spec['data.mtime'].keys() == ['$gte']:
            # запрос от timeline с отсечением по mtime
            # {'type': 'file', 'uid': 128280859, 'data.mtime': {'$gte': 1519208013}}
            root_folder = self._get_folder_for_collection(self.collection_name)
            uid = spec['uid']
            mtime_gte = spec['data.mtime']['$gte']
            mtime_gte = FileDAOItem.convert_mongo_value_to_postgres_for_key('data.mtime', mtime_gte)[1]
            files_query = QueryWithParams(SQL_FILES_BY_UID_MTIME_COLLECTION, {
                'uid': uid,
                'mtime_gte': mtime_gte,
                'root_folder': root_folder
            })
        elif spec == {} or spec is None:
            root_folder = self._get_folder_for_collection(self.collection_name)
            files_query = QueryWithParams(SQL_FILES_BY_COLLECTION, {'root_folder': root_folder})
            folders_query = QueryWithParams(SQL_FOLDERS_BY_COLLECTION, {'root_folder': root_folder})
        elif spec_keys >= {'uid'}:
            uid = spec['uid']
            files_query, folders_query = self._build_queries_by_arbitary_spec(spec)
        else:
            raise NotImplementedError('spec "%s" is not supported now' % spec_keys)

        if sort:
            sql_files_sort, sql_folders_sort = None, None
            if files_query:
                filtered_sort = self._prepare_sort_for_file(sort)
                append_nulls_last = False
                append_nulls_first = False
                if filtered_sort == [('data.etime', -1)]:
                    append_nulls_last = True
                elif filtered_sort == [('data.etime', 1)]:
                    append_nulls_first = True
                sql_files_sort = MongoQueryConverter(FileDAOItem).convert_sort_fields_to_sql(
                    filtered_sort, append_nulls_last=append_nulls_last, append_nulls_first=append_nulls_first)
            if folders_query:
                sql_folders_sort = MongoQueryConverter(FolderDAOItem).convert_sort_fields_to_sql(
                    self._prepare_sort_for_folder(sort))
            if sql_files_sort:
                files_query.query += ' ORDER BY ' + sql_files_sort
            if sql_folders_sort:
                folders_query.query += ' ORDER BY ' + sql_folders_sort

        limit = int(limit)

        read_preference = self.get_read_preference(kwargs)
        if read_preference is not None:
            session = session.create_from_shard_id(
                session._shard_id,
                ucache_hint_uid=session.ucache_hint_uid,
                read_preference=convert_mongo_read_preference(read_preference),
            )

        queries, dao_item_classes = [], []
        if folders_query:
            queries.append(folders_query)
            dao_item_classes.append(FolderDAOItem)
        if files_query:
            queries.append(files_query)
            dao_item_classes.append(FileDAOItem)

        if not queries:
            return PostgresEmptyCursor()

        return PostgresCursorChain([
            PostgresCursor(session, query, dao_item_cls)
            for query, dao_item_cls in zip(queries, dao_item_classes)
        ], limit=limit, offset=skip)

    def update(self, spec, document, upsert=False, multi=False, **kwargs):
        if len(document) == 1 and document.keys()[0] in ('$set', '$unset'):
            if '$unset' in document and document['$unset'] == {'data.file_id': 1}:
                self._remove_file_id_from_document(int(spec['uid']), spec['path'])
            else:
                raise NotImplementedError()
        elif (set(spec.keys()) == {'uid', 'path', 'data.download_counter'} and
              spec['data.download_counter'] == {'$exists': 1} and
              document == {'$inc': {'data.download_counter': 1}}):
            uid, path = spec['uid'], spec['path']
            updated_count = self._increment_download_counter(uid, path)
            return {'updatedExisting': updated_count > 0}
        else:
            self._check_resource_type(document)

            uid = int(spec['uid'])
            path = spec['path']
            resource_type = document['type']

            if 'parent' in document:
                parent_path = document['parent']
                document['parent'] = self._get_folder_fid(uid, parent_path)
            else:
                if resource_type != 'dir' or document['path'] != '/':
                    # document witout parent could be only for root folder '/'
                    raise ValueError('updating document with no parent: %s' % document)

            if resource_type == 'file':
                update_existing = self._update_file(uid, path, document, upsert)
            elif resource_type == 'dir':
                update_existing = self._update_folder(uid, path, document, upsert)
            else:
                raise NotImplementedError()

            return {'updatedExisting': update_existing}

    def insert(self, doc_or_docs, manipulate=True, continue_on_error=False, **kwargs):
        if isinstance(doc_or_docs, dict):
            doc_or_docs = [doc_or_docs]

        for data in doc_or_docs:
            resource_type = data.get('type')

            if 'uid' not in data or 'parent' not in data:
                raise ValueError('data has no uid or parent path')

            uid = int(data['uid'])
            parent_path = data['parent']
            data['parent'] = self._get_folder_fid(uid, parent_path)

            if resource_type == 'dir':
                self._insert_folder(uid, data)
            elif resource_type == 'file':
                self._insert_file(uid, data)
            else:
                raise NotImplementedError()

    def remove(self, spec_or_id=None, multi=True, **kwargs):
        spec = self._preprocess_spec(spec_or_id)
        spec_keys = set(spec.keys())

        if spec_keys in ({'uid', 'path', 'key'}, {'uid', 'path'}):
            uid, path = spec['uid'], spec['path']
            if isinstance(path, basestring):
                files_query = QueryWithParams(SQL_DELETE_FILE_BY_PATH, {'uid': uid, 'path': path})
                folders_query = QueryWithParams(SQL_DELETE_FOLDER_BY_PATH, {'uid': uid, 'path': path})
            else:
                paths = self._get_list_from_in_clause(path)
                files_query = QueryWithParams(SQL_DELETE_FILES_BY_PATHS, {'uid': uid, 'paths': paths})
                folders_query = QueryWithParams(SQL_DELETE_FOLDERS_BY_PATHS, {'uid': uid, 'paths': paths})
        elif spec_keys == {'uid', 'path', 'version'}:
            uid, path, version = spec['uid'], spec['path'], spec['version']
            if not isinstance(path, basestring):
                raise NotImplementedError()

            files_query = QueryWithParams(SQL_DELETE_FILE_BY_PATH_AND_VERSION,
                                          {'uid': uid, 'path': path, 'version': version})
            folders_query = QueryWithParams(SQL_DELETE_FOLDER_BY_PATH_AND_VERSION,
                                            {'uid': uid, 'path': path, 'version': version})
        elif spec_keys == {'uid'}:
            uid = spec['uid']
            files_query = QueryWithParams(SQL_DELETE_FILES_BY_UID, {'uid': uid})
            folders_query = QueryWithParams(SQL_DELETE_FOLDERS_BY_UID, {'uid': uid})
        else:
            raise NotImplementedError()

        session = Session.create_from_uid(uid)
        with session.begin():
            session.execute_queries([files_query, folders_query])

    def get_by_path(self, uid, path, read_preference=ReadPreference.PRIMARY):
        assert isinstance(uid, int)

        read_preference = convert_mongo_read_preference(read_preference)
        session = Session.create_from_uid(uid, read_preference=read_preference)

        file_query = QueryWithParams(SQL_FILE_BY_PATH, {'uid': uid, 'path': path})
        file_row = PostgresCursor(session, file_query, FileDAOItem)[0]

        if file_row is not None:
            file_obj = FileDAOItem.create_from_pg_data(file_row)
            return file_obj

        folder_query = QueryWithParams(SQL_FOLDER_BY_PATH, {'uid': uid, 'path': path})
        folder_row = PostgresCursor(session, folder_query, FolderDAOItem)[0]

        if folder_row is not None:
            folder_obj = FolderDAOItem.create_from_pg_data(folder_row)
            return folder_obj

        return None

    def aggregate(self, query):
        """Не настоящий aggregate, а только один частный его случай, а именно для поддержки ручки get_last_files.
        Он выглядит так:
        [
         {'$match': {'type': 'file', 'uid': u'128280859'}},
         {'$sort': SON([('data.mtime', -1)])},
         {'$limit': 1000},
         {'$match': {'key': {'$regex': u'^\\/disk\\/'}}},
         {'$limit': 3}
        ]
        """
        if isinstance(query, list) and len(query) == 5:
            pipe_keys = []
            for pipe_item in query:
                if len(pipe_item) == 1:
                    pipe_keys.append(pipe_item.keys()[0])
            if pipe_keys == ['$match', '$sort', '$limit', '$match', '$limit']:
                if set(query[0]['$match'].keys()) == {'type', 'uid'}:
                    uid = int(query[0]['$match']['uid'])
                    limit = query[4]['$limit']
                    return self._get_last_files(uid, limit)

        raise NotImplementedError()

    def find_one_file_on_shard_by_hid(self, hid, shard_name):
        pg_hid = FileDAOItem.convert_mongo_value_to_postgres_for_key('hid', hid)[1]
        query = QueryWithParams(SQL_ONE_FILE_BY_HID, {'hid': pg_hid})

        session = Session.create_from_shard_id(shard_name, read_preference=PGReadPreference.secondary_preferred)
        return PostgresCursor(session, query, FileDAOItem)[0]

    def find_one_file_item_on_shard_by_hid(self, hid, shard_name):
        session = Session.create_from_shard_id(shard_name)
        return self._find_one_file_item_on_shard_by_hid(hid, session)

    @classmethod
    def _find_one_file_item_on_shard_by_hid(cls, hid, session):
        pg_hid = FileDAOItem.convert_mongo_value_to_postgres_for_key('hid', hid)[1]
        result = session.execute(SQL_ONE_FILE_BY_HID, {'hid': pg_hid})
        for data in result:
            return FileDAOItem.create_from_pg_data(data)
        return None

    def find_one_file_item_on_all_shards_by_hid(self, hid):
        try:
            sessions = Session.create_for_all_shards(
                skip_unavailable_shards=True,
                read_preference=PGReadPreference.secondary_preferred,
                use_threads=True,
            )
            threadpool_size = POSTGRES_HARDLINKS_THREADPOOL_SIZE

            if threadpool_size == -1:
                threadpool_size = len(sessions)

            def search_hardlink_on_shard(session):
                item = None
                try:
                    item = self._find_one_file_item_on_shard_by_hid(hid, session)
                except QueryCanceledError:
                    log.exception('Hardlink `%s` find query was not completed within %d ms',
                                  hid, POSTGRES_HARDLINKS_QUERY_TIMEOUT)
                except EofDetectedError:
                    log.exception('Hardlink `%s` find query was not completed because connection was closed by server',
                                  hid)
                session.close()
                return item

            pool = ThreadPool(processes=threadpool_size)
            it = pool.imap_unordered(search_hardlink_on_shard, sessions)
            pool.close()

            timeout = float(POSTGRES_HARDLINKS_QUERY_TIMEOUT) / 1000.0
            found_file = None
            try:
                for _ in xrange(len(sessions)):
                    time_before = time.time()

                    result = it.next(timeout)

                    elapsed_time = time.time() - time_before
                    timeout -= elapsed_time
                    if timeout < 0:
                        timeout = 0

                    if not found_file and result:
                        found_file = result
                        break
            except TimeoutError:
                pass
                # есть предположение, что запросы зависают об этот abort
                #for session in sessions:
                #    session.abort_query_execution()  # отменяем операции, если они еще бегут
            finally:
                pool.terminate()

            return found_file
        except Exception:
            error_log.exception(
                'Error while trying to get hardlink with hid `%s`' % hid
            )
            return None

    def find_one_file_by_uid_and_hid(self, uid, hid):
        pg_hid = FileDAOItem.convert_mongo_value_to_postgres_for_key('hid', hid)[1]
        query = QueryWithParams(SQL_ONE_FILE_BY_UID_AND_HID, {'uid': int(uid), 'hid': pg_hid})

        session = Session.create_from_uid(uid, read_preference=PGReadPreference.secondary_preferred)
        return next(PostgresCursor(session, query, FileDAOItem), None)

    def find_by_uid_and_hids(self, uid, hids, limit, offset):
        root_folder = self._get_folder_for_collection(self.collection_name)
        pg_hids = [FileDAOItem.convert_mongo_value_to_postgres_for_key('hid', hid)[1] for hid in hids]
        session = Session.create_from_uid(uid)
        result = session.execute(SQL_FIND_BY_UID_AND_HIDS,
                                 {'uid': int(uid), 'hids': tuple(pg_hids), 'limit': limit, 'offset': offset,
                                  'root_folder': root_folder})
        return (FileDAOItem.create_from_pg_data(x) for x in result)

    def is_one_owner(self, uid, hid):
        pg_hid = FileDAOItem.convert_mongo_value_to_postgres_for_key('hid', hid)[1]
        query = QueryWithParams(SQL_ONE_FILE_BY_HID_FOR_ANOTHER_UID, {'uid': int(uid), 'hid': pg_hid})
        for shard_id in PGQueryExecuter().get_all_shard_ids():
            session = Session.create_from_shard_id(shard_id, read_preference=PGReadPreference.secondary_preferred)
            item = next(PostgresCursor(session, query, FileDAOItem), None)
            if item:
                return False
        return True

    def find_stids_on_shard(self, stids, shard_name, limit=None):
        # запрос от чистки стораджа
        files_query = QueryWithParams(SQL_STIDS_BY_STIDS_IN_FILES, {'stids': tuple(stids)})

        session = Session.create_from_shard_id(shard_name, read_preference=PGReadPreference.secondary_preferred)
        return PostgresCursor(session, files_query, FileDAOItem)

    def _get_folder_fid(self, uid, path):
        assert path.startswith('/')

        session = Session.create_from_uid(uid)
        query = QueryWithParams(SQL_FOLDER_BY_PATH, {'uid': uid, 'path': path})
        mongo_data = PostgresCursor(session, query, FolderDAOItem)[0]

        if mongo_data is None:
            raise RuntimeError('folder with path "%s" was not found' % path)

        folder_obj = FolderDAOItem.create_from_mongo_dict(mongo_data)
        return folder_obj.fid

    def _get_last_files(self, uid, limit):
        # DEPRECATED
        FETCH_LIMIT = 1000
        if limit > FETCH_LIMIT:
            limit = FETCH_LIMIT

        session = Session.create_from_uid(uid, read_preference=PGReadPreference.secondary_preferred)
        #TODO передавать path='/disk' - ошибка. Починим позже.
        file_rows_query = QueryWithParams(SQL_LAST_FILES_FOR_SUBTREE, {'uid': uid, 'path': '/disk', 'limit': limit})

        cursor = PostgresCursor(session, file_rows_query, FileDAOItem)
        return {'ok': 1.0, 'result': list(cursor)}  # монга результаты возвращает тоже в списке, не курсором

    @staticmethod
    def _get_list_from_in_clause(spec_or_value):
        if isinstance(spec_or_value, dict) and len(spec_or_value) == 1 and spec_or_value.keys()[0] == '$in':
            return spec_or_value['$in']
        elif isinstance(spec_or_value, basestring):
            return [spec_or_value]
        else:
            raise NotImplementedError()

    @staticmethod
    def _preprocess_spec(spec):
        if spec is None:
            return None

        result = copy.copy(spec)
        if 'uid' in spec and isinstance(spec['uid'], basestring):
            result['uid'] = int(spec['uid'])  # TODO: process string uids here
        if 'key' in spec and 'path' not in spec:
            result['path'] = result.pop('key')  # костыль для поддержки AllUserDataCollection, который передает key
        return result

    @staticmethod
    def _get_folder_for_collection(coll_name):
        from mpfs.core.factory import get_folder_for_collection  # цикл. импорт
        return get_folder_for_collection(coll_name)

    @staticmethod
    def _prepare_sort_for_file(sort):
        """Функция фильтрует параметры сортировки, которые не подходят для файлов.
        Например, если в sort есть поле type, то для файлов мы его фильтруем, потому что такого мы и так сортируем
        по типу (делаем запросы в разные таблицы).
        """
        new_sort = []
        for key, direction in sort:
            if key == 'type':
                continue
            if key == 'key':
                key = 'name'  # такой вот хак, ровно обратный этому: mpfs.core.services.disk_service.NameFilter
            new_sort.append((key, direction))
        return new_sort

    @staticmethod
    def _prepare_sort_for_folder(sort):
        """Функция фильтрует параметры сортировки, которые не подходят для папок.
        Например, если в sort есть поле size, то для папок мы его фильтруем, потому что такого поля для папок у нас нет.
        """
        new_sort = []
        for key, direction in sort:
            if key in ('type', 'data.size', 'data.etime'):
                continue
            if key == 'key':
                key = 'name'  # такой вот хак, ровно обратный этому: mpfs.core.services.disk_service.NameFilter
            new_sort.append((key, direction))
        return new_sort

    @staticmethod
    def _remove_file_id_from_document(uid, path):
        session = Session.create_from_uid(uid)
        with session.begin():
            session.execute(SQL_REMOVE_FILE_ID_BY_PATH, {'path': path, 'uid': uid})

    @staticmethod
    def _check_resource_type(resource):
        resource_type = resource.get('type')
        if resource_type not in ('file', 'dir'):
            raise NotImplementedError()

    @staticmethod
    def _get_changed_stids(item, file_obj):
        changed_stids = []
        if item is not None:
            if item[storage_files.c.stid] != file_obj.file_stid:
                changed_stids.append((item[storage_files.c.stid], 'file_mid'))
            if item[storage_files.c.digest_stid] != file_obj.digest_stid:
                changed_stids.append((item[storage_files.c.digest_stid], 'digest_mid'))
            if item[storage_files.c.preview_stid] is not None and \
                    item[storage_files.c.preview_stid] != file_obj.preview_stid:
                changed_stids.append((item[storage_files.c.preview_stid], 'pmid'))
        return changed_stids

    @classmethod
    def _update_file(cls, uid, path, document, upsert):
        update_existing = False

        if not upsert:
            raise NotImplementedError()

        changed_stids = []

        session = Session.create_from_uid(uid)
        with session.begin():
            item = session.execute(SQL_FILE_BY_PATH, {'path': path, 'uid': uid}).fetchone()
            file_obj = FileDAOItem.create_from_mongo_dict(document, validate=False)

            _, storage_id = FileDAOItem.convert_mongo_value_to_postgres_for_key('hid', document['hid'])
            if item is not None:
                document['_id'] = item['fid']
                session.execute(SQL_DELETE_FILE_BY_FID, {'uid': uid, 'fid': item['fid']})

                if not file_obj.is_storage_part_equal(FileDAOItem.create_from_pg_data(item)):
                    changed_stids = cls._get_changed_stids(item, file_obj)

                    file_dao = FileDAO(session)
                    if storage_id.get_hex() == item['storage_id']:
                        file_dao.update_storage_part(file_obj)
                    else:
                        file_dao.remove_hanging_storage_files((item['storage_id'],))

                update_existing = True
            elif 'hid' in document:
                storage_item = session.execute(SQL_FIND_STORAGE_FILE_BY_STORAGE_ID,
                                               {'storage_id': storage_id}).fetchone()
                if (storage_item is not None and
                        not file_obj.is_storage_part_equal(FileDAOItem.create_from_pg_data(storage_item))):
                    changed_stids = cls._get_changed_stids(storage_item, file_obj)
                    FileDAO(session).update_storage_part(file_obj)
                update_existing = False

            FileDAO(session).create(file_obj)

        if changed_stids:
            objs = [
                DeletedStid(stid=stid, stid_type=stid_type, stid_source=DeletedStidSources.DAO_UPDATE_FILE)
                for stid, stid_type in changed_stids
            ]
            DeletedStid.controller.bulk_create(objs)

        return update_existing

    @staticmethod
    def _update_folder(uid, path, document, upsert):
        update_existing = False

        session = Session.create_from_uid(uid)
        with session.begin():
            item = session.execute(SQL_FOLDER_BY_PATH, {'path': path, 'uid': uid}).fetchone()
            if not upsert and item is None:
                raise RuntimeError('folder not found')

            folder_obj = FolderDAOItem.create_from_mongo_dict(document, validate=False)

            if item is not None:
                document['_id'] = item['fid']
                FolderDAO(session).update_item_values(folder_obj)
                update_existing = True
            else:
                FolderDAO(session).create(folder_obj)

        return update_existing

    def _insert_folder(self, uid, data):
        insert_data = self._convert_mongo_folder_data_to_postgres(data)
        query = \
            'INSERT INTO disk.folders (' + \
            ','.join(insert_data.keys()) + \
            ') VALUES (' + \
            ','.join([':' + i for i in insert_data.iterkeys()]) + \
            ')'

        session = Session.create_from_uid(uid)
        with session.begin():
            return session.execute(query, insert_data)

    def _insert_file(self, uid, data):
        files_data, storage_files_data = self._convert_mongo_file_data_to_postgres(data)
        files_query = \
            'INSERT INTO disk.files (' + \
            ','.join(files_data.keys()) + \
            ') VALUES (' + \
            ','.join([':' + i for i in files_data.iterkeys()]) + \
            ')'

        storage_files_query = \
            'INSERT INTO disk.storage_files (' + \
            ','.join(storage_files_data.keys()) + \
            ') VALUES (' + \
            ','.join([':' + i for i in storage_files_data.iterkeys()]) + \
            ') ON CONFLICT DO NOTHING'

        session = Session.create_from_uid(uid)
        with session.begin():
            session.execute(files_query, files_data)
            session.execute(storage_files_query, storage_files_data)

    @staticmethod
    def _convert_mongo_folder_data_to_postgres(data):
        folder_obj = FolderDAOItem.create_from_mongo_dict(data)
        pg_data = folder_obj.get_postgres_representation(skip_missing_fields=True)
        converted_data = OrderedDict()
        for column, value in pg_data.iteritems():
            if isinstance(column, basestring):
                converted_data[column] = value
            else:
                converted_data[column.name] = value

        path = converted_data.pop('path', None)
        if path and 'name' not in converted_data:
            converted_data['name'] = path.split('/')[-1]

        return converted_data

    @staticmethod
    def _convert_mongo_file_data_to_postgres(data):
        file_obj = FileDAOItem.create_from_mongo_dict(data)
        pg_data = file_obj.get_postgres_representation(skip_missing_fields=True)
        converted_files_data = OrderedDict()
        converted_storage_files_data = OrderedDict()
        for column, value in pg_data.iteritems():
            if isinstance(column, basestring):
                converted_files_data[column] = value
            elif isinstance(column, FakeColumn) or column.table == files:
                converted_files_data[column.name] = value
            elif column.table == storage_files:
                converted_storage_files_data[column.name] = value

        path = converted_files_data.pop('path', None)
        if path and 'name' not in converted_files_data:
            converted_files_data['name'] = path.split('/')[-1]
        converted_files_data['storage_id'] = converted_storage_files_data['storage_id']

        return converted_files_data, converted_storage_files_data

    def find_snapshot_initial_chunk(self, uid):
        # В PG этот метод не должен ничего возвращать, так как у всех файлов есть file_id
        return []

    def find_snapshot_chunk(self, uid, file_id, last_ids, last_file_type, limit):
        session = Session.create_from_uid(uid)

        if last_file_type == 'file':
            files_query = \
                self._find_query_for_snapshot_chunk_from_files(uid, file_id, last_ids, limit)
            return PostgresCursor(session, files_query, FileDAOItem)

        folders_query = \
            self._find_query_for_snapshot_chunk_from_folders(uid, file_id, last_ids, limit)
        files_query = \
            self._find_query_for_snapshot_chunk_from_files(uid, '', [], limit)

        return PostgresCursorChain([
            PostgresCursor(session, folders_query, FolderDAOItem),
            PostgresCursor(session, files_query, FileDAOItem)
        ], limit=limit)

    def _find_query_for_snapshot_chunk_from_folders(self, uid, file_id, last_ids, limit):
        return QueryWithParams(SQL_SNAPSHOT_FOLDERS_CHUNK,
                               {'uid': uid, 'file_id': '\\x' + file_id,
                                'last_ids': last_ids,
                                'root_folder': self._get_folder_for_collection(self.collection_name) + '%', 'limit': limit})

    def _find_query_for_snapshot_chunk_from_files(self, uid, file_id, last_ids, limit):
        return QueryWithParams(SQL_SNAPSHOT_FILES_CHUNK,
                               {'uid': uid, 'file_id': '\\x' + file_id,
                                'last_ids': last_ids,
                                'root_folder': self._get_folder_for_collection(self.collection_name) + '%', 'limit': limit})

    def _increment_download_counter(self, uid, path):
        session = Session.create_from_uid(uid)
        with session.begin():
            files_result = session.execute(SQL_FILES_INCREMENT_DOWNLOAD_COUNTER, {'uid': uid, 'path': path})
            folders_result = session.execute(SQL_FOLDERS_INCREMENT_DOWNLOAD_COUNTER, {'uid': uid, 'path': path})
            return files_result.rowcount + folders_result.rowcount

    @staticmethod
    def _filter_mongo_specific_extra_parameters(spec):
        filtered_spec = copy.deepcopy(spec)

        if filtered_spec.get('data.visible') == {'$in': [u'1', 1]}:
            filtered_spec['data.visible'] = 1

        if '$or' in filtered_spec and isinstance(filtered_spec['$or'], list) and len(filtered_spec['$or']) == 2:
            if filtered_spec['$or'][0].keys()[0] == 'data.meta' and filtered_spec['$or'][1].keys()[0] == 'data.public':
                del filtered_spec['$or'][0]

        return filtered_spec

    @staticmethod
    def _spec_has_file_specific_filter(spec):
        return 'data.mt' in spec

    def _build_queries_by_arbitary_spec(self, spec):
        # пробуем сварить запрос с помощью генератора, если и это не удалось - кидаем NotImplementerError
        uid = spec['uid']
        spec = self._filter_mongo_specific_extra_parameters(spec)

        select_files, select_folders = True, True
        if 'type' in spec:
            resource_type = spec.pop('type')
            if resource_type == 'file':
                select_folders = False
            elif resource_type == 'dir':
                select_files = False
            else:
                raise NotImplementedError('spec contains unknown type: %s' % resource_type)

        files_query = None
        try:
            if select_files:
                files_subquery, files_params = MongoQueryConverter(FileDAOItem).find_to_sql(spec)
                files_params.update({'uid': uid})
                files_query = QueryWithParams(SQL_FILES_WITH_SELECT_TEMPLATE % files_subquery, files_params)
        except Exception:
            pass  # кинем другое в конце функции, если ни одного запроса варить не удалось

        folders_query = None
        try:
            if select_folders:
                folders_subquery, folders_params = MongoQueryConverter(FolderDAOItem).find_to_sql(spec)
                folders_params.update({'uid': uid})
                folders_query = QueryWithParams(SQL_FOLDERS_WITH_SELECT_TEMPLATE % folders_subquery, folders_params)
        except Exception:
            pass  # кинем другое в конце функции, если ни одного запроса варить не удалось

        if files_query is None and folders_query is None:
            raise NotImplementedError('spec "%s" is not supported now' % set(spec.keys()))

        return files_query, folders_query

    def get_last_files(self, uid, limit=10):
        session = Session.create_from_uid(uid)
        service_limit = 4000
        file_rows_cursor = session.execute(SQL_LAST_FILES_BY_UID, {'uid': uid, 'limit': limit, 'service_limit': service_limit})
        return (FileDAOItem.create_from_pg_data(item) for item in file_rows_cursor)

    def get_last_files_for_subtree(self, uid, path, limit=10):
        session = Session.create_from_uid(uid)
        file_rows_cursor = session.execute(SQL_LAST_FILES_FOR_SUBTREE, {'uid': uid, 'limit': limit, 'path': path})
        return (FileDAOItem.create_from_pg_data(item) for item in file_rows_cursor)

    def link_live_photo_files(self, uid, photo_path, video_path):
        session = Session.create_from_uid(uid)
        with session.begin():
            photo_files_cursor = session.execute(SQL_SELECT_FILE_BY_PATH_FOR_UPDATE, {
                'uid': uid, 'path': photo_path
            })
            video_files_cursor = session.execute(SQL_FILE_BY_PATH, {
                'uid': uid, 'path': video_path
            })
            photo_file = photo_files_cursor.fetchone()
            video_file = video_files_cursor.fetchone()
            session.execute(SQL_INSERT_ADDITIONAL_STORAGE_FILE_LINK, {
                'uid': uid,
                'main_file_fid': photo_file['fid'],
                'additional_file_fid': video_file['fid'],
                'type': AdditionalFileTypes.live_video.value
            })
            session.execute(SQL_SET_LIVE_PHOTO_FILE_FLAG, {
                'uid': uid,
                'fid': photo_file['fid'],
            })

    def unlink_live_photo_files(self, uid, photo_path, video_path):
        session = Session.create_from_uid(uid)
        with session.begin():
            session.execute(SQL_UNSET_LIVE_PHOTO_FILE_FLAG, {
                'uid': uid,
                'photo_path': photo_path,
            })
            session.execute(SQL_REMOVE_ADDITIONAL_STORAGE_FILE_LINK, {
                'uid': uid,
                'main_file_path': photo_path,
                'additional_file_path': video_path,
                'type': AdditionalFileTypes.live_video.value
            })

    def count_live_photo_files(self, uid, photo_hid):
        session = Session.create_from_uid(uid)

        _, photo_storage_id = FileDAOItem.convert_mongo_value_to_postgres_for_key('hid', photo_hid)
        root_folder = self._get_folder_for_collection(self.collection_name)

        files_count = session.execute(SQL_FILE_COUNT_BY_HID_FOR_UID, {
            'uid': uid, 'hid': photo_storage_id, 'root_folder': root_folder
        }).fetchone()[0]
        return files_count

    def get_live_photo_file(self, uid, photo_hid):
        session = Session.create_from_uid(uid)

        _, photo_storage_id = FileDAOItem.convert_mongo_value_to_postgres_for_key('hid', photo_hid)
        root_folder = self._get_folder_for_collection(self.collection_name)

        file_data = session.execute(SQL_FILES_BY_HID_FOR_UID, {
            'uid': uid, 'hid': photo_storage_id, 'root_folder': root_folder
        }).fetchone()

        return FileDAOItem.create_from_pg_data(file_data)


class MongoResourceDAOImplementation(ResourceDAOImplementation):
    def __init__(self, collection_name):
        self._mongo_helper = MongoHelper()
        self.collection_name = collection_name

    @property
    def user_data(self):
        return self._mongo_helper.get_collection(self.collection_name)

    def find(self, spec=None, fields=None, skip=0, limit=0, sort=None, **kwargs):
        if spec is not None:
            spec = self._convert_path_to_id(spec)
        kwargs.update({
            'fields': fields,
            'skip': skip,
            'limit': limit,
            'sort': sort
        })
        return self._mongo_helper.get_collection(self.collection_name).find(spec, **kwargs)

    def insert(self, doc_or_docs, manipulate=True, continue_on_error=False, **kwargs):
        doc_or_docs = self._convert_path_to_id(doc_or_docs)
        kwargs.update({
            'manipulate': manipulate,
            'continue_on_error': continue_on_error
        })
        return self._mongo_helper.get_collection(self.collection_name).insert(doc_or_docs, **kwargs)

    def remove(self, spec_or_id=None, multi=True, **kwargs):
        spec_or_id = self._convert_path_to_id(spec_or_id)
        kwargs.update({
            'multi': multi
        })
        return self._mongo_helper.get_collection(self.collection_name).remove(spec_or_id, **kwargs)

    def update(self, spec, document, upsert=False, multi=False, **kwargs):
        spec = self._convert_path_to_id(spec)
        document = self._convert_path_to_id(document)
        kwargs.update({
            'document': document,
            'upsert': upsert,
            'multi': multi
        })
        return self._mongo_helper.get_collection(self.collection_name).update(spec, **kwargs)

    def find_on_shard(self, spec=None, fields=None, skip=0, limit=0, sort=None, shard_name=None, **kwargs):
        if not shard_name:
            raise ValueError('Shard name is not specified')

        spec = self._convert_path_to_id(spec)
        kwargs.update({
            'fields': fields,
            'skip': skip,
            'limit': limit,
            'sort': sort,
        })
        return self._mongo_helper.get_collection(self.collection_name, shard_name).find(spec, **kwargs)

    def find_stids_on_shard(self, stids, shard_name, limit=None):
        spec = {
            'data.stids': {
                '$elemMatch': {'stid': {'$in': stids}}
            }
        }
        return self.find_on_shard(spec, {'data.stids': True}, shard_name=shard_name, limit=limit)

    def get_by_path(self, uid, path, read_preference=ReadPreference.PRIMARY):
        spec = self._get_mongo_spec(uid, path)
        if isinstance(path, (list, tuple)):
            cursor = self.find(spec, read_preference=read_preference)
            if cursor is None:
                return None
            return [self.create_dao_object_from_resource(r) for r in cursor]
        else:
            cursor = self.find(spec, read_preference=read_preference)
            if cursor is None:
                return None
            for r in cursor:
                return self.create_dao_object_from_resource(r)

    def find_snapshot_initial_chunk(self, uid):
        """Найти все документы пользователя, у которых нет file_id, кроме `/`.

        `/` фильтруем по продуктовым требованиям.

        Не смотря на то, что мы считаем, что у всех ресурсов должны быть `file_id` это не так.
        Как минимум у `/` и `/disk` нет `file_id`.
        """
        cursor = self.find(
            {'uid': uid, 'data.file_id': {'$lte': None}, 'key': {'$ne': '/'}},
            read_preference=ReadPreference.SECONDARY_PREFERRED)
        return cursor

    def find_snapshot_chunk(self, uid, file_id, last_ids, _, limit):
        cursor = self.find(
            {'uid': uid, 'data.file_id': {'$gte': file_id}, '_id': {'$nin': last_ids}},
            limit=limit,
            read_preference=ReadPreference.SECONDARY_PREFERRED)
        cursor.sort([('data.file_id', ASCENDING)])
        return cursor

    @staticmethod
    def create_dao_object_from_resource(resource):
        if resource['type'] == 'file':
            return FileDAOItem.create_from_mongo_dict(resource)
        elif resource['type'] == 'dir':
            return FolderDAOItem.create_from_mongo_dict(resource)
        raise TypeError('Unexpected resource type (%s), file or dir expected' % resource['type'])

    def find_one_file_on_shard_by_hid(self, hid, shard_name):
        raise NotImplementedError()

    def find_one_file_item_on_shard_by_hid(self, hid, shard_name):
        # чтобы не поднимались доки с не раззипованными stids
        spec = {
            'hid': hid,
            'data.stids': {'$exists': True},
        }
        for coll_name in ('user_data', 'trash', 'attach_data', 'photounlim_data', 'additional_data', 'client_data'):
            collection = self._mongo_helper.get_routed_collection_for_shard(shard_name, coll_name)
            cursor = collection.find(spec, limit=1)
            for data in cursor:
                return self.create_dao_object_from_resource(data)
        return None

    def find_one_file_by_uid_and_hid(self, uid, hid):
        coll = self._mongo_helper.get_routed_collection_for_uid(uid, 'user_data')
        return coll.find_one({'uid': uid, 'hid': hid})

    def find_by_uid_and_hids(self, uid, hids, limit, offset):
        coll = self._mongo_helper.get_routed_collection_for_uid(uid, self.collection_name)
        mongo_hids = [Binary(hid) for hid in hids]
        resources = coll.find({'uid': uid, 'hid': {'$in': mongo_hids}},
                              sort=[('hid', DESCENDING)]).skip(offset).limit(limit)
        return (FileDAOItem.create_from_mongo_dict(x) for x in resources)

    def is_one_owner(self, uid, hid):
        for coll in self._mongo_helper.iter_over_all_shards('user_data'):
            doc = coll.find_one({'uid': {'$ne': uid}, 'hid': hid})
            if doc:
                return False
        return True

    @staticmethod
    def _convert_path_to_id(spec):
        if spec is None:
            return None
        elif not isinstance(spec, list):
            specs = [spec]
        else:
            specs = spec

        results = []
        for s in specs:
            result = copy.copy(s)

            path = result.get('path')
            uid = result.get('uid')
            if path is not None and uid is not None:
                result.pop('path')
                if isinstance(path, dict) and len(path) == 1 and path.keys()[0] == '$in':
                    assert all([p.startswith('/') for p in path['$in']])
                    result['_id'] = {'$in': [id_for_key(uid, p) for p in path['$in']]}
                elif isinstance(path, basestring):
                    assert path.startswith('/')
                    result['_id'] = id_for_key(uid, path)
                else:
                    raise NotImplementedError()

            parent = result.get('parent')
            if parent is not None and uid is not None:
                assert parent.startswith('/')
                result.pop('parent')
                result['parent'] = id_for_key(uid, parent)

            results.append(result)

        if not isinstance(spec, list):
            return results[0]
        else:
            return results

    def aggregate(self, query):
        return self.user_data.aggregate(query)

    @staticmethod
    def _get_mongo_spec(uid, path):
        if isinstance(path, (list, tuple)):
            return {
                '_id': {'$in': [hashed(uid + ':' + p) for p in path]},
                'uid': uid
            }
        else:
            return {
                '_id': hashed(uid + ':' + path),
                'uid': uid
            }

    def get_last_files(self, uid, limit=10):
        spec = {'uid': uid, 'type': 'file'}
        opts = {
            'read_preference': ReadPreference.SECONDARY_PREFERRED,
            'limit': limit
        }
        cursors = []
        for coll_name in (self.collection_name, PhotounlimDAO.mongo_coll_name):
            cursor = self._mongo_helper.get_collection(coll_name).find(spec, **opts)
            cursor.sort([('data.mtime', DESCENDING)])
            cursors.append(cursor)
        joined_data = itertools.islice(
            merge2(cursors, key=lambda r: r['data'].get('mtime'), reverse=True),
            0, limit
        )
        return (self.create_dao_object_from_resource(r) for r in joined_data)

    def get_last_files_for_subtree(self, uid, path, limit=10):
        if not path.endswith('/'):
            path += '/'

        FETCH_LIMIT = 1000  # Сколько документов изначально выбираем для фильтрации
        spec = [
            {'$match': {'uid': uid, 'type': 'file'}},
            {'$sort': {"data.mtime": DESCENDING}},
            {'$limit': FETCH_LIMIT},
            {'$match': {'key': {'$regex': '^%s' % re.escape(path)}}},
            {'$limit': limit}
        ]

        routed_coll = self._mongo_helper.get_routed_collection_for_uid(uid, ResourceDAO.mongo_coll_name)
        # Старый pymongo для `aggregate` возвращает словарь со значениями, а новый - курсор
        # https://api.mongodb.com/python/2.6.3/api/pymongo/collection.html#pymongo.collection.Collection.aggregate
        aggregate_result = routed_coll.aggregate(spec, read_preference=ReadPreference.SECONDARY_PREFERRED)
        if isinstance(aggregate_result, dict):
            aggregate_result = aggregate_result['result']
        return (self.create_dao_object_from_resource(r) for r in aggregate_result)

    def link_live_photo_files(self, uid, photo_path, video_path):
        spec = {'uid': uid, 'path': photo_path}
        live_video_id = hashed(uid + ':' + video_path)
        self.update(spec, {
            '$set': {'data.live_video_id': live_video_id, 'data.is_live_photo': True}
        })

    def unlink_live_photo_files(self, uid, photo_path, video_path):
        live_video_id = hashed(uid + ':' + video_path)
        spec = {'uid': uid, 'path': photo_path, 'data.live_video_id': live_video_id}
        self.update(spec, {
            '$unset': {'data.live_video_id': True, 'data.is_live_photo': True}
        })

    def count_live_photo_files(self, uid, photo_hid):
        spec = {'uid': uid, 'hid': Binary(str(photo_hid))}
        cursor = self.find(spec)
        files_count = cursor.count()
        return files_count

    def get_live_photo_file(self, uid, photo_hid):
        spec = {'uid': uid, 'hid': Binary(str(photo_hid))}  # аналогично store в photostream, там тоже идет fullscan
        cursor = self.find(spec)
        return self.create_dao_object_from_resource(cursor[0])


class TrashDAO(ResourceDAO):
    mongo_coll_name = 'trash'


class MongoHiddenDAOImplementation(MongoResourceDAOImplementation):
    def find_files_for_cleaning(self, shard_name, min_file_size, max_datetime, filter_uids=None, limit=None):
        if filter_uids is not None or limit is not None:
            raise NotImplementedError('Unsupported filter_uids or limit parameter for mongo queries')

        max_ts = int(time.mktime(max_datetime.timetuple()))

        collection = self._mongo_helper.get_routed_collection_for_shard(shard_name, self.collection_name)
        spec = {
            'data.size': {'$gte': min_file_size},
            'dtime': {'$lte': max_ts}
        }
        return collection.find(spec, sort=[('data.size', DESCENDING)])

    def remove_files_by_uid_id(self, shard_name, uid_id_list):
        collection = self._mongo_helper.get_routed_collection_for_shard(shard_name, self.collection_name)
        spec = {'_id': {'$in': [i[1] for i in uid_id_list]}}
        return collection.remove(spec)

    def remove_hanging_storage_files_by_ids(self, shard_name, storage_ids):
        pass

    def unset_live_photo_flag_by_uid_id(self, shard_name, uid_id_list):
        pass


class PostgresHiddenDAOImplementation(PostgresResourceDAOImplementation):
    def find_files_for_cleaning(self, shard_name, min_file_size, max_datetime, filter_uids=None, limit=None):
        session = Session.create_from_shard_id(shard_name)

        if filter_uids:
            postgres_uids = tuple(FileDAOItem.convert_mongo_value_to_postgres_for_key('uid', uid)[1] for uid in filter_uids)
            files_query = QueryWithParams(SQL_FILES_BY_SIZE_AND_DTIME_AND_NOT_IN_UIDS,
                                          {'dtime_lte': max_datetime, 'size_gte': min_file_size,
                                           'uids_nin': postgres_uids, 'limit': limit})
        else:
            files_query = QueryWithParams(SQL_FILES_BY_SIZE_AND_DTIME,
                                          {'dtime_lte': max_datetime, 'size_gte': min_file_size, 'limit': limit})

        result_proxy = session.execute(files_query.query, **files_query.params)
        for data in result_proxy:
            yield FileDAOItem.create_from_pg_data(data)

    def remove_files_by_uid_id(self, shard_name, uid_id_list):
        converter = FileDAOItem.convert_mongo_value_to_postgres_for_key

        uid_ids = tuple([
            (converter('uid', uid_id[0])[1], converter('_id', uid_id[1])[1])
            for uid_id in uid_id_list]
        )

        session = Session.create_from_shard_id(shard_name)
        with session.begin():
            try:
                session.execute(SQL_REMOVE_FILES_BY_UID_FID, {'uid_id_pairs': uid_ids})
            except Exception:
                log.exception('Something wrong')

    def remove_hanging_storage_files_by_ids(self, shard_name, storage_ids):
        session = Session.create_from_shard_id(shard_name)
        with session.begin():
            storage_files_deleted = FileDAO(session).remove_hanging_storage_files(storage_ids)
            if storage_files_deleted is None:
                return []
            return [r[0] for r in storage_files_deleted]

    def unset_live_photo_flag_by_uid_id(self, shard_name, uid_id_list):
        converter = FileDAOItem.convert_mongo_value_to_postgres_for_key

        uid_ids = tuple([
            (converter('uid', uid_id[0])[1], converter('_id', uid_id[1])[1])
            for uid_id in uid_id_list]
        )

        session = Session.create_from_shard_id(shard_name)
        return session.execute(SQL_UNSET_LIVE_PHOTO_FILE_FLAGS, {'uid_id_pairs': uid_ids})


class HiddenDAO(ResourceDAO):
    mongo_coll_name = 'hidden_data'

    def __init__(self):
        super(HiddenDAO, self).__init__()
        self._mongo_impl = MongoHiddenDAOImplementation(self.mongo_coll_name)
        self._pg_impl = PostgresHiddenDAOImplementation(self.mongo_coll_name)

    def find_files_for_cleaning(self, shard_name, min_file_size, max_datetime, filter_uids=None, limit=None):
        # запрос от чистки hidden_data
        impl = self._get_impl_by_shard(shard_name)
        return impl.find_files_for_cleaning(shard_name, min_file_size, max_datetime, filter_uids, limit)

    def remove_files_by_uid_id(self, shard_name, uid_id_list):
        impl = self._get_impl_by_shard(shard_name)
        return impl.remove_files_by_uid_id(shard_name, uid_id_list)

    def remove_hanging_storage_files_by_ids(self, shard_name, stids_ids):
        impl = self._get_impl_by_shard(shard_name)
        return impl.remove_hanging_storage_files_by_ids(shard_name, stids_ids)

    def unset_live_photo_flag_by_uid_id(self, shard_name, uid_id_list):
        impl = self._get_impl_by_shard(shard_name)
        return impl.unset_live_photo_flag_by_uid_id(shard_name, uid_id_list)


class AttachDAO(ResourceDAO):
    mongo_coll_name = 'attach_data'


class NotesDAO(ResourceDAO):
    mongo_coll_name = 'notes_data'


class NarodDAO(ResourceDAO):
    mongo_coll_name = 'narod_data'


class PhotounlimDAO(ResourceDAO):
    mongo_coll_name = 'photounlim_data'


class ClientDAO(ResourceDAO):
    mongo_coll_name = 'client_data'


class MongoAdditionalDAOImplementation(MongoHiddenDAOImplementation):
    def get_additional_file(self, main_file_address, additional_file_type):
        spec = {'uid': main_file_address.uid, '_id': hashed(main_file_address.uid + ':' + main_file_address.path)}

        dao = get_dao_by_address(main_file_address)
        main_file = dao.find_one(spec)

        spec = {'uid': main_file_address.uid, '_id': main_file['data']['live_video_id']}
        item = self.find_one(spec)

        return self.create_dao_object_from_resource(item)

    def find_files_by_uid_id_on_shard(self, shard_name, uid_id_list):
        collection = self._mongo_helper.get_routed_collection_for_shard(shard_name, self.collection_name)
        spec = {'_id': {'$in': [i[1] for i in uid_id_list]}}
        cursor = collection.find(spec)
        return [FileDAOItem.create_from_mongo_dict(i) for i in cursor]

    def get_video_ids_by_file_ids_on_shard(self, shard_name, uid_id_list):
        raise NotImplementedError()

    def remove_links_by_uid_id_on_shard(self, shard_name, uid_id_list):
        pass  # ничего не нужно делать на монге


class PostgresAdditionalDAOImplementation(PostgresHiddenDAOImplementation):
    def get_additional_file(self, main_file_address, additional_file_type):
        session = Session.create_from_uid(main_file_address.uid)

        file_data = session.execute(SQL_GET_ADDITIONAL_FILE_BY_MAIN_FILE_PATH_AND_TYPE, {
            'uid': main_file_address.uid, 'path': main_file_address.path, 'type': additional_file_type.value
        }).fetchone()

        return FileDAOItem.create_from_pg_data(file_data)

    def find_files_by_uid_id_on_shard(self, shard_name, uid_id_list):
        converter = FileDAOItem.convert_mongo_value_to_postgres_for_key

        uid_ids = tuple([
            (converter('uid', uid_id[0])[1], converter('_id', uid_id[1])[1])
            for uid_id in uid_id_list]
        )

        session = Session.create_from_shard_id(shard_name)
        cursor = session.execute(SQL_GET_FILES_BY_UID_FID, {'uid_id_pairs': uid_ids})

        return [FileDAOItem.create_from_pg_data(i) for i in cursor]

    def get_video_ids_by_file_ids_on_shard(self, shard_name, uid_id_list):
        converter = FileDAOItem.convert_mongo_value_to_postgres_for_key

        uid_ids = tuple([
            (converter('uid', uid_id[0])[1], converter('_id', uid_id[1])[1])
            for uid_id in uid_id_list]
        )

        session = Session.create_from_shard_id(shard_name)
        cursor = session.execute(SQL_GET_ADDITIONAL_FILE_UID_FID_BY_MAIN_UID_FID, {'uid_id_pairs': uid_ids})
        return [
            (str(i['uid']), i['additional_file_fid'].hex)
            for i in cursor
        ]

    def remove_links_by_uid_id_on_shard(self, shard_name, uid_id_list):
        converter = FileDAOItem.convert_mongo_value_to_postgres_for_key

        uid_ids = tuple([
            (converter('uid', uid_id[0])[1], converter('_id', uid_id[1])[1])
            for uid_id in uid_id_list]
        )

        session = Session.create_from_shard_id(shard_name)
        session.execute(SQL_REMOVE_ADDITIONAL_FILE_LINK_BY_UID_FID, {'uid_id_pairs': uid_ids})


class AdditionalDataDAO(ResourceDAO):
    mongo_coll_name = 'additional_data'

    LIVE_VIDEO_TYPE = AdditionalFileTypes.live_video

    def __init__(self):
        super(AdditionalDataDAO, self).__init__()
        self._mongo_impl = MongoAdditionalDAOImplementation(self.mongo_coll_name)
        self._pg_impl = PostgresAdditionalDAOImplementation(self.mongo_coll_name)

    def get_additional_file(self, main_file_address, additional_file_type):
        assert additional_file_type == AdditionalDataDAO.LIVE_VIDEO_TYPE

        impl = self._get_impl(main_file_address.uid)
        return impl.get_additional_file(main_file_address, additional_file_type)

    def insert_additional_file(self, additional_file):
        impl = self._get_impl(additional_file.uid)

        mongo_dict = additional_file.get_mongo_representation()
        mongo_dict['_id'] = id_for_key(mongo_dict['uid'], mongo_dict['key'])
        mongo_dict['parent'] = parent_for_key(mongo_dict['key'])

        impl.insert(mongo_dict)

    def remove_additional_file(self, additional_file):
        impl = self._get_impl(additional_file.uid)

        mongo_dict = additional_file.get_mongo_representation()
        impl.remove({'uid': mongo_dict['uid'], 'path': mongo_dict['key']})

    def link_additional_file(self, uid, main_file_path, additional_file_path, additional_file_type):
        assert additional_file_type == AdditionalDataDAO.LIVE_VIDEO_TYPE

        dao = get_dao_by_address(Address.Make(uid, main_file_path))
        dao.link_live_photo_files(uid, main_file_path, additional_file_path)

    def unlink_additional_file(self, uid, main_file_path, additional_file_path):
        dao = get_dao_by_address(Address.Make(uid, main_file_path))
        dao.unlink_live_photo_files(uid, main_file_path, additional_file_path)

    def find_files_by_uid_id_on_shard(self, shard_name, uid_id_list):
        impl = self._get_impl_by_shard(shard_name)
        return impl.find_files_by_uid_id_on_shard(shard_name, uid_id_list)

    def get_video_ids_by_file_ids_on_shard(self, shard_name, uid_id_list):
        impl = self._get_impl_by_shard(shard_name)
        return impl.get_video_ids_by_file_ids_on_shard(shard_name, uid_id_list)

    def remove_links_by_uid_id_on_shard(self, shard_name, uid_id_list):
        impl = self._get_impl_by_shard(shard_name)
        return impl.remove_links_by_uid_id_on_shard(shard_name, uid_id_list)

    def remove_files_by_uid_id_on_shard(self, shard_name, uid_id_list):
        impl = self._get_impl_by_shard(shard_name)
        return impl.remove_files_by_uid_id(shard_name, uid_id_list)


STORAGE_PATH_DAO_CLS_MAP = {
    '/disk': ResourceDAO,
    '/trash': TrashDAO,
    '/hidden': HiddenDAO,
    PHOTOUNLIM_AREA_PATH: PhotounlimDAO,
}


def get_dao_by_address(address):
    dao_cls = STORAGE_PATH_DAO_CLS_MAP.get(address.storage_path)
    if not dao_cls:
        raise ValueError(
            'invalid photo path `%s` (only %s is supported)' % (address, ', '.join(STORAGE_PATH_DAO_CLS_MAP.keys()))
        )
    return dao_cls()
