from mpfs.core.filesystem.dao.file import StidsListParser
from mpfs.dao.base import (
    BaseDAOItem,
    BaseDAO,
    QueryWithParams)
from mpfs.dao.cursor import PostgresCursor
from mpfs.dao.fields import (HidField, StidField)
from mpfs.dao.session import Session
from mpfs.metastorage.postgres.queries import (
    SQL_HANGING_STORAGE_FILE_DUPLICATES,
    SQL_STIDS_BY_STIDS_IN_DUPLICATED_STORAGE_FILES, SQL_DELETE_STORAGE_FILE_DUPLICATES_BY_STORAGE_IDS)
from mpfs.metastorage.postgres.schema import duplicated_storage_files


class DuplicatedStorageFilesDAOItem(BaseDAOItem):
    storage_id = HidField(mongo_path='storage_id', pg_path=duplicated_storage_files.c.storage_id)
    stid = StidField(mongo_path='data.stids', mongo_item_parser=StidsListParser('duplicate_stid'),
                     pg_path=duplicated_storage_files.c.stid)
    columns_map = {c.name: c for c in duplicated_storage_files.columns}


class StorageDuplicatesDAO(BaseDAO):
    dao_item_cls = DuplicatedStorageFilesDAOItem

    def find_hanging_duplicates(self, shard_name):
        session = Session.create_from_shard_id(shard_name)
        return session.execute(SQL_HANGING_STORAGE_FILE_DUPLICATES)

    def delete(self, storage_ids, shard_name):
        if not storage_ids:
            return
        session = Session.create_from_shard_id(shard_name)
        session.execute(SQL_DELETE_STORAGE_FILE_DUPLICATES_BY_STORAGE_IDS, {'storage_ids_in': tuple(storage_ids)})

    def find_stids_on_shard(self, stids, shard_name, limit=None):
        if not stids:
            return {}
        session = Session.create_from_shard_id(shard_name)
        duplicates_query = QueryWithParams(SQL_STIDS_BY_STIDS_IN_DUPLICATED_STORAGE_FILES,
                                           {'stids': tuple(stids)})
        return PostgresCursor(session, duplicates_query, DuplicatedStorageFilesDAOItem)
