# -*- coding: utf-8 -*-
import datetime
import hashlib

from mpfs.dao.base import BaseDAO, BaseDAOItem, PostgresBaseDAOImplementation, QueryWithParams, \
    MongoBaseDAOImplementation
from mpfs.dao.cursor import PostgresCursor
from mpfs.dao.fields import StringField, UidField, DateField
from mpfs.dao.migration.migration import BaseCommonCollectionMigration
from mpfs.metastorage.postgres.queries import SQL_FIND_EXPIRED_TRASH_CLEANER_TASKS, SQL_INSERT_TRASH_CLEANER_QUEUE
from mpfs.metastorage.postgres.schema import trash_cleaner_queue


class TrashCleanerQueueDAOItem(BaseDAOItem):
    mongo_collection_name = 'trash_cleaner_queue'
    postgres_table_obj = trash_cleaner_queue
    is_sharded = False

    id = StringField(mongo_path='_id', pg_path=trash_cleaner_queue.c.id)
    date = DateField(mongo_path='date', pg_path=trash_cleaner_queue.c.bound_date)
    uid = UidField(mongo_path='uid', pg_path=trash_cleaner_queue.c.uid)


class TrashCleanerQueueDAO(BaseDAO):
    dao_item_cls = TrashCleanerQueueDAOItem

    def __init__(self):
        super(TrashCleanerQueueDAO, self).__init__()
        self._mongo_impl = MongoTrashCleanerQueueDAOImplementation(self.dao_item_cls)
        self._pg_impl = PostgresTrashCleanerQueueDAOImplementation(self.dao_item_cls)

    def find_expired(self, bound_date):
        return self._get_impl(None).find_expired(bound_date)

    def save(self, task, **kwargs):
        if '_id' not in task:
            task['_id'] = hashlib.md5("%s_%s" % (task['uid'], task['date'])).hexdigest()
        self._check_write_options(kwargs)
        return self._get_impl(None).save(task)


class PostgresTrashCleanerQueueDAOImplementation(PostgresBaseDAOImplementation):
    def find_expired(self, bound_date):
        session = self._get_session()
        if isinstance(bound_date, int):
            bound_date = datetime.datetime.strptime(str(bound_date), "%Y%m%d").date()
        return PostgresCursor(
            session,
            QueryWithParams(SQL_FIND_EXPIRED_TRASH_CLEANER_TASKS, {'bound_date': bound_date}),
            self.dao_item_cls
        )

    def save(self, task, **kwargs):
        session = self._get_session()
        with session.begin():
            session.execute(SQL_INSERT_TRASH_CLEANER_QUEUE,
                            self.dao_item_cls.create_from_mongo_dict(task).as_raw_pg_dict())


class MongoTrashCleanerQueueDAOImplementation(MongoBaseDAOImplementation):
    def find_expired(self, bound_date):
        return super(MongoTrashCleanerQueueDAOImplementation, self).find({"date": {"$lt": bound_date}})

    def save(self, task, **kwargs):
        return super(MongoTrashCleanerQueueDAOImplementation, self).update(
            dict(uid=task['uid'], date=task['date']), task, upsert=True, **kwargs
        )


class TrashCleanerQueueMigration(BaseCommonCollectionMigration):
    dao_cls = TrashCleanerQueueDAO
