# -*- coding: utf-8 -*-
"""

MPFS
CORE

Хардлинки

"""
import hashlib
from copy import deepcopy

import mpfs.engine.process

from mpfs.config import settings
from mpfs.common import errors
from mpfs.core.metastorage.control import disk, trash, attach, lnarod, photounlim
from mpfs.core.services import mulca_service
from mpfs.core.user.constants import EMPTY_FILE_HASH_SHA256, EMPTY_FILE_HASH_MD5
from mpfs.metastorage.mongo.binary import Binary

log = mpfs.engine.process.get_default_log()
error_log = mpfs.engine.process.get_error_log()

SERVICES_MULCA_EMPTY_FILE_MID = settings.services['mulca']['empty_file_mid']
SERVICES_MULCA_EMPTY_FILE_DIGEST_MID = settings.services['mulca']['empty_file_digest_mid']


class FileChecksums(object):
    def __init__(self, md5, sha256, size):
        self.md5 = md5
        self.sha256 = sha256
        self.size = size
        if (not self.md5 or
                not self.sha256 or
                not isinstance(self.size, (int, long))):
            raise ValueError('Bad checksums: %r' % self.as_dict())

    @classmethod
    def from_str(cls, hashes_str):
        result = None
        try:
            md5, sha256, size = [item.split(':')[1]
                                 for item in hashes_str.strip().split(',')]
            size = long(size)
        except Exception:
            pass
        else:
            result = FileChecksums(md5, sha256, size)

        return result

    @property
    def hid(self):
        return construct_hid(self.md5, self.size, self.sha256)

    @property
    def binary_hid(self):
        return Binary(self.hid)

    def copy(self):
        return self.__class__(**self.as_dict())

    def as_dict(self):
        return {
            'md5': self.md5,
            'sha256': self.sha256,
            'size': self.size,
        }

    def __ne__(self, other):
        return not self == other

    def __eq__(self, other):
        if not isinstance(other, self.__class__):
            return False
        return (self.md5 == other.md5 and
                self.sha256 == other.sha256 and
                self.size == other.size)

    def __repr__(self):
        return "%s(md5=%r, sha256=%r, size=%r, hid=%r)" % (
            self.__class__.__name__,
            self.md5,
            self.sha256,
            self.size,
            self.hid,
        )


def construct_hid(md5, size, sha256):
    '''
    Конструирование полного id
    Перенесено из mongo.py
    '''
    historical_id = '/%s_%s_%s' % (sha256, md5, size)
    system_user   = mpfs.engine.process.system_user()
    key_string    = '%s:%s' % (system_user, historical_id)
    return hashlib.md5(key_string).hexdigest()


class AbstractLink(object):
    """
    Новый класс хардлинка
    Чистая виртуальность, надстройка над таблицами hidden, trash, attach, narod и user_data
    """
    SEARCH_COLLECTIONS = (disk, trash, attach, lnarod)
    EMPTY_FILE_HID = construct_hid(EMPTY_FILE_HASH_MD5, 0, EMPTY_FILE_HASH_SHA256)

    DICT_FIELDS = ['md5', 'size', 'sha256', 'file_id', 'digest_id', 'thumbnail_id', 'preview_id', 'previews',
                   'mimetype', 'drweb', 'pmid', 'emid', 'etime', 'video_info', 'width', 'height', 'angle', 'aesthetics']

    EXCLUDE_META_FIELDS = ['id']

    md5 = None
    size = None
    sha256 = None
    file_id = None  # id файла
    digest_id = None    # id дайджеста
    thumbnail_id = None
    preview_id = None
    previews = {}
    mimetype = None
    drweb = None
    pmid = None
    emid = None
    etime = None
    video_info = None
    width = None
    height = None
    angle = None
    aesthetics = None

    _id = ''

    # raw record returned
    _raw_record = None

    _empty_vals = (None, '')

    def __init__(self, md5=None, size=None, sha256=None, load=True, hid=None, *args, **kwargs):
        """__init__(self, md5, size, sha256, load=True, hid='a1b2c3')"""
        self._id = hid or construct_hid(md5, size, sha256)
        if not load:
            return

        ev = self._empty_vals
        is_hid_sources_provided = self.is_valid_checksums(md5, sha256, size)
        if not hid and not is_hid_sources_provided:
            raise errors.HardLinkNotFound()

        # проверка соответствия хардлинковых данных переданным (плохой hid)
        self.load(self._id)
        if is_hid_sources_provided:
            if md5 != self.md5 or int(size) != self.size or sha256 != self.sha256:
                raise errors.HardLinkNotFound()

    @classmethod
    def is_valid_checksums(cls, md5, sha256, size):
        ev = cls._empty_vals
        return not (md5 in ev or size in ev or sha256 in ev)

    @staticmethod
    def is_file_in_storage(mid=None):
        """
        Проверка наличия файла в мульке

        Cчитаем, что файл в хранилище, если мулька явно об этом сказала
        """
        mulca = mulca_service.Mulca()
        file_in_storage = False
        try:
            file_in_storage = mulca.is_file_exist(mid)
            msg = 'ok'
        except mulca.api_error:
            msg = 'mulca bad reponse'
        except Exception as e:
            msg = str(e)
            error_log.exception(msg)

        log.info('is_file_in_storage. Stid: "%s", in storage: "%s", error msg: "%s"' % (mid, file_in_storage, msg))
        return file_in_storage

    def load(self, hid):
        """
        Load hardlink by its hid.
        :param hid:
        """
        self._id = hid
        if self._id == self.EMPTY_FILE_HID:
            self.file_id = SERVICES_MULCA_EMPTY_FILE_MID
            self.digest_id = SERVICES_MULCA_EMPTY_FILE_DIGEST_MID
            self.size = 0
            self.mimetype = 'text/plain'
            self.drweb = 0
            self.md5 = EMPTY_FILE_HASH_MD5
            self.sha256 = EMPTY_FILE_HASH_SHA256
            return

        data = self._search_by_hid(hid)
        self._raw_record = data

        self.md5 = data.md5
        self.size = data.size
        self.sha256 = data.sha256
        self.file_id = data.file_stid
        self.digest_id = data.digest_stid

        for main_field in (self.md5, self.size, self.sha256, self.file_id, self.digest_id):
            if main_field in self._empty_vals:
                raise errors.HardLinkNotFound()

        if hid != construct_hid(self.md5, self.size, self.sha256):
            raise errors.HardLinkNotFound()

        self.mimetype = data.mimetype
        self.drweb = data._fields['antivirus_status'].convert_av_scan_status_to_num(data.antivirus_status)
        self.pmid = data.preview_stid
        self.etime = data.exif_time.get_timestamp() if data.exif_time else 0
        self.video_info = data.video_info
        self.width = data.width
        self.height = data.height
        self.angle = data.angle
        self.aesthetics = data.aesthetics
        self.photoslice_album_type = data.photoslice_album_type
        self.albums_exclusions = data.albums_exclusions
        self.ext_coordinates = data.ext_coordinates

        if not self.mimetype:
            raise errors.HardlinkBroken()

        # проверка наличия файла в хранилище
        if not self.is_file_in_storage(self.file_id):
            raise errors.HardlinkFileNotInStorage(data={'mid': self.file_id})

    @classmethod
    def _search_by_stid(cls, stid):
        return cls._search_record('find_by_stid', (stid,))

    @classmethod
    def _search_by_hid(cls, hid):
        return cls._search_record('find_by_hid', (hid,))

    @classmethod
    def _search_record(cls, method, args):
        raise errors.MPFSNotImplemented()

    def get_resource(self):
        """
        Return resource represented by hardlink.

        :return: File or None
        """
        if not self._raw_record:
            return None

        mongo_dict = self._raw_record.get_mongo_representation()
        uid = mongo_dict['uid']
        path = mongo_dict['key']
        data = mongo_dict['data']
        # there is implicit import loop somewhere in mpfs/core/factory.py so import it locally
        from mpfs.core.factory import get_resource
        return get_resource(uid, path, data=data)

    def dir(self):
        # emulate old behaviour returning dict with initial empty values
        return dict([(a, getattr(self.__class__, a)) for a in self.DICT_FIELDS])

    def dict(self):
        result = {}
        attributes = self.dir()
        for attr in attributes.keys():
            result[attr] = deepcopy(getattr(self, attr))
        return result

    def file_part(self):
        res = self.get_resource()
        meta = res.meta.copy()
        for f in self.EXCLUDE_META_FIELDS:
            meta.pop(f, None)
        return {
            'name': res.name,
            'meta': meta,
            'size': self.size,
            'mimetype': self.mimetype,
        }

    def all_storage_ids(self):
        '''
        Вернуть все хранилищные id
        '''
        result = []
        for item in (self.digest_id, self.file_id, self.thumbnail_id, self.preview_id, self.pmid, self.emid):
            if item:
                result.append(item)
        for size, mid in self.previews.iteritems():
            if mid:
                result.append(mid)
        return result


    @classmethod
    def Create(classname, *args, **kwargs):
        raise errors.MPFSNotImplemented()


    def ListOwners(self, md5, size, sha256):
        hid = _construct_id(md5, size, sha256)
        result = []
        for element in disk.find_by_hid_all(hid):
            result.append(element['uid'])
        return result


    @classmethod
    def info_by_stid(cls, stid):
        data = cls._search_by_stid(stid)[1].value.data
        filemeta = data['meta']
        result = {
            'size': data['size'],
            'md5': filemeta['md5'],
            'sha256': filemeta['sha256'],
            'file_mid': filemeta['file_mid'],
            'digest_mid': filemeta['digest_mid'],
            'pmid': filemeta.get('pmid'),
            'emid': filemeta.get('emid'),
        }
        for k, v in result.iteritems():
            if stid == v:
                result['type'] = k
                break
        return result
