# -*- coding: utf-8 -*-
"""

MPFS
HELPERS

Хелпер локов

"""
import threading
import traceback
from copy import deepcopy

from functools import partial

import mpfs.engine.process

from mpfs.config import settings
from mpfs.core import factory
from mpfs.core.address import Address
from mpfs.common.errors import ResourceNotFound, ResourceLocked, OperationNotFound
from mpfs.core.metastorage.control import fs_locks
from mpfs.core.filesystem.resources.base import Resource

log = mpfs.engine.process.get_default_log()
error_log = mpfs.engine.process.get_error_log()


class LockHelper(object):

    @classmethod
    def resolve(cls, something, scope=None):
        """
        Метод резолвинга того, что отдали на лок
        Это может быть как адрес (строка), так и готовый ресурс
        В зависимости от скоупа применения разнятся действия

        :param something: что-то (ресурс или адрес)
        :param scope: скоуп применения
        :return: (address, resource/None) адрес Address и ресурс Resource, если будет найден
        """
        if scope == 'lock':
            if not isinstance(something, Resource):
                address = Address(something)
                resource = factory.get_resource(address.uid, address.id)
            else:
                resource = something
            address = resource.storage_address
        elif scope in ('check', 'unlock', 'update'):
            if not isinstance(something, Resource):
                address = Address(something)

                try:
                    resource = factory.get_resource(address.uid, address.id)
                    address = resource.storage_address
                except ResourceNotFound:
                    resource = None
                    address = cls._invited_to_owner_address(address)
            else:
                resource = something
                address = resource.storage_address

        return address, resource

    @classmethod
    def lock(cls, something, data=None, time_offset=0, operation=None):
        """
        Залочить ресурс или адрес

        Lock построен на TTL индексе, поэтому
        он автоматически снимается через некоторое время (1 мин.)

        :param something: что-то (ресурс или адрес)
        """
        LockHelper.check(something)

        address, resource = cls.resolve(something, scope='lock')
        log.info('locking %s' % address.id)
        data = cls.get_updated_lock_data_with_operation_type(data, operation)
        fs_locks.set(address.uid, address.path, data, time_offset)

    @staticmethod
    def _invited_to_owner_address(address):
        # адрес участника резолвим в адрес владельца
        from mpfs.core.social.share.processor import ShareProcessor
        group_links = ShareProcessor().list_owned_links(address.uid)
        for group_link in group_links:
            link_path = group_link.path if group_link.path.endswith('/') else group_link.path + '/'
            if address.path.startswith(link_path):
                owner = group_link.group.owner
                owner_path = "%s/%s" % (group_link.group.path.rstrip('/'), address.path[len(link_path):])
                address = Address.Make(owner, owner_path)
                break
        return address

    @classmethod
    def lock_by_address(cls, address, data=None, operation=None):
        """Залочить ресурс по уиду и адресу.

        В отличие от похожего метода lock, не требует наличия ресурса для блокировки.

        Lock построен на TTL индексе, поэтому
        он автоматически снимается через некоторое время (1 мин.)
        """
        address = cls._invited_to_owner_address(address)
        LockHelper.check('%s:%s' % (address.uid, address.path))

        log.info('locking %s:%s' % (address.uid, address.path))
        data = cls.get_updated_lock_data_with_operation_type(data, operation)
        fs_locks.set(address.uid, address.path, data)

    @classmethod
    def get_lock(cls, something):
        address, _ = cls.resolve(something, scope='lock')
        return fs_locks.get(address.uid, address.path)

    @classmethod
    def _get_details_from_lock_data(cls, lock_data):
        details = {}
        try:
            details['uid'] = lock_data['uid']
            details['path'] = lock_data['path']
            details['op_type'] = lock_data['op_type']
            if 'oid' in lock_data:
                details['oid'] = lock_data['oid']
        except:
            # Если не удалось получить детали - отдаем ошибку как есть (без деталей о локе)
            return

        return details

    @classmethod
    def _set_path_with_invited_path(cls, data, invited_uid, owner_uid, owner_path):
        try:
            from mpfs.core.social.share import Group
            data['path'] = Group.get_link_path(invited_uid, owner_uid, owner_path)
        except:
            # В случае любой ошибки ничего не делаем (ошибку в итоге сформируем без деталей)
            error_log.exception("Failed to get lock details: ")
            return

    @classmethod
    def check(cls, something, skip_self=False):
        """
        Версия от https://st.yandex-team.ru/CHEMODAN-19462:

        Получаем все залоченные ресурсы юзера и локи по ОП и далее решаем, что делать.

        Если передан существующий ресурс, то:
        - если внизу что-то происходит, то запрещено что-то делать на уровне выше
        - если вверху что-то происходит, то запрещено что-то делать на уровне выше

        Если передан несуществующий ресурс, то:
        - если вверху что-то происходит, то запрещено что-то делать на уровне выше

        Если ресурс принадлежит ОП, то:
        - если хозяин залочен, то запрещено что-то делать

        :param something: что-то (ресурс или адрес)
        """
        address, resource = cls.resolve(something, scope='check')

        contains = Address.path_contains_path
        equals = partial(contains, equals=True)

        # обработка если ресурс существует
        if resource:
            # получаем локи своего Диска
            self_locks = fs_locks.find_all(resource.uid)

            # собираем локи ОП
            shared_locks = {}
            all_locks = {}
            owner_is_writable = True
            if resource.is_shared:
                link = resource.link
                shared_locks = fs_locks.find_all(link.group.owner)
                owner_is_writable = mpfs.engine.process.usrctl().is_writable(link.group.owner)

            if not owner_is_writable:
                lock_msg = '%s: locked owner %s' % (resource.storage_address.id, link.group.owner)
                log.info("ResourceLocked: %s %s" % (lock_msg, " ".join(traceback.format_exc().splitlines()[-3:])))
                raise ResourceLocked(lock_msg)

            path = resource.storage_address.path
            matched = None
            all_locks.update(self_locks)
            all_locks.update(shared_locks)
            lock_data = None
            for locked_path in all_locks:
                if contains(path, locked_path) or contains(locked_path, path):
                    if skip_self and equals(path, locked_path):
                        continue
                    matched = locked_path
                    lock_data = all_locks[matched].get('data')
                    break

            if matched:
                details = None
                if lock_data:
                    lock_data['uid'] = all_locks[matched]['uid']
                    if matched in shared_locks:
                        cls._set_path_with_invited_path(lock_data,
                                                        resource.uid,
                                                        all_locks[matched]['uid'], matched)
                    else:
                        lock_data['path'] = matched
                    details = cls._get_details_from_lock_data(lock_data)
                if matched in shared_locks:
                    lock_msg = '%s (shared %s): locked %s' % (resource.visible_address.id, resource.storage_address.id, matched)
                    log.info("ResourceLocked: %s %s %s" % (lock_msg, " ".join(traceback.format_exc().splitlines()[-3:]), shared_locks[matched]))
                    raise ResourceLocked(lock_msg, data=details)
                lock_msg = '%s: locked %s' % (resource.storage_address.id, matched)
                log.info("ResourceLocked: %s %s %s" % (lock_msg, " ".join(traceback.format_exc().splitlines()[-3:]), self_locks[matched]))
                raise ResourceLocked(data=details)

        else:
            # получаем локи своего Диска
            self_locks = fs_locks.find_all(address.uid)

            # пытаемся понять, не из ОП ли адрес
            # и если он оттуда - то ищем все залоченные в ОП
            real_address = None
            shared_locks = []
            owner_is_writable = True
            group_owner = None

            from mpfs.core.social.share.processor import ShareProcessor
            for link in ShareProcessor().list_owned_links(address.uid):
                if Address.path_contains_path(link.path, address.path):
                    group = link.group
                    real_address = group.get_group_path(address.path, address.uid)
                    shared_locks = fs_locks.find_all(group.owner)
                    owner_is_writable = mpfs.engine.process.usrctl().is_writable(group.owner)
                    group_owner = group.owner
                    break

            if not owner_is_writable:
                lock_msg = '%s: locked owner %s' % (address.id, group_owner)
                log.info("ResourceLocked: %s %s" % (lock_msg, " ".join(traceback.format_exc().splitlines()[-3:])))
                raise ResourceLocked(lock_msg)

            for locked_path in self_locks:
                if contains(locked_path, address.path):
                    if skip_self and equals(address.path, locked_path):
                        continue

                    lock_msg = '%s: locked %s' % (address.id, locked_path)
                    log.info("ResourceLocked: %s %s %s" % (lock_msg, " ".join(traceback.format_exc().splitlines()[-3:]), self_locks[locked_path]))
                    details = None
                    lock_data = self_locks[locked_path].get('data')
                    if lock_data:
                        lock_data['uid'] = self_locks[locked_path]['uid']
                        lock_data['path'] = locked_path
                        details = cls._get_details_from_lock_data(lock_data)
                    raise ResourceLocked(lock_msg, data=details)

            for locked_path in shared_locks:
                if contains(locked_path, real_address):
                    if skip_self and equals(real_address, locked_path):
                        continue
                    lock_msg = '%s (shared %s): locked %s' % (address.id, real_address, locked_path)
                    log.info("ResourceLocked: %s %s %s" % (lock_msg, " ".join(traceback.format_exc().splitlines()[-3:]), shared_locks[locked_path]))
                    details = None
                    lock_data = shared_locks[locked_path].get('data')
                    if lock_data:
                        lock_data['uid'] = shared_locks[locked_path]['uid']
                        cls._set_path_with_invited_path(lock_data,
                                                        address.uid,
                                                        lock_data['uid'], locked_path)
                        details = cls._get_details_from_lock_data(lock_data)
                    raise ResourceLocked(lock_msg, data=details)

    @classmethod
    def unlock(cls, something):
        """
        Разлочить что-то

        :param something: что-то (ресурс или адрес)
        """
        address, resource = cls.resolve(something, scope='unlock')

        if address:
            log.info('unlocking %s' % address.id)
            fs_locks.release(address.uid, address.path)
        else:
            log.info('nothing to unlock, because %s %s' % (address, resource))

    @classmethod
    def update(cls, something, data=None, time_offset=0):
        """Обновить поле `data` и изменить время удаления записи

        :param something: что-то (ресурс или адрес)
        """
        address, resource = cls.resolve(something, scope='update')

        if address:
            log.info('updating lock for %s' % address.id)
            fs_locks.update(address.uid, address.path, data, time_offset)
        else:
            log.info('nothing to update, because %s %s' % (address, resource))

    @staticmethod
    def get_updated_lock_data_with_operation_type(lock_data, op_type):
        if op_type is None:
            return lock_data
        result = {}

        try:
            if lock_data is not None:
                result = deepcopy(lock_data)
            result['op_type'] = op_type
        except:
            # Игнорируем ошибки, если не получилось проставить тип в детали лока
            pass

        return result

    @classmethod
    def do_trash_append_locks_exist(cls, uid):
        return any([x.get('data', {}).get('op_type') == 'trash_append' for x in fs_locks.find_all(uid).itervalues()])

    @classmethod
    def is_lock_with_data_exist(cls, uid, data_field_name, value):
        return any(lock_data.get('data', {}).get(data_field_name) == value
                   for lock_data in fs_locks.find_all(uid).itervalues())


class LockUpdateTimer(object):
    """
    Таймер, который апдейтит локи в нужное время и перезапускает себя
    """
    def __init__(self, resources=None, addresses=None):
        """
        :param resources: массив ресурсов
        :param addresses: массив адресов
        :return: threading.Timer
        """
        self.resources = resources if resources is not None else ()
        self.addresses = addresses if addresses is not None else ()
        self.lock_helper = LockHelper()
        self.timer = None
        self.stopped = False

    def start(self):
        """
        Создать и запустить таймер
        """
        lock_update_interval = settings.system['system']['filesystem_lock_autoupdate_period']
        self.timer = threading.Timer(lock_update_interval, self.bulk_update, args=(self.resources, self.addresses))
        self.timer.start()

    def stop(self):
        """
        Остановить таймер, если он был создан
        """
        self.stopped = True
        if self.timer:
            self.timer.cancel()

    def bulk_update(self, resources, addresses):
        """
        Основная функция балк-апдейта ресурсов/адресов

        :param resources: массив ресурсов
        :param addresses: массив адресов
        :return:
        """
        if not self.stopped:
            try:
                for resource in resources:
                    self.lock_helper.update(resource)
                for address in addresses:
                    self.lock_helper.update(address)
                self.start()
            except:
                self.stop()
                error_log.error(traceback.format_exc())


class FakeLockUpdateTimer(LockUpdateTimer):
    """
    Исключительно для разработки и автотестов
    Содержит счетчик запусков в классе
    """

    def __init__(self, *args, **kwargs):
        super(FakeLockUpdateTimer, self).__init__(*args, **kwargs)
        self.__class__.count = 0

    def start(self):
        super(FakeLockUpdateTimer, self).start()
        self.__class__.count += 1


if settings.feature_toggles['use_filesystem_lock_autoupdate_period']:
    LockUpdateTimer = FakeLockUpdateTimer
