# -*- coding: utf-8 -*-

from mpfs.core import factory
from mpfs.config import settings
from mpfs.core.address import Address
from mpfs.core.filesystem.iid.models import IIDDocument
from mpfs.core.filesystem.iid.errors import IIDNotFound, IIDDisabled


FEATURE_TOGGLES_IID_ENABLED = settings.feature_toggles['iid_enabled']


def _raise_if_disabled():
    if not FEATURE_TOGGLES_IID_ENABLED:
        raise IIDDisabled()


def _address_by_iid(iid, incoming_raw_address):
    """
    Получить Address по iid-у + raw_address
    """
    _raise_if_disabled()
    iid_doc_id = IIDDocument.generate_id(iid, incoming_raw_address)
    iid_doc = IIDDocument.controller.get(id=iid_doc_id)
    if iid_doc is None:
        raise IIDNotFound()
    return Address(iid_doc.outgoing_raw_address)


def _resource_by_iid(iid, incoming_raw_address):
    """
    Получить ресурс по iid-у + raw_address
    """
    _raise_if_disabled()
    address = _address_by_iid(iid, incoming_raw_address)
    return factory.get_resource(address.uid, address)


def _put_iid(iid, incoming_raw_address, outgoing_raw_address):
    """
    Положить в БД пару iid + raw_address -> address
    """
    _raise_if_disabled()
    # проверям, что ресурсы существуют
    for raw_address in (incoming_raw_address, outgoing_raw_address):
        address = Address(raw_address)
        factory.get_resource(address.uid, address)

    IIDDocument(iid=iid, incoming_raw_address=incoming_raw_address, outgoing_raw_address=outgoing_raw_address).save(force_insert=True)
