# -*- coding: utf-8 -*-
"""

MPFS
CORE

Индексатор ресурсов

"""

import traceback

from collections import defaultdict

import mpfs.engine.process
from mpfs.common.errors import StorageKeyNotFound

from mpfs.core import factory
from mpfs.core.address import Address, ResourceId
from mpfs.core.albums.static import FacesIndexingState
from mpfs.core.filesystem.constants import NOTES_STORAGE_AREA
from mpfs.core.filesystem.photoslice_filter import get_photoslice_time
from mpfs.core.services.index_service import SearchIndexer
from mpfs.core.queue import mpfs_queue
from mpfs.common.util import chunks
from mpfs.config import settings
from mpfs.core.user.constants import PHOTOUNLIM_AREA_PATH
from mpfs.metastorage.mongo.collections.filesystem import is_reindexed_for_quick_move
from mpfs.metastorage.mongo.util import decompress_data

log = mpfs.engine.process.get_default_log()
error_log = mpfs.engine.process.get_error_log()


SERVICES_SEARCH_INDEXER_ENABLE_LIMIT_FOR_DEEP_FOLDERS = settings.services['search_indexer']['enable_limit_for_deep_folders']
SERVICES_SEARCH_INDEXER_MAX_RESOURCE_DEPTH_FOR_INDEXER = settings.services['search_indexer']['max_resource_depth_for_indexer']
SERVICES_SEARCH_INDEXER_SEND_DJFS_CALLBACKS_ON_STORE = settings.services['search_indexer']['send_djfs_callbacks_on_store']


class DiskDataIndexer(object):
    search_on = settings.indexer['search_index']
    cumulative_limit = settings.indexer['cumulative_operations_items_limit']
    """
    Лимит на отправку данных в поиск для операций поддерживающих накопление

    Необходимо для своевременного поступления данных в поиск
    Пример: копирование
    """
    overall_limit = settings.indexer['overall_operations_items_limit']
    """
    Предельное число данных для одной операции

    Актуально для операций не поддерживающих накопление
    Обусловлено предельным размером бинарных данных в монге(16MB)
    Пример: очистка корзины
    """

    shared_index_on = search_on

    def __init__(self):
        self.data = defaultdict(list)
        self.group_data = defaultdict(lambda: defaultdict(list))

    def indexation_allowed(self, uid):
        """
        Проверка: разрешена ли индексация
        """
        return uid.isdigit() and self.search_on

    def data2chunks(self, data):
        cumulative_limit = self.cumulative_limit
        overall_limit = self.overall_limit

        data_len = len(data)
        if data_len > overall_limit:
            for chunk in chunks(data, chunk_size=overall_limit):
                yield chunk
        elif data_len > cumulative_limit:
            yield data

    def indexer_data_mill(self):
        """
        Дробит данные, отправляемые в индексатор.

        Для больших задач не копим все изменения, а посылаем порциями
        Если операция не поддерживает накопление и объем данных очень большой,
        то разбиваем данные на куски.
        """
        for _indexer, _indexer_data in self.data.iteritems():
            clear = False
            for chunk in self.data2chunks(_indexer_data):
                mpfs_queue.put({'data': chunk}, _indexer)
                clear = True
            if clear:
                self.data[_indexer] = list()

        for gid, data in self.group_data.iteritems():
            for _indexer, _indexer_data in data.iteritems():
                clear = False
                for chunk in self.data2chunks(_indexer_data):
                    mpfs_queue.put({'data': chunk, 'gid': gid}, _indexer + '_group')
                    clear = True
                if clear:
                    data[_indexer] = list()

    def push(self, resource, action, *args, **kwargs):
        """
        Пуш информации по одному ресурсу
        """
        from mpfs.core.user.base import User

        user = User(resource.uid)

        if not user.is_standart:
            return

        search_reindex = bool(int(user.states.list('global').get('search_reindex', 0)))

        # if search_reindex already in progress, we should interpret new requests as modify
        # so they will be correctly indexed after search_reindex
        if search_reindex and action == 'update':
            action = 'modify'

        if 'version' not in kwargs:
            kwargs['version'] = resource.version
        kwargs['real_resource_version'] = resource.version

        if resource.is_shared or resource.is_group:
            kwargs['group'] = resource.group
            if resource.is_group_root:
                kwargs['uids'] = [resource.uid, ]
        kwargs['storage_name'] = resource.address.storage_name

        resource_dict = resource.dict()
        try:
            if resource.resource_id:
                resource_dict['resource_id'] = resource.resource_id.serialize()
        except Exception:
            # Были ошибки в ргерессе из-за file_id=0. На всякий случай ловим ошибку и логируем, чтобы не падать.
            error_log.error('Error during building resource_id for Indexer.', exc_info=True)
        self.push_tree(resource.uid, [resource_dict], action, *args, **kwargs)

    def push_tree(self, uid, index, action, *args, **kwargs):
        """
        Пуш индекса всего дерева
        """

        # фильтруем информацию, которая не должна попасть в индекс
        # в настоящий момент это все файлы и папки из раздела /notes
        # плюс ресурсы с большой глубиной вложенности - из-за них ломается DJFS
        filtered_index = []
        for data in index:
            if data.get('original_id'):
                original_resource_address = Address(data['original_id'], uid=uid)
                if original_resource_address.storage_name == NOTES_STORAGE_AREA:
                    continue
                if (SERVICES_SEARCH_INDEXER_ENABLE_LIMIT_FOR_DEEP_FOLDERS and
                        original_resource_address.get_path_depth_lvl() >= SERVICES_SEARCH_INDEXER_MAX_RESOURCE_DEPTH_FOR_INDEXER):
                    continue

            resource_address = Address(data['id'], uid=uid)
            if resource_address.storage_name == NOTES_STORAGE_AREA:
                continue

            if (SERVICES_SEARCH_INDEXER_ENABLE_LIMIT_FOR_DEEP_FOLDERS and
                    resource_address.get_path_depth_lvl() >= SERVICES_SEARCH_INDEXER_MAX_RESOURCE_DEPTH_FOR_INDEXER):
                continue

            filtered_index.append(data)

        if not filtered_index:
            return

        index = filtered_index

        group = kwargs.pop('group', None)
        operation = kwargs.pop('operation', 'other')
        uids = kwargs.pop('uids', [])

        def collect_item_data(item, action):
            item['resource_id'] = self._build_resource_id_by_dict(item, **kwargs)
            data = self.indexing_data(item, action, *args, **kwargs)
            if group:
                if action != 'delete' and group.owner != uid:
                    data['id'] = group.get_group_path(data['id'], data['uid'])
                elif operation == 'move' and action == 'delete':
                    data['actor'] = uid
                if uids:
                    data['uids'] = uids
                real_resource_version = item.get('meta', {}).get('revision', 0)
                if not real_resource_version and 'real_resource_version' in kwargs:
                    real_resource_version = kwargs['real_resource_version']  # костыль для случая, когда эту
                    # функцию вызывают через push, тогда в данных поля revision нет, оно есть в kwargs
                data['real_resource_version'] = int(real_resource_version)  # иногда она еще строкой приезжает

            data['operation'] = operation

            if SERVICES_SEARCH_INDEXER_SEND_DJFS_CALLBACKS_ON_STORE and kwargs.get('append_djfs_callbacks'):
                data['append_djfs_callbacks'] = True

            if is_reindexed_for_quick_move(uid):
                data['is_reindexed_for_quick_move'] = True

            try:
                from mpfs.core.user.base import User
                user = User(uid)
                if user.faces_indexing_state in (FacesIndexingState.RUNNING, FacesIndexingState.REINDEXED):
                    data['clusterize_face_enabled'] = True
            except Exception:
                error_log.exception("Error due to clusterize_face property extraction")

            metric = kwargs.get('metric')
            if metric:
                data['metric'] = metric
            return data

        if group or self.indexation_allowed(uid):
            if group:
                container = self.group_data[group.gid]
            else:
                container = self.data

            for item in index:
                data = collect_item_data(item, action)
                if not data:
                    continue

                photoslice = data.pop('photoslice', None)
                if self.search_on:
                    source_item = kwargs.get('source_item')
                    if photoslice or self._should_notify_smartcache(item, source_item):
                        container['search_photoslice'].append(data)
                    else:
                        container['search'].append(data)

        self.indexer_data_mill()

    @classmethod
    def queue_check_missing_files_and_update_photoslice(cls, uid, paths):
        """
           Для файлов из paths запросить в поиске file_id и попытaться найти в базе,
           по результатам обновить индекс фотосреза. Предполагается, что все paths отсутствуют в базе.
           Индекс обновляем только для фотосррезных файлов.
        """

        for chunk in chunks(paths, cls.overall_limit):
            task_data = {'uid': uid, 'paths': chunk}
            mpfs_queue.put({'data': task_data}, 'search_missing_photoslice')

    @staticmethod
    def _is_photosclice_item(file_data):
        """ Проверить файл на фотосрезность по его данным

        :param dict file_data: данные файла
        :return: True/False
        """
        uid = file_data['uid']
        path = file_data['id']
        address = Address.Make(uid, path)

        mimetype = file_data.get('mimetype')
        etime = file_data.get('meta', {}).get('etime')
        mtime = file_data.get('mtime')
        ctime = file_data.get('ctime')

        photoslice_time = get_photoslice_time(address.path, file_data.get('type'), mimetype, None, etime, ctime, mtime)
        return photoslice_time is not None

    @staticmethod
    def _get_photosclice_time_from_raw_db_item(item):
        """ Функция возвращает время для фотосреза по документу из монги или None, если файл не похож на фотографию

        :param dict item: данные файла в сыром виде из монги
        :return: int/None
        """
        uid = item['uid']
        path = item['key']
        address = Address.Make(uid, path)

        item_data = item.get('data', {})
        mimetype = item_data.get('mimetype')
        etime = item_data.get('etime')
        mtime = item_data.get('mtime')
        ctime = None
        if 'zdata' in item:
            ctime = decompress_data(item['zdata']).get('meta', {}).get('ctime')

        photoslice_time = get_photoslice_time(address.path, item.get('type'), mimetype, None, etime, ctime, mtime)
        return photoslice_time

    def _should_notify_smartcache(self, file_data, source_file_data=None):
        """ Проверить нужно ли уведомлять SmartCache

        Актуально только для файлов фотосреза.

        :param dict source_file_data: данные файла, из которого был получен текущий путем модификаций
        :param dict file_data: элемент индекса текущего файла
        :return: True/False
        """

        file_type = file_data.get('type')
        if file_type != 'file':
            return False

        if self._is_photosclice_item(file_data):
            return True

        # Файл перестал быть фотосрезным в результате, например, копии
        # Исследуем оригинальный файл.
        if source_file_data:
            if self._is_photosclice_item(source_file_data):
                return True

        return False

    def indexing_data(self, item, action, *args, **kwargs):
        """
        Единый метод сбора данных
        """
        action_method = getattr(self, 'indexing_data_%s' % action)
        result = action_method(item, *args, **kwargs)
        result['action'] = action
        return result

    def indexing_data_delete(self, item, *args, **kwargs):
        meta = item.get('meta', {})
        result = {
            'file_id': meta.get('file_id'),
            'uid': int(item.get('uid')),
            # Здесь изменен порядок выбора version,
            # В начале пытаемся брать вермию из kwargs['version'], т.к. там
            # передается новая версия, тогда как в meta['wh_version'] прилетает
            # предыдущая версия. item['version'] всегда(?) пуста
            'version': int(kwargs.get('version') or item.get('version') or meta.get('wh_version') or 0),
        }
        if 'photoslice_time' in meta:
            result['photoslice'] = meta['photoslice_time']
        if item.get('resource_id') is not None:
            result['resource_id'] = item['resource_id']
        return result

    def indexing_data_add(self, item, *args, **kwargs):
        meta = item.get('meta', {})
        type = item['type']
        version = str(item.get('version') or meta.get('wh_version') or kwargs.get('version') or 0)
        if version == 'None':
            version = 0

        if type == 'file':
            result = {
                'file_id': meta['file_id'],
                'mimetype': item['mimetype'],
                'mediatype': item['media_type'],
                'uid': int(item['uid']),
                'name': item['name'],
                'mtime': item['mtime'],
                'ctime': item['ctime'],
                'etime': meta.get('etime'),
                'size': item['size'],
                'id': item['id'],
                'version': int(version),
                'type': item['type'],
                'stid': meta['file_mid'],
                'md5': meta.get('md5'),
                'visible': item['visible'],
                'external_url': meta.get('external_url'),
            }
            if 'photoslice_time' in meta:
                result['photoslice'] = meta['photoslice_time']

            if 'fotki_tags' in meta:
                # https://st.yandex-team.ru/CHEMODAN-38939
                del result['etime']
                if meta['fotki_tags']:
                    result['fotki_tags'] = '\n'.join(meta['fotki_tags'].split(','))

        elif type == 'dir':
            result = {
                "id": item['id'],
                "uid": int(item['uid']),
                'name': item['name'],
                "file_id": meta['file_id'],
                'ctime': item['ctime'],
                "mtime": item['mtime'],
                "type": item['type'],
                'visible': item['visible'],
                'folder_type': meta.get('folder_type'),
                'version': int(version),
                'shared_folder_owner': meta.get('group', {}).get('owner', {}).get('uid', None)
            }
        else:
            raise Exception(item)

        if item.get('resource_id') is not None:
            result['resource_id'] = item['resource_id']

        if kwargs.get('update_photoslice') is True:
            result['photoslice'] = True

        return result

    def indexing_data_modify(self, item, *args, **kwargs):
        return self.indexing_data_add(item, *args, **kwargs)

    def indexing_data_update(self, item, *args, **kwargs):
        meta = item['meta']
        type = item['type']
        version = str(item.get('version') or meta.get('wh_version') or kwargs.get('version') or 0)
        if version == 'None':
            version = 0

        if type == 'file':
            result = {
                'file_id': meta['file_id'],
                'uid': int(item['uid']),
                'name': item['name'],
                'mtime': item['mtime'],
                'ctime': item['ctime'],
                'etime': meta.get('etime'),
                'mediatype': item['media_type'],
                'id': item['id'],
                'visible': item['visible'],
                'version': int(version),
                'size': item['size'],
                'md5': meta.get('md5'),
                'stid': meta['file_mid'],
                'type': item['type'],
                'mimetype': item['mimetype'],
                'external_url': meta.get('external_url'),
            }
            if 'photoslice_time' in meta:
                result['photoslice'] = meta['photoslice_time']

        elif type == 'dir':
            result = {
                "id": item['id'],
                "uid": int(item['uid']),
                'name': item['name'],
                "file_id": meta['file_id'],
                "mtime": item['mtime'],
                'ctime': item['ctime'],
                'visible': item['visible'],
                'folder_type': meta.get('folder_type'),
                'version': int(version),
                'type': item['type'],
                'shared_folder_owner': meta.get('group', {}).get('owner', {}).get('uid', None)
            }
        else:
            raise Exception(item)

        if item.get('resource_id') is not None:
            result['resource_id'] = item['resource_id']
        return result

    def indexing_data_reindex(self, item, *args, **kwargs):
        return self.indexing_data_add(item, *args, **kwargs)

    def push_reindex(self, uid, index_type=None, index_body=1, mediatype=None, area=None, force=0):
        """
        Постановка задачи на переиндексацию
        """
        from mpfs.core.user.base import User

        if area and User(uid).is_standart:
            task_name = '%s_%s' % (area, 'reindex')
            mpfs_queue.put(
                {
                    'uid': uid,
                    'index_body': index_body,
                    'index_type': index_type,
                    'mediatype': mediatype,
                    'force': force,
                },
                task_name
            )

    @staticmethod
    def get_search_reindex_supported_areas():
        return '/disk', '/trash', PHOTOUNLIM_AREA_PATH

    def search_reindex(self, uid, index_body=1, index_type=None, mediatype=None, force=0):
        """
        Переиндексация всего

        index_body - 1/0 - индексировать ли содержимое (поход в тикайту)
        index_type - dir/file - индексировать определенный вид ресурсов (файл/каталог)
        mediatype - [] - массив медиатипов, подлежащих индексации (['video','audio'])
        force - 1/0 - форсировать ли изменения; при форсировании не будет отправлено поле version

        При неотправке version индексатор не проверяет версии и принудительно перетрет документ

        При переиндексации без содержимого (index_body==0) параметр action передается как update
        Это необходимо для того, чтобы переиндексировать метаполя, не трогая уже имещющегося индекса с телами
        """
        from mpfs.core.user.base import User

        search_indexer = SearchIndexer()
        User(uid).set_state(key='search_reindex', value=1)

        from mpfs.core.social.share.link import LinkToGroup
        has_links = len(list(LinkToGroup.iter_all(uid))) > 0

        mediatype = mediatype if mediatype else []

        for path in self.get_search_reindex_supported_areas():
            address = Address.Make(uid, path)
            resource = factory.get_resource(uid, address)

            flat_index_mediatype = None
            # если у юзера нет подключенных ОП, то используем быстрый поиск по mediatype
            # иначе фоллбек на старую схему и далее дофильтрация
            if mediatype and not has_links:
                flat_index_mediatype = mediatype
            try:
                full_index = resource.flat_index(mediatype=flat_index_mediatype)
            except StorageKeyNotFound:
                continue

            for item in full_index.itervalues():
                if index_type and item['type'] != index_type:
                    continue

                if mediatype and item.get('media_type') not in mediatype:
                    continue

                try:
                    file_id = item['meta']['file_id']
                except KeyError:
                    error_log.warn('no file_id for %s:%s' % (uid, item))
                else:
                    if file_id:
                        grab_method = 'modify' if index_body else 'update'
                        item['resource_id'] = self._build_resource_id_by_dict(item)
                        data = self.indexing_data(item, grab_method, search=True)
                        data['operation'] = 'reindex'

                        if force:
                            del data['version']

                        if (settings.indexer['search_index_body'] and
                                index_body and
                                data['type'] == 'file' and
                                'body_text' not in data):
                            body_info = search_indexer.get_file_body(data)
                            data.update(body_info)

                        try:
                            search_indexer.push_change(data)
                        except Exception:
                            error_log.error(traceback.format_exc())

        User(uid).remove_state(key='search_reindex')

    def flush_index_data(self, group_data_first=False):
        def flush_personal():
            """process search for personal data"""
            for indexer, indexer_data in self.data.iteritems():
                if indexer_data:
                    mpfs_queue.put({'data': indexer_data}, indexer)

        def flush_group():
            """process search for group data"""
            for gid, indexer_data in self.group_data.iteritems():
                for indexer, data in indexer_data.iteritems():
                    if data:
                        mpfs_queue.put({'data': data, 'gid': gid}, indexer + '_group')

        if group_data_first:
            flush_group()
            flush_personal()
        else:
            flush_personal()
            flush_group()

    @staticmethod
    def _build_resource_id_by_dict(resource_dict, **kwargs):
        """
        Вернуть resource_id по словарю ресурса и доп. параметрам

        Метод предназначен для получения resource_id для отправки в индексатор. Финальным методом, который кладет
        параметры для индексатора в таск, является push_tree. Существует 3 сценария, в каждом из которых resource_id
        достается по-разному:
        * push_tree вызывается через push. Этот случай наиболее частый и выполнется во всех случаях, кроме описанных
            ниже. Ресурсы через push->push_tree по одиночке добавляются в таски. В push берем resource_id из
            десериализованного ресурса и прокидываем его явно в push_tree.
        * push_tree вызывается со списком полных словарей ресурсов. Выполняется при rm, удалении гостем своего корня
            ОП, выход из группы, принятие приглашения в ОП, trash_drop_all. Для построения правильного uid достаточно
            информации из ресурса и содержащегося в нем поля group.
        * push_tree вызывается со списком урезанных словарей. Выполняется при отключении расшаренности ресурса
            владельцем. Для этого в push_tree пердаются доп. параметры owner_uid и shared_guest_root_folder, с помощью
            которых можно построить правильный resource_id.
        :param resource_dict:
        :param kwargs:
        :rtype str
        """
        if resource_dict.get('resource_id'):
            return resource_dict.get('resource_id')
        group = resource_dict.get('meta', {}).get('group')
        if group and not group.get('is_root'):
            uid = group.get('owner', {}).get('uid')
            if uid is None:
                # Если из группы не удалось достать владельца, то подставляем уид из ресурса. Есть очень странный и
                # редкий кейс: если владелец ОП удалился из паспорта, то все поля group['owner'] равны None, что не
                # позволяет построить resource_id. Такое встречалось только в тех случаях, когда в подобных ОП гости
                # удаляют ресурсы через rm или перезаписью, и об это отправлялся пуш владельцу. В таком случае uid
                # ресурса - это и есть владелец.
                uid = resource_dict['uid']
        elif kwargs.get('owner_uid') and kwargs.get('shared_guest_root_folder')\
                and kwargs['shared_guest_root_folder'] != resource_dict.get('id'):
            uid = kwargs['owner_uid']
        else:
            uid = resource_dict['uid']
        try:
            return ResourceId(uid, resource_dict.get('meta', {}).get('file_id')).serialize()
        except Exception:
            # Были ошибки в ргерессе из-за file_id=0. На всякий случай ловим ошибку и логируем, чтобы не падать.
            error_log.error('Error during building resource_id for Indexer.', exc_info=True)
