# -*- coding: utf-8 -*-
"""

MPFS
CORE

Менеджер занимаемого места 

"""
import time
import os
import traceback
import datetime
from collections import defaultdict

import mpfs.engine.process
from mpfs.common.static.tags.experiment_names import UPLOAD_TRAFFIC_COUNTERS, UPLOAD_MAX_FILE_SIZE_LIMIT
from mpfs.common.util.experiments.logic import experiment_manager
from mpfs.common.util.limits.utils import UploadLimitPeriods

from mpfs.config import settings
from mpfs.core import factory
from mpfs.core.address import Address
from mpfs.core.metastorage.control import disk, trash, disk_info
from mpfs.common.static import SPACE_1GB, SPACE_1MB
from mpfs.common.static.tags.push import Low, Full
from mpfs.core.services.passport_service import Passport
from mpfs.common.util import mailer, safe_convert
from mpfs.common import errors
from mpfs.core.user.constants import DISK_AREA

FEATURE_TOGGLES_SPEED_LIMIT = settings.feature_toggles['speed_limit']

log = mpfs.engine.process.get_default_log()
error_log = mpfs.engine.process.get_error_log()
passport = Passport()


class Quota(object):
    '''
    Класс менеджера квот
    '''

    def _obtain_address(self, **args):
        if 'address' in args:
            return args['address']
        elif 'uid' in args:
            return Address.Make(args['uid'], '/disk')
        else:
            raise errors.QuotaPathError()

    def free(self, **args):
        '''
        Сколько свободного места по указанному адресу
        '''
        address = self._obtain_address(**args)
        return factory.get_service(address).free(address.uid)

    def free_with_shared_support(self, **args):
        """Сколько свободного места по указанному адресу.

        Поддерживает шаренные папки.
        """
        #TODO(kis8ya): Объединить с `free`
        address = self._obtain_address(**args)
        uid = address.uid
        if address.storage_name == DISK_AREA:
            # Для ресурсов в /disk смотрим не ОП ли это
            resource = factory.get_resource(address.uid, address)
            uid = address.uid
            if resource.is_shared:
                uid = resource.owner_uid
        return factory.get_service(address).free(uid)

    def limit(self, **args):
        '''
        Каков лимит по указанному адресу
        '''
        address = self._obtain_address(**args)
        return factory.get_service(address).limit(address.uid)


    def used(self, **args):
        '''
        Сколько использовано
        '''
        address = self._obtain_address(**args)
        return factory.get_service(address).used(address.uid)


    def set_limit(self, limit, **args):
        '''
        Установить лимит
        '''
        address = self._obtain_address(**args)
        return factory.get_service(address).set_limit(address.uid, limit)

    def change_limit_on_delta(self, delta_limit, **args):
        """
        Изменить лимит на дельту
        """
        address = self._obtain_address(**args)
        return factory.get_service(address).change_limit_on_delta(address.uid, delta_limit)

    def report(self, uid):
        '''
        Сводный обзор
        '''
        disk_service = factory.get_service(Address.Make(uid, '/disk'))
        trash_service = factory.get_service(Address.Make(uid, '/trash'))

        limit = disk_service.limit(uid) or 0
        used = disk_service.used(uid) or 0
        free = limit - used if used <= limit else 0
        count = disk_service.files_count(uid)
        trash_used = trash_service.used(uid)
        fsize = disk_service.paid_filesize_limit

        result = {
            'uid': uid,
            'limit': limit,
            'used': used,
            'free': free,
            'trash': trash_used,
            'files_count': count,
            'filesize_limit': fsize,
        }

        if experiment_manager.is_feature_active(UPLOAD_MAX_FILE_SIZE_LIMIT):
            result['paid_filesize_limit'] = disk_service.paid_filesize_limit
            from mpfs.core.user.base import User
            if not User(uid).is_paid():
                result['filesize_limit'] = disk_service.filesize_limit

        return result

    def get_size_for_tree(self, uid, path):
        """
        Получить суммарный размер всех элементов поддерева
        """
        controllers = {
            'disk': disk,
            'trash': trash,
        }
        controller = controllers.get(path.split('/')[1])

        size = 0
        content = controller.folder_content(uid, path).value
        for key, value in content.iteritems():
            child_path = path + os.path.sep + key
            if value.data['type'] == 'dir':
                _size = self.get_size_for_tree(uid, child_path)
                size += _size
            elif value.data['type'] == 'file':
                size += value.data['size']
        return size

    def inspect(self, uid):
        """
        Глобальный пересчет счетчиков
        """
        regular_uid = uid != mpfs.engine.process.share_user()
        group_sizes = defaultdict(int)

        def tree(uid, node, controller=None):
            used, count = 0, 0
            
            node_content = controller.folder_content(uid, node).value
            for key, value in node_content.iteritems():
                child_path = node + os.path.sep + key
                if value.data['type'] == 'dir':
                    __c, __u = tree(uid, child_path, controller=controller)
                    count += __c
                    used += __u
                elif value.data['type'] == 'file':
                    count += 1
                    used += value.data['size']
                    if regular_uid and child_path.startswith('/disk/'):
                        try:
                            group = Group.find(uid=uid, path=child_path, link=False, group=True)
                        except Exception:
                            pass
                        else:
                            group_sizes[group.gid] += value.data['size']
            return count, used
        
        # https://jira.yandex-team.ru/browse/CHEMODAN-5323
        if regular_uid:
            user_path = '/disk'
            from mpfs.core.social.share import Group, LinkToGroup
            Group.load_all(uid)
        else:
            user_path = '/share'

        disk_count, disk_used = tree(uid, user_path, controller=disk)
        trash_count, trash_used = tree(uid, '/trash', controller=trash)

        total_count = disk_count + trash_count
        total_used = disk_used + trash_used
        
        disk_info.put(uid, 'total_size', total_used, None) 
        disk_info.put(uid, 'files_count', total_count, None)
        disk_info.put(uid, 'check_time', int(time.time()), None)
        disk_info.put(uid, 'trash_size', trash_used, None)

        if regular_uid:

            for link in LinkToGroup.iter_all(uid):
                group_sizes[link.group.gid] = 0
                tree(link.group.owner, link.group.path, controller=disk)

            for group_id, group_size in group_sizes.iteritems():
                group = Group.find(gid=group_id, link=False, group=True)
                group.size = group_size
                group.save()

        return self.report(uid)

    def download_traffic(self, uid):
        resp = disk_info.value(uid, 'download_traffic')
        if not resp.value:
            size = 0
        else:
            size = int(resp.value.data.get('bytes'))
        return int(size or 0)

    def download_speed_limited(self, uid):
        if not FEATURE_TOGGLES_SPEED_LIMIT:
            return False

        result = False
        resp = disk_info.value(uid, 'download_traffic')
        if resp.value:
            size = int(resp.value.data.get('bytes'))
            ctime = int(resp.value.data.get('ctime'))
            limit = self.limit(uid=uid)
            daybefore = int(time.mktime((datetime.datetime.now() - datetime.timedelta(days=1)).timetuple()))
            if ctime > daybefore and size > 2 * limit:
                result = True

        return result

    def get_download_speed_limit_end_time(self, uid):
        """Метод зовётся, если вместо ограничения скорости файл полностью блокируется для скачивания, поэтому
        здесь нет проверки FEATURE_TOGGLES_SPEED_LIMIT.

        https://st.yandex-team.ru/CHEMODAN-35609
        """
        resp = disk_info.value(uid, 'download_traffic')
        if resp.value:
            size = int(resp.value.data.get('bytes'))
            ctime = int(resp.value.data.get('ctime'))
            limit = self.limit(uid=uid)
            daybefore = int(time.mktime((datetime.datetime.now() - datetime.timedelta(days=1)).timetuple()))
            if ctime > daybefore and size > 2 * limit:
                return int(time.mktime((datetime.datetime.fromtimestamp(ctime) + datetime.timedelta(days=1)).timetuple()))
        return None

    def update_traffic(self, uid, bytes_downloaded):
        ctime = None
        now = int(time.time())
        val = {}
        resp = disk_info.value(uid, 'download_traffic')
        if resp.value:
            ctime = resp.value.data.get('ctime')
        daybefore = int(time.mktime((datetime.datetime.now() - datetime.timedelta(days=1)).timetuple()))

        if ctime and ctime > daybefore:
            disk_info.increment(uid, 'download_traffic', bytes_downloaded, 'data.bytes')
        else:
            val = {'ctime': now, 'bytes': bytes_downloaded }
            disk_info.put(uid, 'download_traffic', val, None)

    def upload_traffic(self, uid, period):
        counter = disk_info.value(uid, period.counter_key)
        if not counter.value:
            size = 0
        else:
            ctime = counter.value.data.get('ctime')
            period_max_ctime_start = int(
                time.mktime((datetime.datetime.now() + datetime.timedelta(days=period.days_delta)).timetuple())
            )
            if ctime and ctime > period_max_ctime_start:
                size = long(counter.value.data.get('bytes'))
            else:
                size = 0
        return size

    def upload_traffic_data(self, uid, period):
        counter = disk_info.value(uid, period.counter_key)
        return counter.value

    def update_upload_traffic_for_operation(self, operation, bytes_uploaded):
        if not experiment_manager.is_feature_active(UPLOAD_TRAFFIC_COUNTERS):
            return

        if not hasattr(operation, 'subtype') or operation.subtype == 'only_office':
            return

        if not operation.data.get('upload_traffic_counted'):
            self.update_upload_traffic(operation.uid, bytes_uploaded=bytes_uploaded)
            operation.data['upload_traffic_counted'] = True
            operation.save()

    def update_upload_traffic(self, uid, bytes_uploaded):
        if not experiment_manager.is_feature_active(UPLOAD_TRAFFIC_COUNTERS):
            return

        bytes_uploaded = safe_convert(long, bytes_uploaded, fallback=None)
        if not bytes_uploaded:
            return

        now = int(time.time())
        for period in UploadLimitPeriods:
            if (period.value.name not in settings.limits['upload_traffic'] or
                    not settings.limits['upload_traffic'][period.value.name]['counter_enabled']):
                continue
            ctime = None
            counter = disk_info.value(uid, period.value.counter_key)
            if counter.value:
                ctime = counter.value.data.get('ctime')
            period_max_ctime_start = int(
                time.mktime((datetime.datetime.now() + datetime.timedelta(days=period.value.days_delta)).timetuple())
            )

            if ctime and ctime > period_max_ctime_start:
                # ctime is within range (period_max_ctime_start, now]
                disk_info.increment(uid, period.value.counter_key, bytes_uploaded, 'data.bytes')
            else:
                # ctime it too old -> starting a new interval for counter
                val = {'ctime': now, 'bytes': bytes_uploaded}
                disk_info.put(uid, period.value.counter_key, val, None)

    def get_push_type(self, uid):
        space_info = self.report(uid)
        limit = space_info['limit']
        free = space_info['free']
        if free < SPACE_1GB and free < limit*0.1:
            if free < SPACE_1MB:
                result = Full()
            else:
                result = Low()
        else:
            result = None
        return result

    def send_space_email(self, uid, template, space_info):
        from mpfs.core.user.base import User
        bound_stamp = int(time.time()) - 14*24*3600

        user = User(uid)
        cron_states = user.states.list('cron')

        notified = cron_states.get('diskisfullnotify')
        if not notified or notified < bound_stamp:
            try:
                user_info = passport.userinfo(uid)
                email = user_info['email']
            except Exception:
                raise errors.PassportError

            params = {
                'username': user_info['username'],
                'login': user_info['login'],
                'locale': user_info['language'],
                'space': space_info,
            }

            try:
                mailer.send(email, template, params)
            except Exception:
                error_log.error('sending email failed %s' % uid)
                error_log.error(traceback.format_exc())
            else:
                user.states.set('diskisfullnotify', int(time.time()), 'cron')

