# -*- coding: utf-8 -*-
import traceback
import pprint

import mpfs.engine.process
from mpfs.common.util import filetypes

from mpfs.core.filesystem.dao.legacy import is_new_fs_spec_required
from mpfs.core.filesystem.repair.common import CommonRepairTool
from mpfs.core.filesystem.resources.base import Resource
from mpfs.core.user.base import User
from mpfs.metastorage.mongo.collections.filesystem import HiddenDataCollection, UserCollectionZipped
from mpfs.metastorage.mongo.util import parent_for_key

log = mpfs.engine.process.get_default_log()
error_log = mpfs.engine.process.get_error_log()


class HiddenDataUnzipTool(CommonRepairTool):
    """
    Исправлятор зазипованных данных в hidden_data

    https://st.yandex-team.ru/CHEMODAN-22898
    """
    def report(self, uid, collection=None):
        """
        Исследовать ситуацию с незазипованным hidden_data и выдать список путей/_id

        :param uid: uid юзера
        :return: list путей
        """
        User(uid)
        mpfs.engine.process.reset_cached()
        return self._find(str(uid))

    def repair(self, uid, collection=None):
        """
        Исправить неправильно сохраненные данные в hidden_data

        :param uid: uid юзера
        :return: list восстановленных путей
        """
        User(uid)
        mpfs.engine.process.reset_cached()
        bad_ids = self._find(str(uid))
        return self._repair(str(uid), bad_ids)

    def _find(self, uid):
        """
        собираем все подлежащие обработке ресурсы

        результат выглядит так:
        [{u'dd25ffa184af62c9bb68ec2993bc6324': u'/hidden/top'}]

        :param uid: uid юзера
        :return: list путей/_id
        """
        result = {}
        coll_object = HiddenDataCollection()
        raw_coll = coll_object.collection
        self.say('starting to check %s for %s' % (coll_object.name, uid))

        spec = {'uid': uid, 'data.stids': {'$exists': False}}
        fields = ('_id', 'key', 'type')
        for doc in raw_coll.find(spec, fields=fields):
            _id, key, rtype = doc['_id'], doc['key'], doc['type']
            if rtype == 'file':
                result[_id] = key

        self.say('found %s bad documents for %s:%s' % (len(result), uid, coll_object.name))
        if len(result):
            self.say(pprint.pformat(result))

        return result

    @staticmethod
    def _prepare_document(doc, _id, unique_file_ids, cname, status_log, error_log):
        """
        Готовим данные:
        - расзипуем стиды
        - фиксим file_id
        - магия с mediatype
        """
        if not doc:
            raise MigrationFatalError('no document %s' % _id)

        # костыль для hidden_data
        if cname == 'hidden_data':
            if not ('uid' in doc and 'zdata' in doc and 'type' in doc):
                return doc
            if 'data' not in doc:
                doc['data'] = {}
        else:
            if not ('uid' in doc and 'data' in doc and 'zdata' in doc and 'type' in doc):
                return doc

        zipper = UserCollectionZipped()
        rawdata = zipper._unzip_resource_data(doc['type'], doc['data'], doc['zdata'])
        if 'file_id_zipped' in rawdata.get('meta', {}):
            rawdata['meta'].pop('file_id_zipped')
        if 'file_id_zipped' in doc.get('data', {}):
            doc['data'].pop('file_id_zipped')

        # fix file_ids
        file_id = rawdata.get('meta', {}).get('file_id')
        if not file_id or file_id is None:
            file_id = Resource.generate_file_id(doc['uid'], doc['key'])
            unique_file_ids.append(file_id)
        else:
            if file_id in unique_file_ids:
                file_id = Resource.generate_file_id(doc['uid'], doc['key'])
            unique_file_ids.append(file_id)
        zdata, data = zipper._zip_file_data(rawdata)
        # fix mediatype issues
        if 'data' in doc and 'media_type' in doc['data']:
            doc['data']['mt'] = filetypes.getGroupNumber(doc['data']['media_type'])
            del doc['data']['media_type']

        # result
        if data['stids']:
            doc['data']['stids'] = data['stids']

        doc['zdata'] = zdata
        doc['data']['file_id'] = file_id

        return doc

    def _repair(self, uid, bad_ids):
        """
        Создаем все переданные каталоги

        :param uid: uid юзера
        :param bad_ids: iter путей для обработки
        :return: list обработанных _id
        """
        if not len(bad_ids):
            return []

        coll_object = HiddenDataCollection()
        raw_coll = coll_object.collection
        self.say('starting to repair %s for %s' % (coll_object.name, uid))

        ok, fail = {}, {}
        requested_amount = len(bad_ids)
        unique_file_ids = []
        for _id, key in bad_ids.iteritems():
            try:
                if is_new_fs_spec_required(coll_object.name):
                    spec = {'uid': uid, 'path': key}
                else:
                    spec = {'uid': uid, '_id': _id}

                raw_document = raw_coll.find_one(spec)
                doc = self._prepare_document(
                    raw_document, _id, unique_file_ids, coll_object.name,
                    log, error_log
                )

                if is_new_fs_spec_required(coll_object.name):
                    doc['parent'] = parent_for_key(key)

                raw_coll.update(spec, doc)
                ok[_id] = key
            except Exception as e:
                self.say('exception on %s:%s: %s, more info in error log' % (uid, key, e))
                error_log.error(traceback.format_exc())
                fail[_id] = key

        self.say('requested:%s, repaired:%s, failed:%s' % (requested_amount, len(ok), len(fail)))
        return ok
