# -*- coding: utf-8 -*-
import traceback
import time
import pprint

import mpfs.engine.process

from mpfs.common import errors
from mpfs.core.address import Address
from mpfs.core.filesystem.repair.common import CommonRepairTool
from mpfs.core.user.base import User
from mpfs.metastorage.mongo.collections.filesystem import (
    UserDataCollection,
    TrashCollection,
    MiscDataCollection,
    NarodDataCollection,
    AttachDataCollection
)

log = mpfs.engine.process.get_default_log()
error_log = mpfs.engine.process.get_error_log()


class HierarchyRepairTool(CommonRepairTool):
    """
    Исправлятор иерархической структуры в коллекциях с файлами

    https://st.yandex-team.ru/CHEMODAN-21031
    """

    allowed_to_repair = {
        'user_data': UserDataCollection,
        'trash': TrashCollection,
        'misc_data': MiscDataCollection,
        'attach_data': AttachDataCollection,
        'narod_data': NarodDataCollection
    }

    def report(self, uid, collection=None):
        """
        Исследовать ситуацию с поломанной иерархией и выдать список отсутсвующих путей

        Гаврила вел учет ресурсов
        Дерьмо Гаврила находил

        :param uid: uid юзера
        :param collection: имя коллекции
        :return: list путей
        """
        User(uid)
        mpfs.engine.process.reset_cached()

        collclass = self.allowed_to_repair.get(collection)
        if not collclass:
            raise errors.MPFSNotImplemented(collection)

        return self._find(str(uid), collclass)

    def repair(self, uid, collection=None):
        """
        Починить поломанную иерархию, восстановив все пути

        Гаврила взял совок и ёршик
        Пути Гаврила починял

        :param uid: uid юзера
        :param collection: имя коллекции
        :return: list восстановленных путей
        """
        User(uid)
        mpfs.engine.process.reset_cached()

        collclass = self.allowed_to_repair.get(collection)
        if not collclass:
            raise errors.MPFSNotImplemented(collection)

        missing_paths = self._find(str(uid), collclass)
        return self._repair(str(uid), collclass, missing_paths)

    def _find(self, uid, collclass):
        """
        собираем все ресурсы, их _id, parent и key для последующей обработки
        resources и folders после сбора данных выглядят так:

        {u'93ac18819bdc41e6076ca6c59a679b4f': {u'key': u'/disk/top/middle/bottom',
                                       u'parent': u'9ceda49f84a816e0817f9e60965856fe',
                                       'skip': True,
                                       u'type': u'dir'},

        результат выглядит так:
        {u'dd25ffa184af62c9bb68ec2993bc6324': u'/disk/top'}

        :param uid: uid юзера
        :param collclass: класс коллекции
        :return: list путей
        """
        coll_object = collclass()
        raw_coll = coll_object.collection

        self.say('starting to check hierarchy for %s:%s' % (uid, coll_object.name))

        resources, folders = {}, {}
        spec, fields = {'uid': uid}, ('key', 'type', '_id', 'parent')
        for item in raw_coll.find(spec, fields):
            _id = item.pop('_id')
            resources[_id] = item

            if item['type'] == 'dir':
                item['skip'] = False
                folders[_id] = item

        # ищем недостающих родителей
        missing_parents = {}
        for k, v in resources.iteritems():
            try:
                parent = v['parent']
            except KeyError:
                continue

            # пропускаем тех, кого уже обработали
            try:
                skip = True if parent in missing_parents or folders[parent]['skip'] else False
            except KeyError:
                skip = False

            if skip:
                continue

            if parent not in folders:
                # нашли недостающего, выдираем путь до него и складируем на восстановление
                child_path = v['key']
                child_address = Address.Make(uid, child_path)
                parent_address = child_address.get_parent()
                missing_parents[parent] = parent_address.path
            else:
                # если нормальный - просто помечаем его, чтобы не обрабатывать дальше
                folders[parent]['skip'] = True

        self.say('found %s missing hierarchy parents for %s:%s' % (len(missing_parents), uid, coll_object.name))
        if len(missing_parents):
            self.say(pprint.pformat(missing_parents))

        return missing_parents.values()

    def _repair(self, uid, collclass, missing_paths):
        """
        Создаем все переданные каталоги

        :param uid: uid юзера
        :param collclass: класс коллекции
        :param missing_parents: list путей для создания
        :return: list созданных путей
        """
        if not len(missing_paths):
            return []

        coll_object = collclass()
        self.say('starting to repair hierarchy for %s:%s' % (uid, coll_object.name))

        ok, fail = [], []
        missing_paths.sort()
        requested_amount = len(missing_paths)

        while len(missing_paths):
            path = missing_paths.pop(0)

            try:
                method = '_trash_mkdir' if coll_object.name == 'trash' else '_common_mkdir'
                getattr(self, method)(uid, path)
                ok.append(path)
            except errors.MkdirFolderAlreadyExist:
                pass
            except (errors.FolderNotFound, errors.MkdirNotFound):
                parent_address = Address.Make(uid, path).get_parent()
                missing_paths.append(parent_address.path)
                missing_paths.append(path)
            except Exception as e:
                self.say('exception on %s:%s: %s, more info in error log' % (uid, path, e))
                error_log.error(traceback.format_exc())
                fail.append(path)

        self.say('requested:%s, created:%s, failed:%s' % (requested_amount, len(ok), len(fail)))
        return ok

    def _common_mkdir(self, uid, path):
        """
        Создание каталога везде, кроме корзины

        :param uid: uid
        :param path: путь
        """
        address = Address.Make(uid, path)
        self.fs.mkdir(uid, address.id, notify=False)

    def _trash_mkdir(self, uid, path):
        """
        Создание каталога в корзине

        :param uid: uid
        :param path: путь
        """
        address = Address.Make(uid, path)
        original_disk_address = Address.Make(uid, path)
        original_disk_address.change_storage('disk')

        folderdata = {
            'name': address.name,
            'ctime': int(time.time()),
            'utime': int(time.time()),
            'mtime': int(time.time()),
            'id': original_disk_address.path,
        }

        self.fs.mkdir(uid, address.id, notify=False, **{'data': folderdata})
