# -*- coding: utf-8 -*-
"""

MPFS
CORE

Почтовые аттачи

"""
import urllib

from mpfs.config import settings
from mpfs.common.util.crypt import CryptAgent
from mpfs.common import errors
from mpfs.core.address import Address
from mpfs.core.filesystem.resources.legacy import LegacyFile, LegacyFolder, LegacyResource
from mpfs.core.services.attach_service import Attach


class AttachResource(LegacyResource):
    crypt_agent = CryptAgent()

    def make_public(self, symlink):
        if not settings.feature_toggles['mail_long_url']:
            return super(AttachResource, self).make_public(symlink)
        else:
            self.meta['symlink'] = symlink.address.id
            hsh = self.crypt_agent.encrypt(self.get_symlink())
            self.meta['public'] = 1
            self.meta['public_hash'] = hsh
            quoted_hash = urllib.quote(hsh, safe='')  # https://st.yandex-team.ru/CHEMODAN-29829
            self.meta['short_url'] = "%s/mail/?hash=%s" % (settings.system['public_urls']['resource_short'],
                                                           quoted_hash)
            self.meta['short_url_named'] = "%s/?%s" % (self.meta['short_url'], self.name)
            self.save()


class AttachFile(LegacyFile, AttachResource):
    service_class = Attach

    @classmethod
    def from_dict(cls, data, user_principal=None):
        address = Address.Make(data['uid'], data['key'])
        if address.storage_name != 'attach':
            raise errors.ResourceNotFound()
        return super(AttachFile, cls).from_dict(data, user_principal=user_principal)

    @classmethod
    def Create(cls, uid, *args, **kwargs):
        from mpfs.core.user import base as user  # circular import
        try:
            return super(AttachFile, cls).Create(uid, *args, **kwargs)
        except errors.StorageKeyNotFound as e:
            if isinstance(e.data, dict) and e.data.get('key') == '/attach':
                if user.NeedInit(uid, type='attach'):
                    user.Create(uid, type='attach')

                return super(AttachFile, cls).Create(uid, *args, **kwargs)
            else:
                raise e

    def is_smth_public(self):
        return self.meta.get('symlink') is not None or \
            self.meta.get('public') is not None or \
            self.meta.get('public_hash') is not None or \
            self.meta.get('short_url') is not None or \
            self.meta.get('short_url_named') is not None


class AttachFolder(LegacyFolder, AttachResource):
    file_class = AttachFile
    service_class = Attach

    @classmethod
    def from_dict(cls, data, user_principal=None):
        address = Address.Make(data['uid'], data['key'])
        if address.storage_name != 'attach':
            raise errors.ResourceNotFound()
        return super(AttachFolder, cls).from_dict(data, user_principal=user_principal)

    def folders_allowed(self, method=None):
        if method == 'copy_resource':
            return True

        return super(AttachFolder, self).folders_allowed(method=method)

    def get_url(self, **args):
        inline = args.get('inline')
        if inline:
            self.setup(inline=inline)
        return {
            'folder': self.get_folder_url(**args),
        }

    def is_smth_public(self):
        return self.meta.get('symlink') is not None or \
            self.meta.get('public') is not None or \
            self.meta.get('public_hash') is not None or \
            self.meta.get('short_url') is not None or \
            self.meta.get('short_url_named') is not None
