# -*- coding: utf-8 -*-
"""

MPFS
CORE
FILESYSTEM

Базовое описание ресурсов файловой системы

"""
import time

from copy import deepcopy
from hashlib import sha256
from itertools import chain

import mpfs.engine.process
from mpfs.common.util.filetypes import DEFAULT_MIME_TYPE

from mpfs.config import settings
from mpfs.common import errors
from mpfs.core.address import Address, ResourceId
from mpfs.common.forms.file import FileForm
from mpfs.common.forms.folder import FolderForm
from mpfs.common.forms import ResourceForm
from mpfs.core.office.static import OfficeAccessStateConst

log = mpfs.engine.process.get_default_log()
error_log = mpfs.engine.process.get_error_log()

FEATURE_TOGGLES_SETTING_YAROVAYA_MARK_ENABLED = settings.feature_toggles['setting_yarovaya_mark_enabled']


class Resource(object):
    """
    Базовый класс ресурса
    """
    service_class = object

    time_fields = ('ctime', 'mtime', 'utime', 'etime')
    int_fields = time_fields + ('size', 'visible')
    fields = ['type', 'ctime', 'mtime', 'utime', 'name', 'visible', 'labels', 'meta']
    form_class = ResourceForm
    main_fields = ['visible', 'labels', 'mtime', 'ctime', 'utime']
    static_fields = ['id', 'uid', 'type', 'wh_version', ]
    not_saving_fields = ()
    not_saving_meta_fields = ('file_id_zipped', 'folder_id', 'revision', 'public_time')
    uniqiue_fields = ()
    public_fields = ()

    def __init__(self, uid, address, *args, **kwargs):
        """
            self.address - под которым хранятся данные
            self.visible_address - который виден снаружи
        """
        self.address = self.visible_address = self.storage_address = address
        self.request = None
        self._parent = None

        request = kwargs.get('request')
        if request:
            self.request = request
        data = kwargs.get('data') or {}
        self.prev_version = self.version = kwargs.get('version')

        try:
            dtype = data['type']
        except KeyError:
            pass
        else:
            if dtype and dtype != self.type:
                raise errors.ResourceNotFound()

        if data:
            for attr, value in self.dir().iteritems():
                if attr not in ('id', 'path', 'uid'):
                    value = data.get(attr, value)
                    if attr in self.time_fields and value:
                        value = int(value)
                    elif attr == 'mimetype':
                        value = value or DEFAULT_MIME_TYPE
                    setattr(self, attr, value)
        else:
            for attr, value in self.dir().iteritems():
                setattr(self, attr, value)
            self.fill(address)

        self.id = address.path
        self.path = address.path
        # file owner uid
        self.uid = address.uid
        self.committer = uid

        if self.name is None:
            try:
                self.name = data['name']
            except KeyError:
                self.name = address.name

        # Name of all parents in reverse order
        self.name_tree  = data.get('name_tree') or [self.name, ]
        self.path_named = data.get('path_named') or self.path
        actual_time = self.ctime or self.mtime
        if actual_time:
            for field in self.time_fields:
                try:
                    value = getattr(self, field)
                except AttributeError:
                    continue
                else:
                    if not value:
                        setattr(self, field, actual_time)

    def set_parent(self, parent_resource):
        """Установить родительский каталог"""
        assert isinstance(parent_resource, Resource)
        self._parent = parent_resource

    def get_parent(self):
        """
        Получить редительский каталог

        Получить его можно, если только он был руками установлен через set_parent
        Если не был установлен, то вернется None
        """
        return self._parent

    @property
    def _service(self):
        try:
            return self.__service
        except AttributeError:
            self.__service = self.service_class()
            return self.__service

    @property
    def form(self):
        try:
            return self._form
        except AttributeError:
            self._form = self.form_class(self)
            return self._form

    def dir(self):
        return {
            'id':      None, # путь к ресурсу
            'uid':     None, # uid владельца
            'type':    None, # тип ресурса (папка или файл)
            'ctime':   0, # дата создания
            'mtime':   0, # дата изменения
            'utime':   0, # дата загрузки
            'name':    None, # имя
            'visible': 1, # признак видимости
            'labels':  [], # список меток
            'meta':    {}, # dict метаинформации,
        }

    def fill(self, address, **args):
        pass

    def _filter_fields(self, result, safe, unique_fields, exception_keyset):
        if safe:
            for item in self.not_saving_meta_fields:
                result['meta'].pop(item, None)
        if not unique_fields:
            for item in self.unique_fields:
                result['meta'].pop(item, None)
            for item in self.public_fields:
                result['meta'].pop(item, None)
        for k in exception_keyset:
            result.pop(k, None)
            result['meta'].pop(k, None)

    def dict(self, safe=False, unique_fields=True, exception_keyset=()):
        """ Получение данных ресурса в виде dict.

        :param safe: Фильтровать ли метаданные (для сохранения в базу)
        :param unique_fields: Фильтровать ли уникальные поля
        :param exception_keyset: Список ключей для дополнительной фильтрации
        """
        result = {}
        for attr in self.dir():
            result[attr] = deepcopy(getattr(self, attr))
        if self.address.is_storage:
            result['status_dict'] = getattr(self, 'status_dict', {})

        self._filter_fields(result, safe, unique_fields, exception_keyset)

        if isinstance(self, File):
            result['type'] = 'file'
        elif isinstance(self, Folder):
            result['type'] = 'dir'
        else:
            raise errors.NotFile()
        return result

    def rm(self, *args, **kwargs):
        '''
        Абстрактный метод удаления ресурса
        '''
        raise errors.MPFSNotImplemented()

    def check_rw(self, operation=None):
        pass

    def construct(self, uid, address, data={}):
        '''
        Обращение к фабрике - метод объекта
        '''
        return Resource.Construct(uid, address, data)

    @classmethod
    def Create(classname, uid, address, **data):
        raise errors.MPFSNotImplemented()

    @classmethod
    def Construct(cls, uid, address, data={}):
        import mpfs.core.factory
        return mpfs.core.factory.get_resource(uid, address, data)

    @classmethod
    def from_dict(cls, data, user_principal=None):
        """Вернуть объект класса, проинициализированного с помощью ``data``.

        :param user_principal: uid, от имени которого делается построение ресурса по данным
        :type data: dict
        :rtype: :class:`~Resource`
        """
        raise NotImplementedError()

    def update(self, data):
        fields = set(self.dir())
        for key, value in data.iteritems():
            if key in fields:
                if key in self.int_fields and isinstance(value, (str, unicode)):
                    try:
                        value = int(value)
                    except ValueError:
                        pass
                setattr(self, key, value)

    def copy(self, target_address, target_exist):
        raise errors.MPFSNotImplemented()

    def setprop(self, changes, deleted, **kwargs):
        if not kwargs.get('force'):
            for k in chain(changes, deleted):
                if k in self.static_fields:
                    err = errors.SetpropStaticFields(self.static_fields)
                    err.data = {'static_fields' : self.static_fields}
                    raise err
        if not kwargs.get('nomtime'):
            self.mtime = int(time.time())

    def get_full_index(self, root=True, safe=True):
        raise errors.MPFSNotImplemented()

    def index(self, root=False):
        raise errors.MPFSNotImplemented()

    def diff(self, version):
        return self._service.diff(self.address, version)

    def get_size(self):
        raise errors.MPFSNotImplemented

    def info(self, **params):
        return self.list(**params)

    def rename(self, target_address):
        self._service.rename(self.address, target_address)

    def set_request(self, request, force=False):
        if request:
            request.set_form(self.form, force=force)
            self.request = request
            self.form.args = request.args

    def prepare_transfer(self):
        pass

    @property
    def is_file(self):
        return isinstance(self, File)

    @property
    def is_folder(self):
        return isinstance(self, Folder)

    def lock(self):
        return self._service.lock_set(self.address.uid, self.address.path, self.version)

    def unlock(self):
        return self._service.lock_unset(self.address.uid, self.address.path, self.version)

    def check_lock(self):
        return self._service.lock_check(self.address.uid, self.address.path, self.version)

    def update_used(self, val=None):
        if val is None:
            val = int(self.size)
        if val:
            self._service.update_counter(self.address.uid, val)

    def load_views_counter(self):
        pass

    @property
    def is_shared(self, *a, **kw):
        return False

    @property
    def is_shared_root(self, *a, **kw):
        return False

    @property
    def is_shared_internal(self, *a, **kw):
        return False

    @property
    def is_group(self, *a, **kw):
        return False

    @property
    def is_group_root(self, *a, **kw):
        return False

    @property
    def is_group_internal(self, *a, **kw):
        return False

    @property
    def is_trash(self):
        return False

    @property
    def is_hidden(self):
        return False

    def is_infected(self):
        return False

    def is_public_internal(self):
        return False

    def is_smth_public(self):
        return False

    def has_same_public_parent(self, *args, **kwargs):
        return False

    @property
    def with_shared(self):
        '''
            Проверка, есть ли среди потомков свои или чужие расшаренные каталоги
        '''
        return bool(self.meta.get('with_shared'))

    def is_public_or_shared(self):
        return self.is_shared or self.is_group or self.is_public_internal() or self.is_smth_public()

    def should_yarovaya_mark_be_set(self):
        from mpfs.core.filesystem.dao.folder import FolderDAO
        if not FEATURE_TOGGLES_SETTING_YAROVAYA_MARK_ENABLED:
            return False
        return self.is_public_or_shared() or FolderDAO().has_parents_with_yarovaya_mark(self.address)

    @property
    def source_platform(self):
        """С какой платформы (ycrid) создан ресурс"""
        return self.meta.get('source_platform')

    @property
    def source_uid(self):
        """Кто создал ресурс"""
        return self.meta.get('source_uid') or self.owner_uid

    @property
    def owner_uid(self):
        return self.uid

    @property
    def owner_file_id(self):
        return self.meta.get('file_id')

    @property
    def resource_id(self):
        """Вернуть идентификатор ресурса.

        Используя этот идентификатор, можно найти соответствующую запись в базе.
        Для этого используется `uid` из `storage_address` и `file_id` из меты.

        :rtype: :class:`~ResourceId`
        """
        file_id = self.meta.get('file_id')

        if file_id is None and hasattr(self, 'file_id'):  # костыль для корневых папок
            # в постгресе у них теперь есть file_id и он не в мете
            file_id = self.file_id

        if file_id is None:
            return None

        return ResourceId(self.storage_address.uid, file_id)

    @staticmethod
    def generate_file_id(uid, path):
        """Сгенерировать `file_id` для `uid` и `path`.

        :type uid: str
        :type path: unicode | str
        :param path: путь ресурса (без uid'а)
        :rtype: str
        """

        if isinstance(uid, unicode):
            uid = uid.encode('utf-8')

        if isinstance(path, unicode):
            path = path.encode('utf-8')

        try:
            address = Address.Make(uid, path)
        except errors.AddressError:
            # К нашему глубочайшему сожалению есть такой код в МФПС:
            # >>> Resource.generate_file_id(uid, path)
            # >>> # Где:
            # >>> path == Address(path).id
            # ... True
            # https://github.yandex-team.ru/MPFS/MPFS/blob/release-2.38/lib/mpfs/core/operations/filesystem/store.py#L70

            # Пришлось поддержать
            raw_address_id = path
            address = Address(raw_address_id)
            address.change_uid(uid)  # если uid-ы отличаются

        seeds = (uid, address.path, str(time.time()))
        if address.is_system:
            seeds = (uid, address.path)

        return sha256(''.join(seeds)).hexdigest()

    def treat_symlink(self, *args, **kwargs):
        pass

    def save(self, *args, **kwargs):
        pass

    def save_selected(self, *fields):
        pass

    def setup_photoslice_time(self):
        pass

    def set_remove_date(self):
        pass

    def is_file_id_zipped(self):
        """Проверить зазипован ли file_id.

        После создания объекта типа :class:`Resource` определить
        был ли зазипован file_id невозможно. Поэтому был добавлен флаг
        file_id_zipped.

        Метод используется при перемещении в базе file_id из meta в data.

        :rtype: bool
        """
        return 'file_id_zipped' in self.meta

    def update_mids(self, mids):
        raise errors.MPFSNotImplemented

    def set_source_ids(self, source_ids):
        pass

    def clean_urls(self):
        if 'folder_url' in self.meta:
            self.meta.pop('folder_url')
        if 'file_url' in self.meta:
            self.meta.pop('file_url')
        self.meta['read_only'] = True


class Folder(Resource):
    """
    Базовый Класс папки

    Дополнительные поля:
    child_folders - dict со списком вложенных папок
    child_files   - dict со списком файлов
    """

    type = 'dir'
    form_class = FolderForm

    def __init__(self, uid, address, *args, **kwargs):
        super(Folder, self).__init__(uid, address, *args, **kwargs)
        data = kwargs.pop('data', {})

        if not self.address.is_folder:
            self.address.is_folder = True

        # проверяем что есть такая системная папка
        if address.storage_path not in settings.folders:
            raise errors.FolderNotFound()

        # проверяем разрешены ли подпапки в найденной системной папке
        if not address.is_system and not self.folders_allowed():
            raise errors.FolderNotFound(address.id)

        # устанавливаем все данные системной папки
        if address.is_system:
            # получаем данные папки из конфига
            options = settings.folders.get(address.storage_path)
            for attr, value in options.items():
                setattr(self, attr, value)

        if not getattr(self, 'name'):
            self.name = address.name

        self.child_folders = {}
        self.child_files = {}
        self.sorted_folders = []
        self.sorted_files = []
        self.child_map = {}
        self.type = 'dir'
        self.children_list = []
        self.full_index_map = {}
        self.loaded = False
        self.listing = []
        self.children_items = {'files' : [], 'folders' : []}
        self.timeline_filecount = 0

    @property
    def children(self):
        return list(self.child_folders.itervalues()) + list(self.child_files.itervalues())

    @classmethod
    def Construct(cls, uid, address, data={}):
        import mpfs.core.factory
        return mpfs.core.factory.get_resource(uid, address, data)

    @classmethod
    def Create(classname, uid, address, parent, **kwargs):
        # проверяем что такого еще нет
        try:
            classname.Construct(uid, address)
            if not kwargs.get('force'):
                raise errors.FolderAlreadyExist(address.id)
        except errors.ResourceNotFound:
            pass

        data = kwargs.get('data', {})
        if data.get('meta'):
            for k in classname.not_saving_meta_fields:
                data['meta'].pop(k, None)
        for k in classname.not_saving_fields:
            data.pop(k, None)
        # параметр data может быть расширен доп. данными в методе make_folder()
        service_cls = classname.service_class()
        response = service_cls.make_folder(address, parent, data)
        folder = classname(uid, address, **kwargs)
        try:
            folder.version = response.version
        except Exception:
            pass
        folder.set_parent(parent)
        return folder

    def dir(self):
        result = super(Folder, self).dir()
        result['hasfolders'] = 0
        return result

    def timeline(self, *args, **kwargs):
        self.timeline_filecount = self._timeline()

    def _timeline(self):
        count = 0
        self.form.args.forward = False
        self.request.args.forward = False
        self.load()
        for subfolder in self.children_items['folders']:
            subfolder.name_tree.extend(self.name_tree)
            for tg in ('amount', 'offset'):
                if tg in subfolder.form.args['bounds']:
                    del subfolder.form.args['bounds'][tg]
            count += subfolder._timeline()
        for subfile in self.children_items['files']:
            subfile.name_tree.extend(self.name_tree)
        return count+len(self.children_items['files'])

    def tree(self, **params):
        '''
        Рекурсивный проход по списку вложенных каталогов.
        По умолчанию проходит по первому уровню вложенности.
        '''
        def tree_element(element, element_children):
            return { 'this': element, 'list': element_children }

        level = params.pop('level', 1)
        parent = params.pop('parent', '/')
        if params.get('services'):
            return tree_element(self.dict(), [])
        children_list = []
        self.load()

        if not parent: parent = '/'

        if level > 1:
            level = level - 1
            params['level'] = level
            for subfolder in self.children_items['folders']:
                params['parent'] = subfolder.id
                children_list.append(subfolder.tree(**params))
        elif level > 0:
            for child_id in self.sorted_folders:
                children_list.append(tree_element(self.child_folders[child_id], []))
        else:
            self.children_items = {'files' : [], 'folders' : []}

        return tree_element(self.dict(), children_list)


    def parents(self, **params):
        id_split = filter(None, self.id.split('/'))

        def manage_tree(lvl=1, data={}):
            parent_address = Address(str(self.uid) + ':/' + '/'.join(id_split[:lvl]))
            parent = type(self).Construct(self.uid, parent_address)
            this_list = parent.list()
            result = []
            child_id = '/' + '/'.join(id_split[:lvl+1])
            for folder_item in parent.children_items['folders']:
                if folder_item.id == child_id:
                    if lvl < len(id_split):
                        result.append(manage_tree(lvl+1))
                    elif lvl == len(id_split):
                        result.append(self.tree(**params))
                else:
                    folder_item.load()
                    folder_dir = folder_item.dict()
                    result.append({'this' : folder_dir, 'list' : []})
            this_list['list'] = result
            return this_list

        if len(id_split) > 1:
            return manage_tree()
        else:
            return self.tree(**params)

    def list(self, **params):
        '''
        Листинг папки
        '''
        _listing = self.load(**params)
        _description = self.dict()
        return {'this': _description, 'list': _listing}

    def load(self):
        '''
        Частичная загрузка файлов и подпапок по параметрам
        '''
        if not self.loaded:
            self.listing = self._service.listing(self)
            self.loaded = True
        return self.listing

    def load_folders(self):
        '''
        Загрузка всех подпапок
        '''
        self.load() # пока так, дальше придумаем

    def load_files(self):
        '''
        Загрузка всех файлов
        '''
        self.load() # пока так, дальше придумаем

    def count_folders(self):
        self.load_folders()
        count = 0
        for folder in self.child_folders:
            if not self.child_folders[folder] is None:
                count += 1
        return count

    def count_files(self):
        self.load_files()
        count = 0
        for file in self.child_files:
            if not self.child_files[file] is None:
                count += 1
        return count

    def file(self, resource_id):
        '''
        Получить объект файла
        '''
        return self.subfile(resource_id)

    def subfolder(self, resource_id):
        '''
        Получить объект подпапки
        '''
        if not resource_id in self.child_folders:
            raise errors.FolderNotFound()
        return self.__class__(self.uid, self.address.get_child_folder(resource_id), data=self.child_folders[resource_id], parent=self)

    def subfile(self, resource_id, is_fullkey=False):
        if not is_fullkey:
            if not resource_id in self.child_files:
                raise errors.FileNotFound()
            child_address = self.address.get_child_file(resource_id)
            return self.file_class(self.uid, child_address, data=self.child_files[resource_id], parent=self)
        else:
            return self.file_class(self.uid, Address.Make(self.uid, resource_id, is_file=True), data=self.full_index_map[resource_id], parent=self)

    def rm(self, *args, **kwargs):
        '''
        Удаление папки со всеми подпапками и файлами
        '''
        self.load_folders()
        for dirname in self.child_folders:
            self.subfolder(dirname).rm(*args, **kwargs)

        self.load_files()
        for filename in self.child_files:
            self.file(filename).rm(*args, **kwargs)

        self.child_folders = {}
        self.child_files = {}

    def copy(self, target_path, target_exist=None, force=False):
        raise errors.MPFSNotImplemented()

    def construct_child_file(self, fid, data, version=None):
        address = self.address.get_child_file(fid)
        fle = self.file_class(self.committer, address, data=data, parent=self, version=version)
        if self.request:
            fle.set_request(self.request)
        self.children_items['files'].append(fle)
        return fle

    def construct_child_folder(self, fid, data, version=None, use_key=False):
        address = Address.Make(self.uid, data['key']) if use_key else self.address.get_child_folder(fid)
        fol = self.__class__(self.committer, address, data=data, parent=self, version=version)
        if self.request:
            fol.set_request(self.request)
        self.children_items['folders'].append(fol)

    def create_new_child_file(self, uid, address, **filedata):
        if not 'data' in filedata:
            filedata['data'] = {}
        filedata['data']['utime'] = int(time.time())
        result = self.create_child_file(uid, address, **filedata)
        return result

    def create_child_file(self, uid, address, **filedata):
        return self.file_class.Create(uid, address, self, **filedata)

    def create_new_child_folder(self, uid, address, **folderdata):
        if not 'data' in folderdata:
            folderdata['data'] = {}
        if 'meta' not in folderdata['data']:
            folderdata['data']['meta'] = {}
        folderdata['data']['utime'] = int(time.time())
        folderdata['data']['meta']['modify_uid'] = uid
        result = self.create_child_folder(uid, address, **folderdata)
        return result

    def create_child_folder(self, uid, address, **folderdata):
        return self.__class__.Create(uid, address, self, **folderdata)

    def folders_allowed(self, method=None):
        if method is None:
            return True
        return settings.folders[self.address.storage_path]['allow_folders']

    @classmethod
    def reset(cls):
        global folders_allowed_checked
        folders_allowed_checked = {}

    @classmethod
    def from_dict(cls, data, user_principal=None):
        if data['type'] != 'dir':
            raise errors.ResourceNotFound()

        address = Address.Make(data['uid'], data['key'])
        user_principal = user_principal or data['uid']
        return cls(user_principal, address, data=data['data'])


class File(Resource):
    """
    Класс файла

    При создании может принять dict data с атрибутами
    Это предусмотрено для случаев, когда мы делаем массовые операции над папкой -
    например rm() - и у нас сразу есть все данные файла
    """

    type = 'file'
    form_class = FileForm

    not_saving_meta_fields = (
        'office_online_url',
    ) + Resource.not_saving_meta_fields

    fields = Resource.fields + ['mimetype', 'size', 'source']
    main_fields = Resource.main_fields + ['mimetype', 'source', 'size']

    def __init__(self, *args, **kwargs):
        super(File, self).__init__(*args, **kwargs)
        self.setup(**kwargs)

    def setup(self, **args):
        # Устанавливаем source по умолчанию
        if not self.source and self._service:
            self.source = self._service.name

        #Устанавливаем урл файла по умолчанию для владельца файла
        self.meta['file_url'] = self.get_file_url(**args)

        # Преобразуем size в int
        self.size = int(self.size or 0)

        # Выставляем drweb по умолчанию
        self.meta['drweb'] = int(self.meta.get('drweb') or 0)

        if ('office_access_state' in self.meta and
                self.meta['office_access_state'] is None):
            self.meta['office_access_state'] = OfficeAccessStateConst.get_default_value()

    def list(self, **params):
        return {'this': self.dict(), 'list': []}

    def dir(self):
        result = super(File, self).dir()
        result['mimetype'] = None  # mime-type
        result['size'] = None
        result['source'] = None   # источник файла
        return result

    def get_url(self, **args):
        return { 'file': self.meta['file_url'] }

    def get_file_url(self, **args):
        raise errors.MPFSNotImplemented()

    def get_direct_url(self):
        raise errors.MPFSNotImplemented()

    def copy(self, target_path, target_exist=None, force=False):
        raise errors.MPFSNotImplemented()

    def get_size(self):
        return self.size

    def is_infected(self):
        drweb = self.meta.get('drweb', 0)
        return int(drweb) == 2

    @classmethod
    def from_dict(cls, data, user_principal=None):
        if data['type'] != 'file':
            raise errors.ResourceNotFound()

        address = Address.Make(data['uid'], data['key'])
        user_principal = user_principal or data['uid']
        return cls(user_principal, address, data=data['data'])
