# -*- coding: utf-8 -*-
"""

MPFS
CORE

Папка Disk

"""
import time
import os
import urllib
import datetime

from collections import OrderedDict, defaultdict
from copy import deepcopy

import mpfs.engine.process
from mpfs.core.office.logic.sharing_url import sync_office_fields_from_link_data
from mpfs.core.office.static import OfficeAccessStateConst, OfficeClientIDConst, EditorConst

from mpfs.engine.process import get_cloud_req_id
from mpfs.config import settings
from mpfs.core import history
from mpfs.core.address import Address, SymlinkAddress, ResourceId
from mpfs.core.filesystem import hardlinks
from mpfs.core.filesystem.symlinks import Symlink
from mpfs.core.filesystem.hardlinks.common import FileChecksums
from mpfs.core.filesystem.resources.base import Folder, File
from mpfs.core.filesystem.photoslice_filter import get_photoslice_time
from mpfs.core.filesystem.live_photo import LivePhotoFilesManager
from mpfs.core.services import zaberun_service, logreader_service
from mpfs.core.services.disk_service import Disk
from mpfs.core.user.constants import PHOTOUNLIM_AREA
from mpfs.common import errors
from mpfs.common.util import logger, filetypes
from mpfs.common.util.crypt import CryptAgent
from mpfs.common.util.ycrid_parser import YcridParser
from mpfs.common.static.tags import *
from mpfs.common.util.filetypes import getGroupByName
from mpfs.core.services.clck_service import clck
from mpfs.core.services.previewer_service import Previewer
from mpfs.common.errors.share import (GroupNoPermit,
                                      GroupNotFound,
                                      ShareNotFound,)
from mpfs.core.office.util import build_office_online_url, SharingURLAddressHelper
from mpfs.core.office.util import get_editor
from mpfs.core.filesystem.cleaner.models import DeletedStid, DeletedStidSources
from mpfs.metastorage.mongo.collections.filesystem import is_reindexed_for_quick_move

log = mpfs.engine.process.get_default_log()
error_log = mpfs.engine.process.get_error_log()
requests_log = mpfs.engine.process.get_requests_log()
listing_log = logger.get('stat-listing')
video_listing_log = logger.get('stat-video-listing')

crypt_agent = CryptAgent()
logreader = logreader_service.LogreaderService()


OFFICE_TURN_OFF_FOR_ALL = settings.office['open_document_button']['turn_off_for_all']
OFFICE_AVAILABLE_FOR_UIDS = settings.office['open_document_button']['available_for_uids']
OFFICE_AVAILABLE_FOR_YATEAM = settings.office['open_document_button']['available_for_yateam']
FEATURE_TOGGLES_DISABLE_OLD_PREVIEWS = settings.feature_toggles['disable_old_previews']
FEATURE_TOGGLES_BLOCKINGS = settings.feature_toggles['blockings']
FEATURE_TOGGLES_BLOCKINGS_ON_SPEED_LIMIT = settings.feature_toggles['blockings_on_speed_limit']
FEATURE_TOGGLES_BLOCKINGS_ON_OVERDRAW = settings.feature_toggles['blockings_on_overdraw']
FEATURE_TOGGLES_SETTING_YAROVAYA_MARK_ENABLED = settings.feature_toggles['setting_yarovaya_mark_enabled']
SETPROP_SYMLINK_FIELDNAME = settings.system['setprop_symlink_fieldname']
ALLOWED_MIMETYPES_FOR_ORIGINAL_INLINING = frozenset(settings.system['allowed_mimetypes_for_original_inlining'])
ENABLE_CHECK_MIMETYPE_ON_ORIGINAL_INLINING = settings.system['enable_check_mimetype_on_original_inlining']


def get_blockings_collection():
    from mpfs.metastorage.mongo.collections.blockings import BlockingsCollection
    return BlockingsCollection()


def is_original_inlining_allowed(mimetype):
    """

    :param str mimetype:
    :return bool:
    """
    return mimetype in ALLOWED_MIMETYPES_FOR_ORIGINAL_INLINING or not ENABLE_CHECK_MIMETYPE_ON_ORIGINAL_INLINING


class BlockingsController(object):
    def get_blockings(self, hid_or_hids, uid):
        if FEATURE_TOGGLES_BLOCKINGS_ON_SPEED_LIMIT:
            from mpfs.core.filesystem.quota import Quota
            quota = Quota()
            end_time = quota.get_download_speed_limit_end_time(uid)
            if end_time:
                if isinstance(hid_or_hids, list):
                    return [self.__imitate_blockings_return_value(x, end_time) for x in hid_or_hids]
                else:
                    return self.__imitate_blockings_return_value(hid_or_hids, end_time)
        if FEATURE_TOGGLES_BLOCKINGS_ON_OVERDRAW:
            from mpfs.core.user.base import User
            if User(uid).is_overdrawn():
                end_time = int(time.mktime((datetime.datetime.now() + datetime.timedelta(days=1)).timetuple()))
                if isinstance(hid_or_hids, list):
                    return [self.__imitate_blockings_return_value(x, end_time) for x in hid_or_hids]
                else:
                    return self.__imitate_blockings_return_value(hid_or_hids, end_time)

        if isinstance(hid_or_hids, list):
            return get_blockings_collection().find(hid_or_hids)
        else:
            return get_blockings_collection().get(hid_or_hids)

    @staticmethod
    def __imitate_blockings_return_value(hid, end_time):
        """Возвращается значение в таком же виде, как лежит в коллецкии blockings"""
        return {'_id': hid, 'public': True, 'simple': {'public': end_time}}


class BlockingsMixin(object):
    blockings_enabled_handlers = ('public_info', 'public_list')
    _zaberun_service = zaberun_service.Zaberun()

    @property
    def unlimited_users(self):
        return self._zaberun_service.unlimited_users

    def get_blockings(self):
        hid = getattr(self, 'hid', None)
        address = getattr(self, 'address', None)
        if hid and address:
            return BlockingsController().get_blockings(hid, address.uid)

    def _is_blockings_needed(self):
        # FIXME: we shouldn't check request here
        # BUT we cant't modify result of this function in core.base.info() or in filesystem.info()
        # because formatter mpfs.frontend.formatter.disk.json.JSON.info()
        # totally ignore result of this function.
        # Also we can't get blockings in formatter because formatter works after all exception handling is done
        # and if get_blockings() throw any exception (theoretically it can) we wont handle it properly
        # as @vikham says.
        is_proper_handler = self.request and hasattr(self.request.http_req, 'view_args') \
            and self.request.http_req.view_args['path'].strip('/').split('/')[-1] in self.blockings_enabled_handlers
        # Проверяем, что пользователь не в списке unlimited (см. https://jira.yandex-team.ru/browse/CHEMODAN-16058)
        is_needed = is_proper_handler and (self.uid not in self.unlimited_users)
        return is_needed


MAX_PREVIEW_SIZE = (2 << 31) - 1


class MPFSResource(object):
    """
    Класс с общим кодом `MPFSFolder` и `MPFSFile`
    """

    @property
    def comment_ids(self):
        if self.address.is_storage:
            return {}
        return {
            'public_resource': str(self.resource_id),
            'private_resource': str(self.resource_id),
        }

    @property
    def history(self):
        try:
            return self._history
        except AttributeError:
            self._history = history.History(self)
            return self._history


class MPFSFile(BlockingsMixin, MPFSResource, File):

    service_class = Disk

    fields = File.fields + ['hid', 'media_type']
    main_fields = File.main_fields + ['hid', 'media_type']
    hardlink_fields = ('md5', 'sha256', 'size')

    not_saving_fields = (
        'original_id',
        'original_parent_id',
        'append_time',
        'id',
        'uid',
    ) + File.not_saving_fields

    not_saving_meta_fields = (
        'preview',
        'photoslice_time',
        'exif_url',
        'custom_preview',
        'digest_url',
        'file_url',
        'fotki_data_url',
        'thumbnail',
        'wh_version',
        'sizes',
        'group',
        'uid',
        'id',
        'modify_uid',  # сохраняем его только для share/group-ресурсов.
        'revision',
        'versioning_status',
        'autouploaded',
    ) + File.not_saving_meta_fields

    unique_fields = (
        'file_id',
    )

    required_stid_fields = {'file_mid', 'digest_mid'}
    optional_stid_fields = {'pmid'}
    stid_fields = tuple(required_stid_fields | optional_stid_fields)

    def __init__(self, uid, address, **args):
        File.__init__(self, uid, address, **args)
        # FIXME: Костыль ради CHEMODAN-14123, принудительно устанавливающий media_type = 'video' если есть video_info
        if hasattr(self, 'meta') and 'video_info' in self.meta:
            setattr(self, u'media_type', u'video')
            setattr(self, u'mediatype', u'video')
        data = args.get('data', {})
        if data:
            self.meta['wh_version'] = args.get('version')
            self.version = data.get('version')
            self.key = data.get('key')
        # Возвращает настоящую версию ресурса из базы
        if self.version is not None:
            self.meta['revision'] = long(self.version)
        self.name = self.address.name

        modify_uid = self.meta.get('modify_uid', None)
        if modify_uid is None:
            modify_uid = self.uid
        self.meta['modify_uid'] = modify_uid
        self.meta['autouploaded'] = self.is_autouploaded

    def list(self, **params):
        ret = super(MPFSFile, self).list(**params)
        if FEATURE_TOGGLES_BLOCKINGS:
            if self._is_blockings_needed() and 'this' in ret and 'meta' in ret['this']:
                blockings = BlockingsController().get_blockings(self.hid, self.address.uid)
                if blockings:
                    blockings.pop('_id', None)
                    # It doesn't matter if we update returning data, because formatter just ignore it.
                    ret['this']['meta']['blockings'] = blockings
                    # And that's where magic begins: lately formatter will get self.meta through request.form
                    # and copy it to output meta, so we have to set self.meta['blockings'] here
                    # to get it in info handler output.
                    self.meta['blockings'] = blockings
        append_meta_to_office_files(self, self.request)
        return ret

    def dir(self):
        result = super(MPFSFile, self).dir()
        result['hid'] = None
        result['media_type'] = None
        return result

    def get_preview_mime(self):
        if self.mimetype in ('image/gif', 'image/png', 'image/x-png'):
            return self.mimetype
        else:
            return 'image/jpeg'

    def fill(self, address, **args):
        version = args.get('version')
        loaded = self._service.value(address, version)
        if loaded.value is None:
            raise errors.FileNotFound(address.id)
        if loaded.value.data.get('type') == 'dir' or address.is_storage:
            raise errors.NotFile()
        for k, v in loaded.value.data.iteritems():
            setattr(self, k, v)
        self.meta['wh_version'] = loaded.version
        self.version = loaded.value.version

    def dstore_data(self):
        #=======================================================================
        # Возвращаются только те поля, которые нужно сохранить
        # при дельта-обновлении
        #=======================================================================
        fields = ('file_url', 'digest_url', 'sizes', 'file_mid', 'digest_mid',
                  'sha256', 'md5', 'size', 'preview', 'file_id',
                  'previews', 'preview_sizes', 'drweb', 'uid', 'id', 'pmid', 'emid')
        data = self.dict(safe=True)
        for k in fields:
            try:
                data.pop(k)
            except KeyError:
                pass
            try:
                data['meta'].pop(k)
            except KeyError:
                pass
        return data

    @classmethod
    def build_previews(cls, uid, file, content_type=None):
        """
        Генерирует превьюшки для ресурса.

        :param uid: uid пользователя для которого генерятся превьюшки, если генерятся публичные, то нужно передать "0"
        :param file: файл для которого генерятся превьюшки.
        :param content_type: MIME-тип превьюшек.
        :return: Маппинг имён превьюшек на url-ики превьюшек, вида {"S": "http://...", ...}
        """
        from mpfs.core.user.constants import PUBLIC_UID

        content_type = content_type or file.get_preview_mime()
        service = cls.service_class()
        previews = OrderedDict()
        is_file_public = uid == PUBLIC_UID
        preview_params = {'content_type': content_type, 'inline': True, 'eternal': not is_file_public}

        if FEATURE_TOGGLES_DISABLE_OLD_PREVIEWS:
            previews_stids = {}
            big_preview_stid = file.meta.get('pmid')
        else:
            previews_stids = file.meta.get('previews', {})
            big_preview_stid = file.meta.get('pmid', previews_stids.get('XXXL'))

        if file.media_type == 'image':
            # ссылка на оригинал бывает только у изображений
            previews['ORIGINAL'] = service.url(
                uid,
                file.file_mid(),
                file.name,
                content_type=file.mimetype,
                inline=is_original_inlining_allowed(file.mimetype),
                fsize=file.get_size(),
                hid=file.hid,
                media_type=file.media_type,
                expire_seconds=None,  # default_expire_seconds
                md5=file.meta.get('md5'))

        if big_preview_stid:
            previews['DEFAULT'] = service.preview_url(uid, big_preview_stid, file.name, **preview_params)

        if settings.feature_toggles['dynamicpreviews'] and big_preview_stid:
            for size in settings.system['preview_names']:
                previews[size] = previews['DEFAULT'] + '&size=%s&crop=0' % size
            # генерим шаблон кастомного превью, size и crop в котором будут позже заменены на размеры указанные клиентом
            previews['C'] = previews['DEFAULT'] + '&size=%s&crop=%s' % (CUSTOM_SIZE, CUSTOM_CROP)
        else:
            for size in settings.system['preview_names']:
                preview_stid = previews_stids.get(size)
                if preview_stid:
                    previews[size] = service.url(uid, preview_stid, file.name, **preview_params)

        return previews

    def has_previews(self):
        """Проверяет есть ли у файла превьюшки."""
        if FEATURE_TOGGLES_DISABLE_OLD_PREVIEWS:
            return bool(self.meta.get('pmid'))
        else:
            return bool(self.meta.get('previews') or self.meta.get('pmid'))

    def setup_previews(self, uid=None):
        """Генерирует ссылки на превьюшки и устанавливает их в соответствующие атрибуты файла."""
        uid = uid or self.committer
        if self.has_previews():
            previews = self.build_previews(uid, self)
            if 'S' in previews:
                self.meta['preview'] = previews['S']
                self.meta['custom_preview'] = previews['S']
            if 'XXXS' in previews:
                self.meta['thumbnail'] = previews['XXXS']
            self.meta['sizes'] = [{'name': k, 'url': v} for k, v in previews.iteritems()]

    def regenerate_preview(self, timeout=None):
        """Перегенирить превью, а для видео дополнительно video_info"""
        old_pmid = self.meta.get('pmid')

        result = Previewer().regenerate_preview(self.name,
                                             self.size,
                                             self.mimetype,
                                             self.file_mid(),
                                             timeout=timeout)
        self.meta['pmid'] = result.pmid
        if result.has_video_info():
            self.meta['video_info'] = result.video_info
        if result.original_width is not None and result.original_height is not None:
            self.meta['width'] = result.original_width
            self.meta['height'] = result.original_height
        if result.rotate_angle is not None:
            self.meta['angle'] = result.rotate_angle
        self.save(skip_parents_check=True)

        if old_pmid:
            objs = [DeletedStid(stid=old_pmid, stid_type='pmid', stid_source=DeletedStidSources.REGENERATE_PREVIEW)]
            DeletedStid.controller.bulk_create(objs, get_size_from_storage=True)

    def regenerate_digest(self):
        """Перегенирить digest"""
        old_digest = self.digest_mid()
        new_digest = Previewer().regenerate_digest(self.file_mid())
        self.meta['digest_mid'] = new_digest
        self.save(skip_parents_check=True)

        if old_digest:
            objs = [
                DeletedStid(stid=old_digest, stid_type='digest_mid', stid_source=DeletedStidSources.REGENERATE_DIGEST)
            ]
            DeletedStid.controller.bulk_create(objs, get_size_from_storage=True)

    def setup(self, **args):
        version = args.get('version')

        # Выставляем media_type
#         self.media_type = filetypes.getGroupByMimetype(self.mimetype, self.name)
        if not hasattr(self, 'media_type') or not getattr(self, 'media_type'):
            self.media_type = filetypes.getGroupByName(self.name, self.mimetype)

        super(MPFSFile, self).setup(**args)
        self.meta['wh_version'] = version
        self.setup_previews()

        self.meta['digest_url'] = self.get_digest_url(**args)
        if 'emid' in self.meta:
            self.meta['exif_url'] = self.get_exif_url(**args)

        if 'fotki_data_stid' in self.meta:
            self.meta['fotki_data_url'] = self.get_fotki_data_url(**args)

        photoslice_time = get_photoslice_time(self.address.path, self.type, self.mimetype,
                                              self.meta.get('md5'), self.meta.get('etime'), self.ctime, self.mtime)
        if photoslice_time:
            self.meta['photoslice_time'] = photoslice_time

    def rm(self, *args, **kwargs):
        size = self.get_size()
        try:
            with_changelog = kwargs['changelog']
        except KeyError:
            with_changelog = True

        if self.meta.get('is_live_photo'):
            LivePhotoFilesManager.remove_live_video(self.address)

        if with_changelog:
            self._service.remove(self.address, self.version, data=self.dict(safe=True))
        else:
            self._service.remove_single(self.address, self.version, changelog=False)

        return size

    def get_full_index(self, root=True, safe=True):
        return {self.id : self.dict(safe=safe)}

    @classmethod
    def Create(classname, uid, address, parent, **kw):
        data = kw.get('data')

        if data and isinstance(data, dict):
            data = deepcopy(data)

            value = {
                'visible'    : 1,
                'ctime'      : int(data.pop('ctime', time.time())),
                'mtime'      : int(data.pop('mtime', time.time())),
                'meta'       : {},
            }
            if 'source_uid' in data:
                value['source_uid'] = data.pop('source_uid')

            for k in classname.fields:
                try:
                    v = data.pop(k)
                except KeyError:
                    pass
                else:
                    if k not in classname.static_fields:
                        value[k] = v

            value['meta'].update(data)
            value['size'] = int(value['size'])
            value['media_type'] = getGroupByName(address.name, value['mimetype'])

            value['source_platform'] = YcridParser.get_platform(get_cloud_req_id())
            for k in classname.not_saving_fields:
                value.pop(k, '')
                value['meta'].pop(k, None)

            for k in classname.not_saving_meta_fields:
                value['meta'].pop(k, None)

            if not value['meta'].get('file_id'):
                value['meta']['file_id'] = classname.generate_file_id(uid, address.path)

            hardlink = classname.getSelfHardlink(value)
            value['hid'] = hardlink._id

            external_setprop = SETPROP_SYMLINK_FIELDNAME in value['meta']

            result = classname.service_class().put(address, value, external_setprop=external_setprop)
            value['type'] = 'file'
            kw['data'] = value
            file_item = classname(uid, address, **kw)
            file_item.version = int(result.version)
            file_item.name = address.name
        else:
            raise errors.MakeFileError()
        file_item.set_parent(parent)
        return file_item

    @classmethod
    def getSelfHardlink(classname, filedata):
        sha256 = filedata['meta']['sha256']
        md5    = filedata['meta']['md5']
        size   = filedata['size']
        return hardlinks.HardLink(md5, size, sha256, load=None)

    def file_mid(self):
        return self.meta.get('file_mid')

    def get_video_info(self):
        return self.meta.get('video_info')

    def digest_mid(self):
        return self.meta.get('digest_mid')

    def get_fotki_data_stid(self):
        return self.meta.get('fotki_data_stid')

    def preview_mid(self):
        return self.meta.get('pmid')

    def thumbnail_mid(self):
        if FEATURE_TOGGLES_DISABLE_OLD_PREVIEWS:
            return self.meta.get('thumbnail')
        else:
            previews = self.meta.get('previews', {})
            return previews.get('XXXS')

    def md5(self):
        return self.meta.get('md5')

    def sha256(self):
        return self.meta.get('sha256')

    def get_checksums(self):
        return FileChecksums(self.md5(), self.sha256(), self.size)

    def drweb(self):
        return self.meta.get('drweb')

    def get_width(self):
        return self.meta.get('width')

    def get_height(self):
        return self.meta.get('height')

    def get_angle(self):
        return self.meta.get('angle')

    def get_albums_exclusions(self):
        exclusions = self.meta.get('albums_exclusions')
        # Значение в базке может быть None, поэтому не
        # exclusions = self.meta.get('albums_exclusions', [])
        if exclusions is None:
            return []
        return exclusions

    def possible_link(self):
        try:
            if self.meta['md5'] and self.meta['sha256'] and self.size:
                return True
        except Exception:
            pass
        return False

    def get_url(self, **args):
        inline = args.get('inline')
        if inline:
            self.setup(inline=inline)
        return {
            'file':   self.meta['file_url'] ,
            'digest': self.meta['digest_url']
        }

    def get_public_download_url(self, **args):
        return self._service.public_download_url(
            self.file_mid(),
            self.name,
            **args
        )

    def get_file_url(self, **args):
        return self._service.url(
            self.committer,
            self.file_mid(),
            self.name,
            content_type=self.mimetype,
            inline=args.get('inline'),
            fsize=self.get_size(),
            hid=self.hid,
            media_type=self.media_type,
            expire_seconds=args.get('expire_seconds'),
            md5=self.meta.get('md5'),
            owner_uid=args.get('owner_uid'),
        )

    def get_url_with_hash(self, **args):
        from mpfs.core.filesystem.quota import Quota
        quota = Quota()
        download_speed_limited = quota.download_speed_limited(self.address.uid)
        url = self._service.url(
            args.get('override_uid') or self.committer,
            self.file_mid(),
            args.get('override_name') or self.name,
            content_type=self.mimetype,
            inline=args.get('inline'),
            hash=args.get('hash') or self.get_public_hash(),
            limit=int(download_speed_limited),
            fsize=self.get_size(),
            hid=self.hid,
            media_type=self.media_type,
            expire_seconds=args.get('expire_seconds'),
            owner_uid=args.get('owner_uid') or self.committer,
        )
        result = {
                'file': url,
                'digest': self.meta['digest_url'],
        }

        return result

    def get_digest_url(self, **args):
        return self._service.url(
            self.committer,
            self.digest_mid(),
            self.name,
            inline=args.get('inline'),
            expire_seconds=args.get('expire_seconds'),
        )

    def get_exif_url(self, **args):
        return self._service.url(
            self.committer,
            self.meta['emid'],
            self.name,
            content_type='text/json',
            inline=True,
            expire_seconds=args.get('expire_seconds'),
        )

    def get_fotki_data_url(self, **args):
        return self._service.url(
            self.committer,
            self.get_fotki_data_stid(),
            self.name,
            content_type='text/json',
            inline=True,
            expire_seconds=args.get('expire_seconds')
        )

    def get_preview_url(self, **args):
        return self._service.url(
            self.committer,
            self.preview_mid(),
            self.name,
            content_type=self.get_preview_mime(),
            inline=True,
            expire_seconds=args.get('expire_seconds'),
        )

    def get_aviary_preview_url(self, **args):
        from mpfs.core.user.constants import PUBLIC_UID

        big_preview = None
        try:
            big_preview = self.meta['pmid']
        except KeyError:
            if not FEATURE_TOGGLES_DISABLE_OLD_PREVIEWS:
                previews = self.meta.get('previews', {})
                big_preview = previews.get('XXXL')

        if big_preview:
            kw = {
                  'content_type' : self.get_preview_mime(),
                  'inline'  : True,
                  'eternal' : False,
                  }

            return self._service.preview_url(PUBLIC_UID, big_preview, self.name, **kw)
        else:
            return None

    def get_aviary_orig_url(self, **args):
        from mpfs.core.user.constants import PUBLIC_UID
        kw = {
            'content_type': self.mimetype,
            'inline': True,
            'eternal': False,
            'size': '%sx%s' % (MAX_PREVIEW_SIZE, MAX_PREVIEW_SIZE)
        }
        return self._service.preview_url(PUBLIC_UID, self.meta['file_mid'], self.name, **kw)

    def get_thumbnail_url(self, **args):
        return self._service.url(
            self.committer,
            self.thumbnail_mid(),
            self.name,
            content_type=self.get_preview_mime(),
            inline=True,
            expire_seconds=args.get('expire_seconds'),
        )

    def get_direct_url(self, modified):
        if modified:
            stamp = int(
                time.mktime(
                    time.strptime(modified, '%a, %d %b %Y %H:%M:%S GMT')
                )
            ) +  time.timezone
            if stamp > self.mtime:
                raise errors.DocviewerNotModifiedError()

        return {
            'url'     : self._service.direct_url(self.file_mid()),
            'mimetype': self.mimetype,
            'name'    : self.name
        }

    def setprop(self, changes, deleted, **kwargs):
        if isinstance(changes.get('meta'), dict):
            changes.update(changes.pop('meta'))
        super(MPFSFile, self).setprop(changes, deleted, **kwargs)
        changed = False
        update_hid = False
        data = self.dict(safe=True)

        for k, v in changes.iteritems():
            if k in self.main_fields and data[k] != v:
                data[k] = v
                changed = True
            elif k not in self.fields:
                if k not in data['meta'] or data['meta'][k] != v:
                    data['meta'][k] = v
                    changed = True
                    if k in self.hardlink_fields:
                        update_hid = True

        for k in deleted:
            data['meta'].pop(k, None)

        if update_hid:
            hardlink = self.getSelfHardlink(data)
            data['hid'] = hardlink._id

        self.external_setprop = SETPROP_SYMLINK_FIELDNAME in data['meta']

        if changed or deleted:
            self.update(data)
            if not kwargs.get('nomtime'):
                self.mtime = int(time.time())
            self.save(external_setprop=self.external_setprop, changes=changes)
            self.setup_photoslice_time()

    def save(self, exceptions=(), **kwargs):
        value = self.dict(safe=True)

        kwargs['external_setprop'] = SETPROP_SYMLINK_FIELDNAME in value['meta']

        not_saving_fields = ('id',)
        for item in not_saving_fields:
            if item in value:
                value.pop(item, '')
        for k in set(value):
            if value[k] in ('', None, [], {}):
                value.pop(k)
        self.prev_version = self.version
        res = self._service.edit_file(self.storage_address, value, self.version, **kwargs)

        try:
            self.version = int(res.version)
        except Exception:
            pass

    def index(self, root=False):
        result = {
            'key'    : self.address.path,
            'type'   : self.type,
            'md5'    : self.meta.get('md5', ''),
            'size'   : self.size,
            'op'     : 'new',
            'sha256' : self.meta.get('sha256', ''),
        }
        self.op = 'new'
        if root:
            return {'version' : self.meta['wh_version'], 'result': [result]}
        else:
            return [result]

    def all_storage_ids(self):
        '''
        Вернуть все хранилищные id
        '''
        result = []
        for item in ('file_mid', 'digest_mid', 'pmid'):
            mid = self.meta.get(item, None)
            if mid:
                result.append(mid)
        return result

    def tree_index(self):
        result = {}
        result['key'] = self.address.path
        result['version'] = self.version
        result['data'] = self.dict()
        return {'this' : result, 'list' : []}

    def print_to_listing_log(self):
        if FEATURE_TOGGLES_DISABLE_OLD_PREVIEWS:
            is_add_to_listing_log_needed = not self.meta.get('pmid')
        else:
            is_add_to_listing_log_needed = not self.meta.get('pmid') and not self.meta.get('previews')

        if is_add_to_listing_log_needed:
            listing_log.info('uid: %s hid: %s stid: %s mimetype: %s size: %s mtime: %s utime: %s path: %s media_type: %s' %
                         (self.uid, self.hid, self.file_mid(), self.mimetype, self.size, self.mtime, self.utime,
                          urllib.quote(self.address.path.encode("utf-8")), self.media_type))

        if ('video_info' in self.meta or self.media_type == 'video'):
            video_listing_log.info('uid: %s hid: %s stid: %s mimetype: %s size: %s mtime: %s utime: %s path: %s media_type: %s has_info: %s has_preview: %s' %
                         (self.uid, self.hid, self.file_mid(), self.mimetype, self.size, self.mtime, self.utime,
                          urllib.quote(self.address.path.encode("utf-8")),
                          self.media_type, 'video_info' in self.meta, 'pmid' in self.meta))

    def update_mids(self, mids):

        """
        Проверяет у ресурса mids-ы и при необходимости меняет их.

        :param dict mids: словарь mid-ов, ключи: file_mid, pmid, digest_mid
        :return tuple is_changed, old_mids: признак того, что ресурс изменился, словарь со старыми mid'ами
        """
        old_mids = {}
        is_changed = False
        for mid_name in self.stid_fields:
            if mid_name not in mids:
                continue
            new_mid = mids[mid_name]
            if not new_mid and mid_name in self.required_stid_fields:
                raise ValueError('New value of %s is incorrect - %s' % (mid_name, new_mid))
            old_mid = self.meta.get(mid_name)
            if old_mid != new_mid:
                old_mids[mid_name] = old_mid
                self.meta[mid_name] = new_mid
                is_changed = True

        return is_changed, old_mids

    @classmethod
    def from_dict(cls, data, user_principal=None):
        if data['type'] != 'file':
            raise errors.ResourceNotFound()

        address = Address.Make(data['uid'], data['key'])
        user_principal = user_principal or data['uid']
        return cls(user_principal, address, version=data['wh_version'], data=data['data'])

    @property
    def is_autouploaded(self):
        if self.address.storage_name == PHOTOUNLIM_AREA:
            return True

        from mpfs.core.filesystem.base import Filesystem
        photostream_address = Filesystem.get_photostream_address(self.uid)
        return self.address.is_subfolder(photostream_address)

    def set_source_ids(self, source_ids):
        self.meta['source_ids'] = source_ids


class MPFSFolder(BlockingsMixin, MPFSResource, Folder):

    service_class = Disk

    not_saving_fields = (
        'uid',
        'hasfolders',
    )

    not_saving_meta_fields = (
        'wh_version',
        'with_shared',
        'group',
        'shared',
        'uid',
        'folder_url',
        'id',
        'modify_uid',
        'revision'
    )

    unique_fields = (
        'file_id',
    )

    def __init__(self, uid, address, **args):
        data = args.get('data')
        version = args.get('version')
        Folder.__init__(self, uid, address, **args)
        self.key = None
        if version:
            self.meta['wh_version'] = version
        if data:
            try:
                self.version = data['version']
            except KeyError:
                self.version = None
            self.key = data.get('key')
        # Возвращает настоящую версию ресурса из базы
        if self.version:
            self.meta['revision'] = long(self.version)
        self.name = self.address.name

        modify_uid = self.meta.get('modify_uid', None)
        if modify_uid is None:
            modify_uid = self.uid
        self.meta['modify_uid'] = modify_uid

    def fill(self, address, **args):
        self.init(**args)

    def init(self, **args):
        loaded = self._service.value(self.address, args.get('version'))
        if loaded.value is None:
            if not self.address.is_storage:
                raise errors.FolderNotFound(self.address.id)
        else:
            for attr, val in loaded.value.data.iteritems():
                setattr(self, attr, val)
            self.meta['wh_version'] = loaded.version
            # Возвращает настоящую версию ресурса из базы
            self.meta['revision'] = long(loaded.value.version)
            self.version = loaded.value.version
            self.prev_version = loaded.version

    def get_folder_url(self, **args):
        return self._service.folder_url(
            self.committer,
            self.visible_address.path,
            self.visible_address.name,
            content_type='application/zip',
            inline=args.get('inline'),
            expire_seconds=args.get('expire_seconds'),
            owner_uid=args.get('owner_uid'),
        )

    @classmethod
    def Create(classname, uid, address, parent, **kwargs):
        try:
            data = kwargs['data']
        except KeyError:
            file_id = None
            kwargs['data'] = {
                              'meta' : {},
                              }
        else:
            try:
                meta_data = data['meta']
            except KeyError:
                file_id = None
                kwargs['data']['meta'] = {}
            else:
                file_id = meta_data.get('file_id')
        if not file_id:
            kwargs['data']['meta']['file_id'] = classname.generate_file_id(uid, address.path)

        return super(MPFSFolder, classname).Create(uid, address, parent, **kwargs)

    def rm(self, *args, **kwargs):
        '''
        Удаление папки со всеми подпапками и файлами
        '''
        size = self.get_size()
        if kwargs.get('drop'):
            result = self._service.remove(self.address, self.version, data=self.dict(safe=True))
            try:
                self.deleted_version = int(result.version)
            except ValueError:
                self.deleted_version = 0
        else:
            self._service.remove_single(self.address, self.version, data=self.dict(safe=True))
        return size

    def rm_content(self, *args, **kwargs):
        size = self._service.remove_folder_content(self.address.uid, self.address.path)
        self.update_used(size * -1)
        return size

    def setprop(self, changes, deleted, **kwargs):
        force = kwargs.get('force')
        super(MPFSFolder, self).setprop(changes, deleted, **kwargs)
        self.mtime = int(time.time())
        data = self.dict()
        for k, v in changes.iteritems():
            if k in self.main_fields:
                data[k] = v
            else:
                data['meta'][k] = v
        for k in deleted:
            data['meta'].pop(k, None)
        for k in ('original_id', 'original_parent_id', 'append_time', 'id'):
            data.pop(k, '')
            data['meta'].pop(k, '')
        for k in set(data):
            if data[k] is None or data[k] == '': data.pop(k)
        for k, v in data.iteritems():
            setattr(self, k, v)
        data = self.dict()
        for k in self.not_saving_meta_fields:
            if not (force and k in changes):
                data['meta'].pop(k, '')
        for k in self.not_saving_fields:
            if not (force and k in changes):
                data.pop(k, None)

        self.external_setprop = SETPROP_SYMLINK_FIELDNAME in data['meta']
        result = self._service.edit_folder(self.address, data, external_setprop=self.external_setprop)
        self.version = result.value.version

    def save(self, exceptions=()):
        data = self.dict()
        for k in self.not_saving_meta_fields:
            if k not in exceptions:
                data['meta'].pop(k, None)
        for k in self.not_saving_fields:
            if k not in exceptions:
                data.pop(k, None)
        self.version = self._service.edit_folder(self.storage_address, data).value.version

    def save_selected(self, **fields):
        self.version = self._service.change_folder_fields(
                            self.storage_address, fields,
                            version=fields.pop('version', None))
        for k,v in fields.iteritems():
            setattr(self, k, v)

    def tree(self, **params):
        params['services'] = False
        return self._service.tree(self, **params)

    def info(self, **params):
        '''
        Info папки
        '''
        _listing =[]
        try:
            meta = self.request.meta
            if meta and (meta, list):
                content_params = self.request and self.request.meta and 'hasfolders' in self.request.meta
                if self.address.is_storage and not content_params and self.request.meta != []:
                    self.init()
                elif content_params or self.request.meta == []:
                    _listing = self._service.listing(self)
        except AttributeError:
            pass
        return {'this': self.dict(), 'list': _listing}

    def load_full_index_items(self):
        '''
            Загрузка всех файлов вглубь по дереву как объектов
        '''
        self.load()
        for child_folder in self.children_items['folders']:
            child_folder.load_full_index_items()

    def get_full_index(self, root=True, safe=True):
        """ Возвращает плоский список дерева элементов с корнем в текущей папке.

        Даные загружаются один раз и кешируются.

        :param root: Оставлять ли корень (текущую папку)
        :param safe: Фильтровать ли поля данных файлов (для хранения в базе)
        :return:
        """
        if not self.full_index_map:
            self.load()

            for child_file in self.child_files:
                self.child_files[child_file]['meta']['wh_version'] = self.meta['wh_version']
                child_file_instance = self.subfile(child_file)
                child_file_instance.meta['wh_version'] = self.meta['wh_version']
                self.full_index_map[child_file_instance.id] = child_file_instance.dict(safe=False)

            for child_folder in self.child_folders:
                self.child_folders[child_folder]['meta']['wh_version'] = self.meta['wh_version']
                subfolder = self.subfolder(child_folder)
                self.full_index_map.update(subfolder.get_full_index(root=True, safe=False))

            if self.id not in self.full_index_map:
                self.full_index_map[self.id] = self.dict(safe=False)

        full_index = self.full_index_map
        if not root or safe:
            full_index = deepcopy(self.full_index_map)

        if not root:
            full_index.pop(self.id, None)

        if safe:
            for file_data in full_index.itervalues():
                self._filter_fields(file_data, safe, unique_fields=True, exception_keyset=())

        return full_index

    def flat_index(self, root=False, mediatype=None):
        result = {}
        res, ver = self._service.flat_index(self.uid, self.id, mediatype=mediatype)
        for key, data in res.iteritems():
            address = Address('%s:%s' % (self.uid, key))
            version = data.get('version')
            if data['type'] == 'dir':
                resource = self.__class__(
                    self.committer,
                    address,
                    data=data,
                    version=version
                )
            elif data['type'] == 'file':
                resource = self.file_class(
                    self.committer,
                    address,
                    data=data,
                    version=version
                )
            result[resource.id] = resource.dict()
        result[self.id] = self.dict()
        return result

    def index(self, root=False):
        """
            Возвращает текущую и все нижележащие папки и файлы в виде списка.

            Не включает содержимое нижележащих общих папок для приглашенного пользователя.
            Если текущая папка находится внутри общей для приглашенного пользователя,
            то будет исключение StorageKeyNotFound
        """
        res, ver = self._service.flat_index(self.uid, self.id)
        result = [{'key' : self.id, 'type' : self.type, 'op' : 'new', },]
        for k, v in res.iteritems():
            each_data = {'key' : k, 'type' : v['type'], 'op' : 'new'}
            if v['type'] == 'file':
                each_data.update({
                    'md5'    : v['meta'].get('md5', ''),
                    'size'   : int(v['size']),
                    'sha256' : v['meta'].get('sha256', '')
                })
            result.append(each_data)
        if root:
            return {'result' : result, 'version' : str(ver)}
        else:
            return result

    def tree_index(self):
        return self._service.tree_index(self.uid, self.id)

    def is_photostream(self):
        return self.meta.get('folder_type') == 'photostream'

    def get_size(self):
        self.get_full_index()
        # TODO: check int size
        return reduce(
            lambda x, y: x + y,
            [0, ] +
            map(int, filter(
                None, map(lambda x: x.get('size'), self.full_index_map.itervalues())
                )
            )
        )

    @property
    def size(self):
        return self.get_size()

    def timeline(self, *args, **kwargs):
        self._service.timeline(self, *args, **kwargs)
        for subfile in self.children_items['files']:
            subfile.name_tree = filter(None, subfile.key.split(os.path.sep))[::-1]
            if self.request:
                subfile.print_to_listing_log()

    def list(self, **params):
        self.load(**params)
        _listing = []
        _description = self.dict()

        if FEATURE_TOGGLES_BLOCKINGS:
            if self._is_blockings_needed():
                # Then farce child files with blockings.
                # Collect hids to map hid -> file
                hids = {}
                for f in self.children_items['files']:
                    hid = getattr(f, 'hid', None)
                    if hid:
                        hids[hid] = f
                # retrive all blockings in one request
                if hids:
                    blockings_list = BlockingsController().get_blockings(hids.keys(), self.address.uid)
                    # implant blockings to appropriate children meta
                    for b in blockings_list:
                        hid = b.get('_id', None)
                        if hid:
                            file = hids[hid]
                            b.pop('_id', None)
                            file.meta['blockings'] = b

        for resource_type in ('folders', 'files'):
            for each in self.children_items[resource_type]:
                append_meta_to_office_files(each, self.request)
                _listing.append(each.dict())
                if self.request and each.type == 'file':
                    each.print_to_listing_log()
        return {'this': self.dict(), 'list': _listing}

    def get_url_with_hash(self, **args):
        from mpfs.core.filesystem.quota import Quota
        quota = Quota()
        download_speed_limited = quota.download_speed_limited(self.address.uid)
        url = self._service.folder_url(
            args.get('override_uid') or self.committer,
            args.get('hash') or self.get_public_hash(),
            self.name,
            content_type= 'application/zip',
            inline=args.get('inline'),
            hash=args.get('hash') or self.get_public_hash(),
            limit=int(download_speed_limited),
            expire_seconds=args.get('expire_seconds'),
            owner_uid=args.get('owner_uid'),
        )
        result = {
            'folder': url,
        }
        return result

    @classmethod
    def from_dict(cls, data, user_principal=None):
        if data['type'] != 'dir':
            raise errors.ResourceNotFound()

        address = Address.Make(data['uid'], data['key'])
        user_principal = user_principal or data['uid']
        return cls(user_principal, address, version=data['wh_version'], data=data['data'])


class DiskResource(object):

    source = 'disk'

    public_fields = ('symlink', 'public', 'public_hash', 'short_url', 'short_url_named', 'download_counter',
                     'page_blocked_items_num',)

    not_saving_meta_fields = (
        'short_url_named',
    )

    def __deepcopy__(self, memo):
        """
        https://st.yandex-team.ru/CHEMODAN-26167
        """
        return self

    def __init__(self, *args, **kwargs):
        short_url = self.meta.get('short_url')
        if short_url:
            self.meta['short_url_named'] = short_url + '?' + self.name

    def remove_symlink(self):
        self.meta.pop('symlink', None)
        self.save()

    def set_symlink(self, symlink):
        self.meta['symlink'] = symlink.address.id
        self.save()

    def get_symlink(self):
        return self.meta.get('symlink', None)

    def treat_symlink(self, copy_symlinks, rm_symlinks):
        _symlink = self.get_symlink()
        symlink = None
        if _symlink:
            symlink_address = SymlinkAddress(_symlink)
            try:
                symlink = Symlink(symlink_address)
            except errors.SymlinkNotFound:
                error_log.error("symlink not found, skipped")
            else:
                if copy_symlinks:
                    symlink.set_target(self.address)
                    symlink.set_resource_id(self.resource_id)
                else:
                    self.make_private(fields=('symlink',))
                    if rm_symlinks:
                        symlink.delete()
        if not (symlink or copy_symlinks):
            self.make_private(fields=('symlink',))

    def make_public(self, symlink):
        self.meta['symlink'] = symlink.address.id
        if FEATURE_TOGGLES_SETTING_YAROVAYA_MARK_ENABLED:
            self.meta['yarovaya_mark'] = True
        hsh = crypt_agent.encrypt(self.get_symlink())
        self.meta['public'] = 1
        self.meta['public_hash'] = hsh
        if self.get_public_url() is not None:
            if FEATURE_TOGGLES_DISABLE_OLD_PREVIEWS:
                has_preview = self.meta.get('pmid') or self.meta.get('screenshot')
            else:
                has_preview = self.meta.get('pmid') or self.meta.get('previews') or self.meta.get('screenshot')
            mode = 'image' if has_preview else 'default'
            _, short_url = clck.generate(self.get_public_url(), mode=mode)
            self.meta['short_url'] = short_url
            self.meta['short_url_named'] = short_url + '?' + self.name
            self.save(exceptions=('folder_type', ))

    def make_blocked(self):
        self.meta['blocked'] = 1
        self.save()

    def all_storage_ids(self):
        return []

    def make_unblocked(self):
        if self.is_blocked():
            del(self.meta['blocked'])
            self.save()

    def is_fully_public(self):
        return self.meta.get('symlink') is not None and \
            self.meta.get('public') is not None and \
            self.meta.get('public_hash') is not None and \
            self.meta.get('short_url') is not None and \
            self.meta.get('short_url_named') is not None

    def is_smth_public(self):
        return self.meta.get('symlink') is not None or \
            self.meta.get('public') is not None or \
            self.meta.get('public_hash') is not None or \
            self.meta.get('short_url') is not None or \
            self.meta.get('short_url_named') is not None

    def copy_public_info(self, data):
        for field in self.public_fields:
            self.meta[field] = data.meta.get(field)
        self.save()

    def is_blocked(self):
        return bool(self.meta.get('blocked'))

    def assert_blocked(self):
        resource_is_blocked_msg_tpl = 'Address "%s" resource is blocked.'
        if self.is_blocked():
            log.info(resource_is_blocked_msg_tpl % Address.Make(self.uid, self.path).id)
            raise errors.ResourceBlocked()

    def get_public_hash(self):
        return self.meta.get('public_hash')

    def get_public_url(self):
        return settings.system['public_urls']['resource'] % urllib.quote(self.meta.get('public_hash', None))

    def get_short_url(self):
        return self.meta.get('short_url')

    def load_views_counter(self):
        public_hash = self.meta.get('public_hash')
        if public_hash:
            counter = logreader.get_one_counter(public_hash)
            self.meta[VIEWS_COUNTER] = counter

    def get_short_url_named(self):
        return self.meta.get('short_url_named', None)

    @property
    def office_access_state(self):
        office_access_state = self.meta.get('office_access_state')
        if office_access_state is None:
            return OfficeAccessStateConst.get_default_value()
        return office_access_state

    @property
    def office_doc_short_id(self):
        return self.meta.get('office_doc_short_id')

    def remove_public(self):
        changed = False
        for public_field in ('public', 'download_counter'):
            try:
                del(self.meta[public_field])
            except KeyError:
                continue
            else:
                changed = True
        if changed:
            self.meta['published'] = 1
            self.save(exceptions=('folder_type', ))

    def set_remove_date(self):
        '''
        Устанавливает в support_mpfs дату удаления
        для заблокированных файлов
        '''
        from mpfs.core.support import comment
        if self.is_blocked():
            comment.update(self.uid, {'address': self.address.id}, {'del_date': int(time.time())})

    def restore_public(self):
        try:
            self.meta.pop('published')
        except KeyError:
            pass
        else:
            self.meta['public'] = 1
            self.save()

    def make_private(self, fields=()):
        changed = False
        for public_field in ('public', 'public_hash', 'short_url', 'short_url_named',
                             'download_counter', 'office_access_state') + fields:
            try:
                del(self.meta[public_field])
            except KeyError:
                continue
            else:
                changed = True
        if changed:
            self.save(exceptions=('folder_type', ))

    def download_counter_inc(self, count=None):
        if count is None:
            count = 1
        elif count == 0:
            return

        saved = False

        try:
            current_number = self.meta['download_counter']
        except KeyError:
            current_number = 0
        else:
            if settings.feature_toggles['kladun_counter_in_data']:
                saved = self._service.increment_download_counter(self, count)
        if not saved:
            self.meta['download_counter'] = current_number + count
            self.save()

    def is_public_internal(self):
        '''
            Проверка, лежит ли элемент внутри публичного каталога
        '''
        try:
            self.get_symlink_obj()
        except errors.SymlinkNotFound:
            result = False
        else:
            result = True
        return result

    def get_symlink_obj(self):
        try:
            return self._symlink
        except AttributeError:
            self._symlink = Symlink.find(self.address.get_parent())
            return self._symlink

    def has_same_public_parent(self, path):
        try:
            symlink = self.get_symlink_obj()
        except errors.SymlinkNotFound:
            result = False
        else:
            symlink_folder_address = Address(symlink.tgt)
            result = path.startswith(symlink_folder_address.path + '/')
        return result

    def set_tags(self, scope, data):
        #=======================================================================
        # сохранение file_id в data вместо zdata
        self.save()
        #=======================================================================
        self._service.set_tags(self.storage_address.uid, self.meta['file_id'],
                               scope, data)

    def get_tags(self):
        return self._service.tags_for_element(self.storage_address.uid,
                                              self.meta['file_id'])

    def set_yarovaya_mark(self, mark_value=True):
        if self.meta.get('yarovaya_mark') == mark_value:
            return
        self.meta['yarovaya_mark'] = mark_value
        self.save()


class DiskFile(DiskResource, MPFSFile):

    not_saving_meta_fields = DiskResource.not_saving_meta_fields + MPFSFile.not_saving_meta_fields

    def __init__(self, *args, **kwargs):
        try:
            created = self.__created__
        except Exception:
            created = False
        if not created:
            MPFSFile.__init__(self, *args, **kwargs)
            DiskResource.__init__(self, *args, **kwargs)

        self.meta['versioning_status'] ='versionable'

    def __new__(cls, *args, **kwargs):
        owner, address = args[:2]
        owner = str(owner)
        try:
            public_owner = not int(owner)
        except Exception:
            public_owner = False

        if mpfs.engine.process.use_shared_folders() and not kwargs.get('disk_file'):
            from mpfs.core.filesystem.resources.share import SharedFile
            from mpfs.core.filesystem.resources.group import GroupFile
            if not public_owner:
                if owner != address.uid:
                    link = None
                    try:
                        link = kwargs['link']
                    except KeyError:
                        from mpfs.core.social.share import LinkToGroup, Group
                        try:
                            group = Group.find(uid=address.uid, path=address.path)
                        except ShareNotFound:
                            pass
                        else:
                            try:
                                link = LinkToGroup.load(uid=owner, gid=group.gid)
                            except Exception:
                                cls = GroupFile
                                kwargs['group'] = group
                            else:
                                kwargs['link'] = link
                    if link:
                        cls = SharedFile
                else:
                    try:
                        group = kwargs['group']
                    except KeyError:
                        from mpfs.core.social.share import Group
                        try:
                            group = Group.find(uid=owner, path=address.parent_path)
                        except GroupNotFound:
                            group = None
                    if group is not None:
                        cls = GroupFile
        result = super(MPFSFile, cls).__new__(cls)
        result.__init__(*args, **kwargs)
        result.__created__ = True
        return result

    @classmethod
    def from_dict(cls, data, user_principal=None):
        address = Address.Make(data['uid'], data['key'])
        if address.storage_name != 'disk':
            raise errors.ResourceNotFound()

        disk_file = super(DiskFile, cls).from_dict(data, user_principal=user_principal)
        if type(disk_file) is not DiskFile:
            # shared or group is not allowed
            raise errors.ResourceNotFound()
        return disk_file


class DiskFolder(DiskResource, MPFSFolder):
    file_class = DiskFile

    not_saving_meta_fields = DiskResource.not_saving_meta_fields + MPFSFolder.not_saving_meta_fields

    def __init__(self, *args, **kwargs):
        try:
            created = self.__created__
        except AttributeError:
            created = False
        if not created:
            MPFSFolder.__init__(self, *args, **kwargs)
            DiskResource.__init__(self, *args, **kwargs)
        if not (self.is_shared or self.is_group) and mpfs.engine.process.use_shared_folders():
            from mpfs.core.social.share.processor import ShareProcessor
            with_shared = ShareProcessor().check_folder_with_shared(self.uid, self.id)
            if with_shared:
                self.meta['with_shared'] = 1
                self.meta['shared_rights'] = with_shared
        if not self.address.is_storage:
            self.meta['folder_url'] = self.get_folder_url(**kwargs)

    def __new__(cls, *args, **kwargs):
        owner, address = str(args[0]), args[1]
        if mpfs.engine.process.use_shared_folders() and not kwargs.get('disk_folder'):
            try:
                public_owner = not int(owner)
            except Exception:
                public_owner = False
            group = None
            from mpfs.core.filesystem.resources.share import SharedRootFolder, SharedFolder
            from mpfs.core.filesystem.resources.group import GroupRootFolder, GroupFolder
            try:
                link = kwargs['link']
            except KeyError:
                from mpfs.core.social.share import Group, LinkToGroup
                if public_owner:
                    owner = address.uid
                if owner != address.uid:
                    try:
                        group = Group.find(uid=address.uid, path=address.path)
                    except ShareNotFound:
                        pass
                    else:
                        try:
                            link = LinkToGroup.load(uid=owner, gid=group.gid)
                        except ShareNotFound:
                            if group.path == address.path:
                                cls = GroupRootFolder
                            else:
                                cls = GroupFolder
                            kwargs['group'] = group
                        else:
                            kwargs['link'] = link
                            if group.path == address.path:
                                cls = SharedRootFolder
                            else:
                                cls = SharedFolder
                else:
                    try:
                        group = kwargs['group']
                    except KeyError:
                        try:
                            group = Group.find(uid=owner, path=address.path)
                        except GroupNotFound:
                            pass
                    if group:
                        if group.owner == owner:
                            if group.path == address.path:
                                cls = GroupRootFolder
                            else:
                                cls = GroupFolder
                            kwargs['group'] = group
                        else:
                            link = LinkToGroup.load(uid=owner, gid=group.gid)
                            kwargs.pop('group', '')
                            kwargs['link'] = link
                            if link.path == address.path:
                                cls = SharedRootFolder
                            else:
                                cls = SharedFolder
                    else:
                        try:
                            link = Group.find(uid=owner, path=address.path, group=False, link=True)
                        except GroupNotFound:
                            pass
                        else:
                            if link.path == address.path:
                                cls = SharedRootFolder
                            else:
                                cls = SharedFolder
            else:
                if link.path == address.path:
                    cls = SharedRootFolder
                else:
                    cls = SharedFolder
        result = super(MPFSFolder, cls).__new__(cls)
        result.__init__(*args, **kwargs)
        result.__created__ = True
        return result

    def remove_internal_shared_groups_and_links(self):
        self.load()
        from mpfs.core.social.share import ShareProcessor
        if self.with_shared and mpfs.engine.process.use_shared_folders():
            ShareProcessor().remove_folder_references(self.uid, self.address.path)

    def rm(self, *args, **kwargs):
        remove_links = kwargs.pop('remove_links', False)
        self.load()
        if not kwargs.pop('keep_group', False):
            from mpfs.core.social.share import ShareProcessor
            if self.is_group_root and remove_links:
                ShareProcessor().delete_group_root(self.group, notify_owner=False)
            elif self.with_shared and mpfs.engine.process.use_shared_folders():
                self.remove_internal_shared_groups_and_links()
        result = MPFSFolder.rm(self, *args, **kwargs)
        return result

    def set_unshared(self):
        if FEATURE_TOGGLES_SETTING_YAROVAYA_MARK_ENABLED:
            self.set_yarovaya_mark()
        self._service.folder_unshared(self)

    def get_url(self, **args):
        inline = args.get('inline')
        if inline:
            self.setup(inline=inline)
        return {
            'folder':   self.meta.get('folder_url'),
        }

    def flat_index(self, *args, **kwargs):
        res, ver = self._service.flat_index(self.address.uid, self.address.path, *args, **kwargs)
        result = {}
        for key, data in res.iteritems():
            address = Address('%s:%s' % (self.address.uid, key))
            version = data['version']
            if data['type'] == 'dir':
                resource = DiskFolder(self.committer, address, data=data, version=version)
                if resource.is_shared_root:
                    result.update(resource.flat_index())
            elif data['type'] == 'file':
                resource = DiskFile(self.committer, address, data=data, version=version)
            result[resource.id] = resource.dict()
        result[self.id] = self.dict()
        return result

    def check_rw(self, operation=None):
        if operation in ('set_public', 'set_private') and self.meta.get('with_shared'):
            from mpfs.core.social.share.processor import ShareProcessor
            lc = []
            ShareProcessor().check_folder_with_shared(self.uid, self.id, link_container=lc)
            for link in lc:
                if not link.is_rw():
                    raise GroupNoPermit()

    def get_children_tags_list(self):
        return self._service.tags_in_folder_list(self.storage_address.uid,
                                            self.storage_address.path)

    def get_children_tags_tree(self):
        return self._service.tags_in_folder_tree(self.storage_address.uid,
                                            self.storage_address.path)

    def get_children_list_by_tags(self, data):
        return self._service.elements_in_folder_list(self.storage_address.uid,
                                            self.storage_address.path, data)

    def get_children_tree_by_tags(self, data):
        return self._service.elements_in_folder_tree(self.storage_address.uid,
                                            self.storage_address.path, data)

    def tags_photo_list(self, filters, system_tags, user_tags):
        return self._service.tags_photo_list(self, filters, system_tags, user_tags)

    def set_last_files(self):
        """Установить последние дочерние файлы относительно текущего пути,
        включая общие подпапки.
        """
        self._service.set_last_files(self)
        for subfile in self.children_items['files']:
            subfile.name_tree = filter(None, subfile.key.split(os.path.sep))[::-1]
            if self.request:
                subfile.print_to_listing_log()
            append_meta_to_office_files(subfile, self.request)

    @classmethod
    def from_dict(cls, data, user_principal=None):
        address = Address.Make(data['uid'], data['key'])
        if address.storage_name != 'disk':
            raise errors.ResourceNotFound()

        if address.is_root:
            raise errors.ResourceNotFound()

        disk_folder = super(DiskFolder, cls).from_dict(data, user_principal=user_principal)
        if type(disk_folder) is not DiskFolder:
            # shared or group is not allowed
            raise errors.ResourceNotFound()
        return disk_folder


def append_meta_to_office_files(file_resource, request):
    if not isinstance(file_resource, (MPFSFile, DiskFile)):
        return

    if request is None:
        return

    uid = getattr(request, 'uid', None)
    if uid is None:
        uid = file_resource.uid
    tld = request.tld

    client_id = OfficeClientIDConst.DISK
    if request.method_name in ('public_info', 'public_list'):
        client_id = OfficeClientIDConst.PUBLIC

    editor = get_editor(request.user or uid)
    if editor is None:
        file_resource.meta.pop('office_access_state', None)
        return

    remove_sharing_url_field = False
    from mpfs.core.filesystem.resources.share import SharedResource
    owner_editor = None
    if isinstance(file_resource, SharedResource):
        owner_editor = get_editor(file_resource.owner_uid)
        if (owner_editor is None or
                owner_editor.type_label != EditorConst.ONLY_OFFICE):
            remove_sharing_url_field = True
    if editor.is_edit_possible(file_resource):
        document_id = None
        if client_id == OfficeClientIDConst.DISK:
            document_id = file_resource.id[1:]   # split leading / from path
        elif client_id == OfficeClientIDConst.PUBLIC:
            public_hash = file_resource.get_public_hash()
            if public_hash is not None:
                document_id = public_hash
            else:
                public_hash = request.private_hash
                document_id = '%s:/%s' % (public_hash, file_resource.name)

        if document_id is not None:
            file_resource.meta['office_online_url'] = build_office_online_url(client_id, document_id, tld)
            file_resource.meta['office_online_editor_type'] = editor.type_label

        if file_resource.office_doc_short_id:
            sharing_url_addr = SharingURLAddressHelper.build_sharing_url_addr(
                file_resource.owner_uid,
                file_resource.office_doc_short_id
            )
            file_resource.meta['office_online_sharing_url'] = build_office_online_url(
                client_id=OfficeClientIDConst.SHARING_URL,
                document_id=sharing_url_addr.serialize(),
                tld=tld
            )

        filename, ext = os.path.splitext(file_resource.name)
        ext = ext[1:].lower()
        SHARING_URL_SUPPORTED_EXT = {'docx', 'xlsx', 'pptx'}
        if not file_resource.is_fully_public() or ext not in SHARING_URL_SUPPORTED_EXT:
            remove_sharing_url_field = True
    else:
        remove_sharing_url_field = True

    if remove_sharing_url_field:
        file_resource.meta.pop('office_access_state', None)
        file_resource.meta.pop('office_online_sharing_url', None)
    else:
        if not owner_editor:
            owner_editor = get_editor(file_resource.owner_uid)
        if not owner_editor or owner_editor.type_label != EditorConst.ONLY_OFFICE or file_resource.office_access_state != OfficeAccessStateConst.ALL:
            file_resource.meta.pop('office_online_sharing_url', None)
