# -*- coding: utf-8 -*-
"""

MPFS
CORE

Папка Fotki

"""
import mpfs.engine.process

from mpfs.common import errors
from mpfs.core.filesystem.resources import base
from mpfs.core.services.fotki_service import Fotki
from mpfs.core.address import Address


log = mpfs.engine.process.get_default_log()
error_log = mpfs.engine.process.get_error_log()


class FotkiFile(base.File):

    service_class = Fotki

    static_fields = ('ownerId', 'authId', 'imageId')

    def __init__(self, uid, address, **args):
        self.media_type = 'image'
        base.File.__init__(self, uid, address, **args)

    def fill(self, address, **args):
        return self._service.load_file(self)

    def setprop(self, changes, deleted, **kwargs):
        super(FotkiFile, self).setprop(changes, deleted, **kwargs)
        result = self._service.edit_file(self.address, **changes)
        return result

    def get_url(self, **args):
        if args.get('orig'):
            url, cookie = self._service.get_original_photo_url(self)
            return {'file' : url, 'cookie' : cookie}
        else:
            return { 'file': self.meta['file_url'] }

    def get_file_url(self, **args):
        _url = ''
        _sizes = self.meta.get('sizes', [])
        if _sizes:
            original_url = filter(lambda x: x.get('name') == 'orig', _sizes)
            if original_url:
                _url = original_url[0].get('url')
            else:
                _url = _sizes[-1].get('url')
        return _url

    def get_direct_url(self):
        return {
            'url' : self.get_file_url(),
            'mimetype': self._service.default_content_type,
            'name': self.name
        }

    @classmethod
    def from_dict(cls, data, user_principal=None):
        address = Address.Make(data['uid'], data['key'])
        if address.storage_name != 'fotki':
            raise errors.ResourceNotFound()
        return super(FotkiFile, cls).from_dict(data, user_principal=user_principal)


class FotkiFolder(base.Folder):

    file_class = FotkiFile
    service_class = Fotki

    static_fields = ('userId', 'albumId')

    @classmethod
    def Create(classname, address, **data):

        parent = data.get('parent', classname.Construct(address.uid, address.get_parent()))

        result = classname.service_class().make_folder(address, parent)
        address.id = result['id']
        folder = classname(address, data=result)

        return folder

    def load(self):
        return self._service.listing(self)

    def setprop(self, changes, deleted, **kwargs):
        super(FotkiFolder, self).setprop(changes, deleted, **kwargs)
        result = self._service.edit_folder(self.address, **changes)
        return result

    @classmethod
    def from_dict(cls, data, user_principal=None):
        address = Address.Make(data['uid'], data['key'])
        if address.storage_name != 'fotki':
            raise errors.ResourceNotFound()
        return super(FotkiFolder, cls).from_dict(data, user_principal=user_principal)
