# -*- coding: utf-8 -*-
'''
    Общие ресурсы для владельца группы
'''
import mpfs.engine.process
from mpfs.common import errors
from mpfs.config import settings
from mpfs.core.address import Address
from mpfs.core.filesystem.resources.disk import DiskFile, DiskFolder
from mpfs.core.filesystem.helpers.counter import Counter
from mpfs.core.services.passport_service import Passport
from mpfs.core.last_files.logic import SharedLastFilesProcessor
from mpfs.core.social.dao.groups import GroupsDAO

FEATURE_TOGGLES_SETTING_YAROVAYA_MARK_ENABLED = settings.feature_toggles['setting_yarovaya_mark_enabled']

passport_instance = Passport()

log = mpfs.engine.process.get_default_log()


class GroupResource(object):

    def __init__(self, *args, **kwargs):
        try:
            group = kwargs['group']
        except KeyError:
            from mpfs.core.social.share import Group
            try:
                gid = kwargs['gid']
            except KeyError:
                try:
                    parent = kwargs['parent']
                except KeyError:
                    self.group = Group.find(uid=self.address.uid, path=self.address.path)
                else:
                    self.group = parent.group
            else:
                self.group = Group.load(gid)
        else:
            self.group = group
        self.meta['group'] = {
                              'is_shared' : 1,
                              'is_owned' : 1,
                              'is_root' : 0,
                              'gid' : self.group.gid,
                              'owner' : passport_instance.userinfo_summary(self.group.owner),
                              }

    def update_used(self, val=None):
        val = val or int(self.size)
        Counter().add_group(self.group.gid, val)
        Counter().add('disk', self.uid, val)

    @property
    def is_group_root(self):
        return False

    @property
    def is_group(self, *a, **kw):
        return True

    @property
    def is_group_internal(self, *a, **kw):
        return True


class GroupFile(GroupResource, DiskFile):
    not_saving_meta_fields = tuple(x for x in DiskFile.not_saving_meta_fields if x != 'modify_uid')

    def __init__(self, *args, **kwargs):
        try:
            created = self.__created__
        except Exception:
            created = False
        if created:
            return
        DiskFile.__init__(self, *args, **kwargs)
        GroupResource.__init__(self, *args, **kwargs)

    def rm(self, *args, **kwargs):
        result = super(GroupFile, self).rm(*args, **kwargs)
        SharedLastFilesProcessor().update_for_group_async(self.group.gid)
        return result

    @classmethod
    def from_dict(cls, data, user_principal=None):
        if 'group' not in data:
            raise errors.ResourceNotFound()

        group = data['group']
        if group.owner != data['uid']:
            raise errors.ResourceNotFound()

        address = Address.Make(group.owner, data['key'])
        if address.storage_name != 'disk':
            raise errors.ResourceNotFound()

        user_principal = user_principal or group.owner
        group_file = GroupFile(
            user_principal, address, version=data['wh_version'],
            group=group, data=data['data'])
        if not isinstance(group_file, GroupFile):
            raise errors.ResourceNotFound()

        return group_file


class GroupFolder(GroupResource, DiskFolder):
    not_saving_meta_fields = tuple(x for x in DiskFile.not_saving_meta_fields if x != 'modify_uid')
    file_class = GroupFile

    def __init__(self, *args, **kwargs):
        try:
            created = self.__created__
        except Exception:
            created = False
        if created:
            return
        DiskFolder.__init__(self, *args, **kwargs)
        GroupResource.__init__(self, *args, **kwargs)

    @classmethod
    def from_dict(cls, data, user_principal=None):
        if 'group' not in data:
            raise errors.ResourceNotFound()

        group = data['group']
        if group.owner != data['uid']:
            raise errors.ResourceNotFound()

        address = Address.Make(group.owner, data['key'])
        if address.storage_name != 'disk':
            raise errors.ResourceNotFound()

        user_principal = user_principal or group.owner
        group_folder = GroupFolder(
            user_principal, address, version=data['wh_version'],
            group=group, data=data['data'])
        if not isinstance(group_folder, GroupFolder):
            raise errors.ResourceNotFound()

        return group_folder


class GroupRootFolder(DiskFolder, GroupResource):

    file_class = GroupFile

    def __init__(self, *args, **kwargs):
        try:
            created = self.__created__
        except Exception:
            created = False
        if created:
            return
        DiskFolder.__init__(self, *args, **kwargs)
        GroupResource.__init__(self, *args, **kwargs)
        self.meta['group']['is_root'] = 1
        self.meta['group']['user_count'] = self.group.user_count() + self.group.invites_count()
        self.meta['group']['size'] = self.group.size

    @property
    def is_group_root(self):
        return True

    @property
    def is_group(self, *a, **kw):
        return True

    def rename(self, target_address):
        GroupsDAO().check_readonly()
        result = self._service.rename(self.address, target_address)
        self._service.update_symlinks(self.address, target_address)
        self.group.rename(target_address.path)
        return result

    def create_child_folder(self, uid, address, **folderdata):
        return GroupFolder.Create(uid, address, self, **folderdata)

    def set_shared(self):
        if FEATURE_TOGGLES_SETTING_YAROVAYA_MARK_ENABLED:
            self.set_yarovaya_mark()
        self.version = self._service.owners_folder_created(self)

    @classmethod
    def from_dict(cls, data, user_principal=None):
        if 'group' not in data:
            raise errors.ResourceNotFound()

        group = data['group']
        if group.owner != data['uid']:
            raise errors.ResourceNotFound()

        address = Address.Make(group.owner, data['key'])
        if address.storage_name != 'disk':
            raise errors.ResourceNotFound()

        user_principal = user_principal or group.owner
        group_root_folder = GroupRootFolder(
            user_principal, address, version=data['wh_version'],
            group=data['group'], data=data['data'])
        if not isinstance(group_root_folder, GroupRootFolder):
            raise errors.ResourceNotFound()

        return group_root_folder
