# -*- coding: utf-8 -*-

import mpfs.engine.process

from mpfs.core.address import Address
from mpfs.common import errors
from mpfs.core.metastorage.control import notes
from mpfs.core.filesystem.constants import NOTES_STORAGE_AREA, NOTES_STORAGE_PATH
from mpfs.core.filesystem.resources.disk import MPFSFile, MPFSFolder
from mpfs.core.services.notes_storage_service import NotesStorageService

default_log = mpfs.engine.process.get_default_log()


def create_notes_section_for_user(uid):
    notes.create(uid)
    notes.make_folder(uid, '/notes', {})

    from mpfs.core import factory
    from mpfs.core.filesystem.indexer import DiskDataIndexer

    folder = factory.get_resource(uid, '/notes')
    disk_data_indexer = DiskDataIndexer()
    disk_data_indexer.push(folder, 'modify', operation='mkdir')
    disk_data_indexer.flush_index_data()

    default_log.debug('created %s domain for %s' % (notes.name, uid))


class NotesSectionAutoCreationMixin(object):

    @classmethod
    def Create(cls, uid, address, parent, **kw):
        """Создать файл/папку в разделе /notes.

        Если файл/папку создать не удалось по причине того, что отсутствует раздел /notes, то он будет создан.
        """
        try:
            return super(NotesSectionAutoCreationMixin, cls).Create(uid, address, parent, **kw)
        except errors.StorageKeyNotFound as e:
            if isinstance(e.data, dict) and e.data.get('key') == NOTES_STORAGE_PATH:
                create_notes_section_for_user(uid)
                return super(NotesSectionAutoCreationMixin, cls).Create(uid, address, parent, **kw)


class NotesResourceSubTypeMixin(object):
    def remove_public(self, *args, **kwargs):
        return

    def is_fully_public(self, *args, **kwargs):
        return False


class NotesFile(NotesResourceSubTypeMixin, NotesSectionAutoCreationMixin, MPFSFile):
    service_class = NotesStorageService


class NotesFolder(NotesResourceSubTypeMixin, NotesSectionAutoCreationMixin, MPFSFolder):
    service_class = NotesStorageService
    file_class = NotesFile

    @classmethod
    def from_dict(cls, data, user_principal=None):
        address = Address.Make(data['uid'], data['key'])
        if address.storage_name != NOTES_STORAGE_AREA:
            raise errors.ResourceNotFound()
        return super(NotesFolder, cls).from_dict(data, user_principal=user_principal)
