# -*- coding: utf-8 -*-
import base64

import mpfs.engine.process
from mpfs.common.util.user_agent_parser import UserAgentParser
from mpfs.common.util.ycrid_parser import YcridParser
from mpfs.common.util.filetypes import getGroupByName, getGroupOnlyByName, MediaType
from mpfs.config import settings
from mpfs.core.address import Address
from mpfs.common import errors
from mpfs.core.filesystem.resources.disk import DiskResource, MPFSFile, MPFSFolder
from mpfs.core.services.photounlim_service import Photounlim
from mpfs.core.user.constants import PHOTOUNLIM_AREA

SERVICES_PHOTOUNLIM_PERMITTED_MEDIA_TYPES = settings.photounlim['permitted_media_types']
PHOTOUNLIM_ALLOW_YANDEX_SEARCH_MOBILE = settings.photounlim['allow_yandex_search_mobile']
PHOTOUNLIM_ALLOW_YANDEX_SEARCH_MOBILE_UIDS = settings.photounlim['allow_yandex_search_mobile_uids']
FEATURE_TOGGLES_PHOTOUNLIM_FILTER_MEDIA_TYPES_ENABLED = \
    settings.photounlim['photounlim_filter_media_types_enabled']
FEATURE_TOGGLES_PHOTOUNLIM_FILTER_MISSING_HASHES_OR_SIZE = \
    settings.photounlim['photounlim_filter_missing_hashes_or_size']


class PhotounlimFile(DiskResource, MPFSFile):
    not_saving_meta_fields = DiskResource.not_saving_meta_fields + MPFSFile.not_saving_meta_fields + ('storage_type',)
    service_class = Photounlim

    def __init__(self, *args, **kwargs):
        MPFSFile.__init__(self, *args, **kwargs)
        DiskResource.__init__(self, *args, **kwargs)
        self.meta['storage_type'] = PHOTOUNLIM_AREA


class PhotounlimFolder(MPFSFolder):
    file_class = PhotounlimFile
    service_class = Photounlim

    @classmethod
    def from_dict(cls, data, user_principal=None):
        address = Address.Make(data['uid'], data['key'])
        if address.storage_name != PHOTOUNLIM_AREA:
            raise errors.ResourceNotFound()
        return super(PhotounlimFolder, cls).from_dict(data, user_principal=user_principal)


class PhotounlimLastModifiedIterationKey(object):
    DELIMITER = '/'

    def __init__(self, mtime_gte, offset):
        self._mtime_gte = mtime_gte
        self._offset = offset

    def serialize(self):
        return base64.b64encode(self.DELIMITER.join((self._mtime_gte,
                                                     str(self._offset))))

    @classmethod
    def generate_next(cls, items, prev_mtime, prev_offset, prev_iteration_key):
        """Генерирует новый ключ итерирования.

        :param items: Элементы текущей страницы последний модифированных элементов раздела photounlim
        :param prev_mtime: Значение mtime (нижняя граница), по которому получали текущую страницу элементов
        :param prev_offset: Количество элементов, которые пропускались при выборке по prev_mtime
        :param prev_iteration_key: Предыдущий ключ итерирования (его возвращаем если нет элементов в новой выборке)
        """
        if not items:
            if prev_iteration_key:
                return cls.parse(prev_iteration_key)
            else:
                return cls(prev_mtime, 0)

        new_mtime = str(items[-1]['mtime'])
        if prev_mtime == new_mtime:
            new_offset = prev_offset + len(items)
        else:
            # Для следующей страницы выборки нужно будет пропустить как минимум
            # один элемент - последний в текущей выборке.
            new_offset = 1
            # Сравнивать начинаем с предпоследнего элемента
            cur_index = len(items) - 2
            # пока есть элементы для проверок и mtime'ы одинаковые
            while cur_index >= 0 and items[cur_index]['mtime'] == items[-1]['mtime']:
                # Повышаем offset, чтобы потом пропустить эти элементы (при выборке по их mtime'у
                new_offset += 1
                cur_index -= 1

        return cls(new_mtime, new_offset)

    @classmethod
    def parse(cls, iteration_key):
        """Разбирает ключ итерирования на составляющие.

        :returns: :class:`mpfs.core.filesystem.resources.photounlim.PhotounlimLastModifiedIterationKey`
        """
        decoded_iteration_key = base64.b64decode(iteration_key)
        mtime_gte, offset = decoded_iteration_key.split('/', 1)
        offset = int(offset)
        return cls(mtime_gte, offset)

    def get_mtime_gte(self):
        return self._mtime_gte

    def get_offset(self):
        return self._offset


def is_converting_address_to_photounlim_for_uid_needed(address, uid, user_agent):
    ycrid = mpfs.engine.process.get_cloud_req_id()
    check_yandex_search_mobile = PHOTOUNLIM_ALLOW_YANDEX_SEARCH_MOBILE or uid in PHOTOUNLIM_ALLOW_YANDEX_SEARCH_MOBILE_UIDS
    if address.storage_name == 'photostream' and (YcridParser.is_yandex_disk_mobile(ycrid) or check_yandex_search_mobile and UserAgentParser.is_yandex_search_mobile(user_agent)):
        from mpfs.core.user.base import User
        user = User(uid)
        if user.is_unlimited_autouploading_enabled():
            return True
    return False


def is_converting_address_to_newunlim_for_uid_needed(address, uid, user_agent):
    ycrid = mpfs.engine.process.get_cloud_req_id()
    check_yandex_search_mobile = PHOTOUNLIM_ALLOW_YANDEX_SEARCH_MOBILE or uid in PHOTOUNLIM_ALLOW_YANDEX_SEARCH_MOBILE_UIDS
    if address.storage_name == 'photostream' and (YcridParser.is_yandex_disk_mobile(ycrid) or check_yandex_search_mobile and UserAgentParser.is_yandex_search_mobile(user_agent)):
        from mpfs.core.user.base import User
        user = User(uid)
        media_type = getGroupOnlyByName(address.name)
        if media_type == MediaType.VIDEO:
            unlimited_video_autouploading_enabled = user.get_unlimited_autouploading()['unlimited_video_autouploading_enabled']
            if unlimited_video_autouploading_enabled is None:
                unlimited_video_autouploading_enabled = User(uid).is_unlimited_autouploading_enabled()
            if unlimited_video_autouploading_enabled:
                return True
        if media_type == MediaType.IMAGE:
            unlimited_photo_autouploading_enabled = user.get_unlimited_autouploading()['unlimited_photo_autouploading_enabled']
            if unlimited_photo_autouploading_enabled is None:
                unlimited_photo_autouploading_enabled = User(uid).is_unlimited_autouploading_enabled()
            if unlimited_photo_autouploading_enabled:
                return True
    return False


def is_photounlim_fraudulent_store(address, md5=None, size=None):
    if (FEATURE_TOGGLES_PHOTOUNLIM_FILTER_MEDIA_TYPES_ENABLED
            and getGroupByName(address.name) not in SERVICES_PHOTOUNLIM_PERMITTED_MEDIA_TYPES):
        return True

    if (FEATURE_TOGGLES_PHOTOUNLIM_FILTER_MISSING_HASHES_OR_SIZE
            and not (md5 and size)):
        return True

    return False


def is_unlim_forbidden_due_to_experiment(uid, address, photostream_destination=None):
    """
    :param uid:
    :param address:
    :type address: mpfs.core.address.Address
    :type photostream_destination: str
    :return:
    """
    from mpfs.core.user.base import User
    from mpfs.core.user.dao.user import UserDAO
    user = User(uid)
    unlimited_autouploading = UserDAO().get_unlimited_autouploading(uid)
    video_status = unlimited_autouploading['unlimited_video_autouploading_enabled']
    if video_status is None:
        video_status = user.is_unlimited_autouploading_enabled()

    # если пользователь льёт видео и льёт в безлимит
    if getGroupOnlyByName(address.name) == MediaType.VIDEO and photostream_destination and \
            (photostream_destination == PHOTOUNLIM_AREA):

        # но у нас стоит видеобезлимит выключенным
        if not video_status:
            # то запрещаем ему лить в безлимит и возвращаем причину
            return True, unlimited_autouploading['unlimited_video_autouploading_reason']

        # если он попал в эксперимент, и не платник но стоит включенный видеобезлимит
        if video_status and not user.is_unlimited_video_autouploading_allowed():
            reason = 'by_experiment'
            # отключаем ему безлимит с причиной by_experiment
            # так как у нас нет другого места где мы узнаём что пользователь попал в эксперимент и
            # ему надо выключить видеобезлимит, то делаем это при попытке залить в безлимит
            user.set_unlimited_autouploading(video_status=False, video_reason=reason)
            return True, reason
    return False, None
