# -*- coding: utf-8 -*-
'''
Описание общих с другими
ресурсов пользователей
'''

import mpfs.engine.process
from mpfs.config import settings

from mpfs.core.filesystem.resources.disk import DiskFile, DiskFolder
from mpfs.core.social.share import LinkToGroup
from mpfs.core.address import Address, GroupAddress
from mpfs.common.errors import FolderAlreadyExist, ResourceNotFound
from mpfs.common.errors.share import GroupNoPermit
from mpfs.core.filesystem.helpers.counter import Counter
from mpfs.core.services.passport_service import Passport
from mpfs.core.last_files.logic import SharedLastFilesProcessor


FEATURE_TOGGLES_SETTING_YAROVAYA_MARK_ENABLED = settings.feature_toggles['setting_yarovaya_mark_enabled']

log = mpfs.engine.process.get_default_log()
passport_instance = Passport()


class SharedResource(object):

    def __init__(self, uid, address, **args):
        self.uid = uid
        self.key = self.id = self.link.get_link_path(address.path)
        self.visible_address = Address('%s:%s' % (uid, self.key))
        self.meta['group'] = {
            'is_shared': 1,
            'is_owned': 0,
            'is_root': 0,
            'gid': self.group.gid,
            'rights': self.link.rights,
            'owner': passport_instance.userinfo_summary(self.link.group.owner),
        }

    def rename(self, target_address):
        self._service.rename(self.address, target_address)

    def check_rw(self, operation=None):
        if not self.link.is_rw():
            raise GroupNoPermit()

    @property
    def is_shared(self):
        return True

    @property
    def is_shared_internal(self, *a, **kw):
        return True

    @property
    def owner_uid(self):
        return self.link.group.owner

    def update_used(self, val=None):
        val = val or int(self.size)
        Counter().add_group(self.group.gid, val)
        Counter().add('disk', self.group.owner, val)


class SharedFile(SharedResource, DiskFile):
    not_saving_meta_fields = tuple(x for x in DiskFile.not_saving_meta_fields if x != 'modify_uid')

    def __init__(self, uid, address, **args):
        try:
            created = self.__created__
        except Exception:
            created = False
        if created:
            return
        try:
            self.link = args['link']
        except KeyError:
            self.link = args['parent'].link
        self.group = args['group'] = self.link.group
        DiskFile.__init__(self, uid, address, **args)
        SharedResource.__init__(self, uid, address, **args)

    @classmethod
    def Create(classname, uid, address, parent, **kw):
        group_address = GroupAddress(address.id, parent.address.id + '/' + address.name)
        data = kw.get('data', {})
        if isinstance(data, dict):
            data['source_uid'] = uid
        else:
            data = {'source_uid': uid}
        if 'meta' not in data:
            data['meta'] = {}
        data['meta']['source_uid'] = uid
        kw['data'] = data
        return super(SharedFile, classname).Create(uid, group_address, parent, **kw)

    def rm(self, *args, **kwargs):
        result = super(DiskFile, self).rm(*args, **kwargs)
        SharedLastFilesProcessor().update_for_group_async(self.group.gid)
        return result

    @classmethod
    def from_dict(cls, data, user_principal=None):
        if 'link' not in data:
            raise ResourceNotFound()

        link = data['link']
        if link.uid == data['uid']:
            raise ResourceNotFound()

        address = Address.Make(link.uid, link.get_link_path(data['key']))
        group_address = GroupAddress(
            address.id, '%s:%s' % (link.group.owner, data['key']))
        user_principal = user_principal or link.uid
        shared_file = SharedFile(
            user_principal, group_address, version=data['wh_version'],
            link=link, data=data['data'])
        if not isinstance(shared_file, SharedFile):
            raise ResourceNotFound()

        return shared_file


class SharedFolder(SharedResource, DiskFolder):
    not_saving_meta_fields = tuple(x for x in DiskFile.not_saving_meta_fields if x != 'modify_uid')
    file_class = SharedFile

    def __init__(self, uid, address, **args):
        try:
            created = self.__created__
        except Exception:
            created = False
        if created:
            return
        try:
            self.link = args['link']
        except KeyError:
            self.link = args['parent'].link
        self.group = args['group'] = self.link.group
        DiskFolder.__init__(self, uid, address, **args)
        SharedResource.__init__(self, uid, address, **args)
        self.meta['folder_url'] = self.get_folder_url(**args)

    @classmethod
    def Create(classname, uid, address, parent, **folderdata):
        # проверяем что такого еще нет
        data = folderdata.get('data', {})

        if not data.get('meta', {}).get('file_id'):
            if 'meta' not in data:
                data['meta'] = {}
            data['meta']['file_id'] = classname.generate_file_id(uid, address.path)
            folderdata['data'] = data

        for k in classname.not_saving_fields:
            data.pop(k, None)

        for k in classname.not_saving_meta_fields:
            data['meta'].pop(k, None)

        try:
            classname.Construct(uid, address)
            raise FolderAlreadyExist(address.id)
        except ResourceNotFound:
            data = folderdata.get('data', {})
            address = GroupAddress(address.id, parent.address.id + '/' + address.name)
            result = classname.service_class().make_folder(address, parent, data)
            folderdata['version'] = result.version
            folderdata['data']['version'] = result.version  # здесь это та самая версия, которая вставилась в базу
            folder = classname(uid, address, link=parent.link, **folderdata)
            folder.set_parent(parent)
            return folder

    def rm(self, *args, **kwargs):
        return super(DiskFolder, self).rm(*args, **kwargs)

    def tree_index(self):
        return self._service.tree_index(self.owner_uid, self.path)

    @classmethod
    def from_dict(cls, data, user_principal=None):
        if 'link' not in data:
            raise ResourceNotFound()

        link = data['link']
        if link.uid == data['uid']:
            raise ResourceNotFound()

        address = Address.Make(link.uid, link.get_link_path(data['key']))
        group_address = GroupAddress(
            address.id, '%s:%s' % (link.group.owner, data['key']))
        user_principal = user_principal or link.uid
        shared_folder = SharedFolder(
            user_principal, group_address, version=data['wh_version'],
            link=link, data=data['data'])
        if not isinstance(shared_folder, SharedFolder):
            raise ResourceNotFound()

        return shared_folder


class SharedRootFolder(DiskFolder, SharedResource):

    file_class = SharedFile

    def __init__(self, uid, address, **args):
        try:
            created = self.__created__
        except Exception:
            created = False
        if created:
            return
        try:
            self._link = args['link']
        except KeyError:
            try:
                original_resource = args['original_resource']
            except KeyError:
                try:
                    gid = args['data']['meta']['group']['gid']
                except KeyError:
                    self._link = LinkToGroup.load(path=address.path, uid=address.uid)
                else:
                    self._link = LinkToGroup.load(gid=gid, uid=address.uid)
                    if self._link.path != address.path:
                        self._link.path = address.path
                        self._link.save()
            else:
                self._link = original_resource._link

        real_addr = Address('%s:%s' % (self.link.group.owner, self.link.group.path))
        self.group = args['group'] = self.link.group
        DiskFolder.__init__(self, uid, address, **args)
        self.address = real_addr
        self.meta['group'] = {
                              'is_shared': 1,
                              'is_owned': 0,
                              'is_root': 1,
                              'rights': self.link.rights,
                              'gid': self.group.gid,
                              'owner': passport_instance.userinfo_summary(self.link.group.owner),
                              'user_count': self.link.group.user_count(),
                              'size': self.link.group.size,
                              }
        self.id = address.path
        self.uid = address.uid
        self.visible_address = self.storage_address = address
        self.name = address.name

    @classmethod
    def Create(classname, uid, address, parent, **kwargs):
        if FEATURE_TOGGLES_SETTING_YAROVAYA_MARK_ENABLED:
            kwargs.setdefault('data', {'meta': {}}).setdefault('meta', {})['yarovaya_mark'] = True
        return super(SharedRootFolder, classname).Create(uid, address, parent, **kwargs)

    @property
    def link(self):
        return self._link or LinkToGroup.load(path=self.address.path, uid=self.committer)

    @property
    def comment_ids(self):
        return {
            'public_resource': str(self.resource_id),
            'private_resource': str(self.group.get_folder().resource_id),
        }

    def rm(self, *args, **kwargs):
        '''
        Remove this folder and link
        '''
        version = 0
        if kwargs.get('keep_group'):
            result = self._service.remove(self.visible_address, data=self.dict(safe=True))
            version = self.deleted_version = int(result.version)
        else:
            from mpfs.core.social.share import ShareProcessor
            ShareProcessor().leave_group(self.uid, self.group.gid,
                                         group=self.group, folder=self)
        return version

    def create_child_folder(self, uid, address, **folderdata):
        return SharedFolder.Create(uid, address, self, **folderdata)

    def rename(self, target_address):
        self.link.rename(target_address.path)
        self._service.update_symlinks(self.storage_address, target_address)
        result = self._service.rename(self.visible_address, target_address)
        return result

    def check_rw(self, operation=None):
        if operation not in ('move', 'rm'):
            return SharedResource.check_rw(self, operation=operation)

    @property
    def is_shared(self):
        return True

    @property
    def is_shared_root(self, *a, **kw):
        return True

    @property
    def owner_uid(self):
        return self.link.group.owner

    @property
    def owner_file_id(self):
        if hasattr(self, '_owner_file_id'):
            return self._owner_file_id

        if self.uid != self.owner_uid:
            file_id = self.link.group.get_folder().meta['file_id']
        else:
            file_id = self.meta['file_id']

        self._owner_file_id = file_id
        return file_id

    def tree_index(self):
        return self._service.tree_index(self.address.uid, self.address.path)

    def make_private_copy(self):
        self_index = self.tree_index()

        def make_element(element):
            original_path_split = filter(None, element['this']['key'].split('/'))
            group_folder_path_split = filter(None, self.link.group.path.split('/'))
            path = self.link.path + '/' + '/'.join(original_path_split[len(group_folder_path_split):])
            address = Address('%s:%s' % (self.uid, path))
            if element['this']['type'] == 'dir':
                DiskFolder.Create(self.uid, address, self, data=element['this']['data'], force=True, disk_folder=True)
                for child in element['list']:
                    make_element(child)
            else:
                for k in DiskFile.public_fields:
                    element['this']['data']['meta'].pop(k, '')
                new_file = DiskFile.Create(self.uid, address,
                                           parent=self, data=element['this']['data'], disk_file=True)
                new_file.update_used()

        for each in self_index:
            make_element(each)
        self.meta.pop('shared', '')
        self.meta.pop('editor', '')
        self.meta.pop('group', '')
        self.address = self.visible_address
        self.save()
        self.deleted_version = self.version

    def lock(self):
        return self._service.lock_set(self.address.uid, self.address.path)

    def unlock(self):
        return self._service.lock_unset(self.address.uid, self.address.path)

    def check_lock(self):
        return self._service.lock_check(self.address.uid, self.address.path)

    '''

    def lock(self):
        return self._service.lock_set(self.visible_address.uid, self.visible_address.path, self.version)

    def unlock(self):
        return self._service.lock_unset(self.visible_address.uid, self.visible_address.path, self.version)

    def check_lock(self):
        return self._service.lock_check(self.visible_address.uid, self.visible_address.path, self.version)
    '''

    def set_shared(self):
        if FEATURE_TOGGLES_SETTING_YAROVAYA_MARK_ENABLED:
            self.set_yarovaya_mark()
        self.version = self._service.users_folder_created(self)

    def update_rights(self):
        self._service.users_folder_updated(self)

    @classmethod
    def from_dict(cls, data, user_principal=None):
        if 'link' not in data:
            raise ResourceNotFound()

        link = data['link']
        # uids должны совпадать, эта папка создается у гостя.
        if link.uid != data['uid']:
            raise ResourceNotFound()

        address = Address.Make(link.uid, link.path)
        user_principal = user_principal or link.uid
        shared_root_folder = SharedRootFolder(
            user_principal, address, version=data['wh_version'],
            link=link, data=data['data'])
        if not isinstance(shared_root_folder, SharedRootFolder):
            raise ResourceNotFound()

        return shared_root_folder
