# -*- coding: utf-8 -*-
"""

MPFS
CORE

Папка share (общеупотребительная шарилка)

"""
import mpfs.engine.process

from mpfs.core.address import Address
from mpfs.core.filesystem.resources.disk import MPFSFolder, MPFSFile, DiskResource
from mpfs.core.services.stock_service import Stock
from mpfs.common import errors
from mpfs.common.util.urls import update_qs_params

log = mpfs.engine.process.get_default_log()
error_log = mpfs.engine.process.get_error_log()

SHARE_UID = mpfs.engine.process.share_user()


class StockResource(DiskResource):
    pass


class StockFile(MPFSFile, StockResource):

    service_class = Stock

    def __init__(self, uid, address, *args, **kwargs):
        address.change_uid(SHARE_UID)

        # нужно для генерации специальной ссылки на заберун для скачивания особых инсталляторов, не требующих логина
        # (с прошитыми токенами) - https://st.yandex-team.ru/CHEMODAN-27973
        parent = kwargs.get('parent')
        if parent:
            auto_login = getattr(parent.request, 'auto_login', False)
            installer_source = getattr(parent.request, 'installer_source', None)
            open_url_after_install = getattr(parent.request, 'open_url_after_install', None)
            file_meta = kwargs.get('data', {}).get('meta', {})
            if auto_login and file_meta.get('auto_login') and uid != '0':
                # проверяем что в запросе был параметр auto_login, в файле есть установленный через setprop
                # атрибут al, и юзер инициализирован или может быть проинициализирован то генерируем ссылку с ним
                if mpfs.engine.process.usrctl().is_user_init(uid):
                    kwargs['auto_login'] = True
                else:
                    from mpfs.core.user.standart import StandartUser  # cycle import
                    if StandartUser.can_init_user(uid):
                        kwargs['auto_login'] = True
            if installer_source is not None and file_meta.get('patch_installer'):
                # проверяем, что если в запросе был параметр installer_source и в файле есть установленный через setprop
                # атрибут patch_installer, то генерируем ссылку с параметром src
                if not installer_source:
                    installer_source = 'Yandex.Unknown'
                kwargs['installer_source'] = installer_source
            if open_url_after_install and file_meta.get('auto_login'):
                kwargs['open_url_after_install'] = open_url_after_install
        super(StockFile, self).__init__(uid, address, *args, **kwargs)

    def get_file_url(self, **kwargs):
        url = super(StockFile, self).get_file_url(**kwargs)
        qs_params = {}
        if kwargs.get('auto_login'):
            qs_params['al'] = 1
        if kwargs.get('open_url_after_install'):
            qs_params['ouai'] = kwargs['open_url_after_install']
        if kwargs.get('installer_source'):
            qs_params['src'] = kwargs['installer_source']
        if qs_params:
            url = update_qs_params(url, qs_params)
        return url


class StockFolder(MPFSFolder, StockResource):

    file_class = StockFile
    service_class = Stock

    def __init__(self, uid, address, *args, **kwargs):
        address.change_uid(SHARE_UID)
        super(StockFolder, self).__init__(uid, address, *args, **kwargs)
        self.meta['default'] = 1
        self.meta['default_name'] = 1

    @classmethod
    def from_dict(cls, data, user_principal=None):
        address = Address.Make(data['uid'], data['key'])
        if address.storage_name != 'share':
            raise errors.ResourceNotFound()
        return super(StockFolder, cls).from_dict(data, user_principal=user_principal)
