# -*- coding: utf-8 -*-
"""

MPFS
CORE

Корзина, Trash

"""
import time

import mpfs.engine.process

from mpfs.core.filesystem.resources.disk import DiskResource, MPFSFolder, MPFSFile
from mpfs.core.services.trash_service import Trash
from mpfs.common import errors
from mpfs.core.address import Address
from mpfs.metastorage.mongo.util import parent_for_key

log = mpfs.engine.process.get_default_log()
error_log = mpfs.engine.process.get_error_log()


class TrashResource(DiskResource):

    def __init__(self, *args, **kwargs):
        DiskResource.__init__(self, *args, **kwargs)
        if self.original_id:
            self.name = filter(None, self.original_id.split('/'))[-1]
        if self.original_id and 'original_parent_id' not in self.meta:
            self.meta['original_parent_id'] = parent_for_key(self.original_id) + '/'  # слеш нужен, т.к. parent - папка

    def prepare_transfer(self):
        for k in ('original_id', 'original_parent_id', 'append_time'):
            self.meta.pop(k, '')

    def check_rw(self, operation=None):
        if operation and operation in ('set_public','set_public_settings',):
            raise errors.NotPermitted()

    @property
    def is_trash(self):
        return True

    def _is_blockings_needed(self):
        return False


class TrashFile(TrashResource, MPFSFile):

    service_class = Trash

    not_saving_meta_fields = TrashResource.not_saving_meta_fields + MPFSFile.not_saving_meta_fields

    def __init__(self, *args, **kwargs):
        MPFSFile.__init__(self, *args, **kwargs)
        TrashResource.__init__(self, *args, **kwargs)

    def dir(self):
        result = super(TrashFile, self).dir()
        result['original_id'] = None
        return result

    def setup(self, **args):
        super(TrashFile, self).setup(**args)
        # Устанавливаем source по умолчанию
        if not self.source and self._service:
            self.source = self._service.name

        # Преобразуем size в int
        self.size = int(self.size or 0)

        # Выставляем drweb по умолчанию
        self.meta['drweb'] = int(self.meta.get('drweb') or 0)

    @classmethod
    def Create(classname, uid, address, parent, **kw):
        data = kw.get('data')
        file_item = None

        if data and isinstance(data, dict):
            value = {}

            for k, v in data.iteritems():
                if k not in classname.static_fields:
                    value[k] = v

            if 'meta' in value:
                value['meta'].pop('wh_version', '')
            else:
                value['meta'] = {}

            if not value['meta'].get('file_id'):
                value['meta']['file_id'] = classname.generate_file_id(uid, address.path)

            value.update({
                'visible': 1,
                'source': Address(address.uid + ':' + data['id']).storage_name,
            })
            Trash.update_value_with_trash_attributes(value, address, data['id'])

            value['hid'] = data['hid']
            value.pop('id', '')
            value.pop('media_type', '')
            value['meta'].pop('id', '')

            response = classname.service_class().put(address, value)
            value['version'] = int(response.version)
            value['type'] = 'file'
            value['name'] = data['name']
            kw['data'] = value
            file_item = classname(uid, address, **kw)
        file_item.set_parent(parent)
        return file_item

    @classmethod
    def from_dict(cls, data, user_principal=None):
        address = Address.Make(data['uid'], data['key'])
        if address.storage_name != 'trash':
            raise errors.ResourceNotFound()
        return super(TrashFile, cls).from_dict(data, user_principal=user_principal)


class TrashFolder(TrashResource, MPFSFolder):

    file_class = TrashFile
    service_class = Trash

    not_saving_meta_fields = TrashResource.not_saving_meta_fields + MPFSFolder.not_saving_meta_fields

    def __init__(self, *args, **kwargs):
        MPFSFolder.__init__(self, *args, **kwargs)
        TrashResource.__init__(self, *args, **kwargs)
        if self.address.is_storage:
            if self.request and self.request.method_name in ('info',):
                if not self.request.meta or 'empty' in self.request.meta:
                    self.empty = self._service.is_empty(self.uid)

    def drop(self):
        if self.address.is_storage:
            size = self.get_size()
            self._service.drop(self.uid)
            self.update_used(-1 * int(size))
        else:
            raise errors.FolderError()

    def rm(self, *args, **kwargs):
        if self.address.is_storage:
            raise errors.MPFSNotImplemented
        return super(TrashFolder, self).rm(*args, **kwargs)

    def dir(self):
        result = super(TrashFolder, self).dir()
        result['original_id'] = None
        if self.address.is_storage:
            result['empty'] = None
        return result

    def info(self):
        return {'this': self.dict(), 'list': []}

    @classmethod
    def from_dict(cls, data, user_principal=None):
        address = Address.Make(data['uid'], data['key'])
        if address.storage_name != 'trash':
            raise errors.ResourceNotFound()
        return super(TrashFolder, cls).from_dict(data, user_principal=user_principal)
