# -*- coding: utf-8 -*-
"""

MPFS
CORE

Симлинки

"""
import time
import re
from copy import deepcopy
from collections import defaultdict
import datetime
import operator
import itertools
import uuid

import mpfs.engine.process
import mpfs.common.errors as errors
from mpfs.common.util.crypt import CryptAgent
from mpfs.config import settings
import hashlib

from mpfs.core.metastorage.control import link_data
from mpfs.common.util import Cached, datetime_to_unixtime, UnicodeBase64
from mpfs.core.address import SymlinkAddress
from mpfs.core.office.auth import _timestamp_millis
from mpfs.core.social.dao.link_data import LinkDataDAO, LinkDataDAOItem
from mpfs.metastorage.postgres.schema import ResourceType
from mpfs.metastorage.mongo.util import generate_version_number

log = mpfs.engine.process.get_default_log()
error_log = mpfs.engine.process.get_error_log()

CREDENTIAL_CRYPTER_PUBLIC_PASSWORD_SALT = settings.credential_crypter['public_password_salt']
CREDENTIAL_CRYPTER_PUBLIC_PASSWORD_TOKEN_SECRET = settings.credential_crypter['public_password_token_secret']
DISK_PUBLIC_PASSWORD_TOKEN_TTL = settings.services['disk']['public_password_token_ttl']

PASSWORD_TOKEN_SRC_SEPARATOR = ':'


class Symlink(Cached):

    _dao_obj = LinkDataDAO()
    _instances = defaultdict(dict)

    '''
    Класс симлинка
    Объект, который ссылается на реальный файл пользователя
    '''
    def __init__(self, address, data={}, allow_deleted=False):
        if not data:
            result = link_data.show(address.uid, address.path)
            if result.value is None:
                raise errors.SymlinkNotFound()
            elif result.value.data.get('dtime') and not allow_deleted:
                raise errors.SymlinkNotFound()
            else:
                data = result.value.data

        for key, value in self.dir().iteritems():
            setattr(self, key, value)

        for key, value in data.iteritems():
            setattr(self, key, value)

        self.address = address

    @classmethod
    def find(cls, address):
        uid = address.uid
        path_split = address.path.split('/')
        split_length = len(path_split)
        for i in xrange(split_length, 2, -1):
            path = '/'.join(path_split[:i])
            symlink = cls._instances['uid'].get(path)
            if symlink:
                return symlink
        for i in xrange(split_length, 2, -1):
            path = '/'.join(path_split[:i])
            symlink_data = link_data.find_one_by_field(uid, {'data.tgt' : '%s:%s' % (uid, path)})
            if symlink_data:
                symlink = Symlink(cls.address_from_key(uid, symlink_data['key']), data=symlink_data['data'])
                cls._instances['uid'][path] = symlink
                return symlink
        raise errors.SymlinkNotFound()

    @classmethod
    def find_by_office_doc_short_id(cls, uid, office_doc_short_id):
        symlinks_data = link_data.find_all({'uid': uid,
                                           'data.office_doc_short_id' : office_doc_short_id,
                                           'data.dtime': {'$exists': False}})
        if symlinks_data:
            return [Symlink(cls.address_from_key(uid, symlink_data['key']), data=symlink_data['data'])
                    for symlink_data in symlinks_data]
        raise errors.SymlinkNotFound()

    @classmethod
    def find_by_file_id(cls, uid, file_id):
        symlinks_data = link_data.find_all({'uid': uid,
                                            'data.file_id': file_id,
                                            'data.dtime': {'$exists': False}})
        if symlinks_data:
            return [Symlink(cls.address_from_key(uid, symlink_data['key']), data=symlink_data['data'])
                    for symlink_data in symlinks_data]
        raise errors.SymlinkNotFound()

    @classmethod
    def find_previous_office_doc_short_id(cls, symlink):
        return LinkDataDAO().find_previous_office_doc_short_id(uid=symlink.get_uid(),
                                                               file_id=symlink.get_file_id())

    @classmethod
    def find_multiple(cls, symlink_addresses):
        """Получить симлинки по их адресам.

        Количество запросов в базу прямо пропорционально количеству разных пользователей, которым принадлежат
        переданные адреса.

        :param symlink_addresses: Список адресов симлинков.
        :type symlink_addresses: list[SymlinkAddress]
        """
        # группируем симлинки по уидам
        key_func = operator.attrgetter('uid')
        symlink_addresses = sorted(symlink_addresses, key=key_func)
        ret = []
        for uid, uid_addresses in itertools.groupby(symlink_addresses, key=key_func):
            _ids = []
            for symlink_address in uid_addresses:
                path = symlink_address.path
                _id = link_data.build_id(uid, path)
                _ids.append(_id)

            symlinks_data = link_data.find_all({'_id': {'$in': _ids}, 'uid': uid})
            for symlink_data in symlinks_data:
                symlink = Symlink(
                    cls.address_from_key(symlink_data['uid'], symlink_data['key']),
                    data=symlink_data['data']
                )
                ret.append(symlink)
        return ret

    @classmethod
    def list_all(cls, uid, order_by=None, reverse=False, offset=0, amount=0):
        """
        Вернёт список всех симлинков пользователя.

        :param uid:
        :param order_by: Список имён полей для сортировки симлинков.
        :param offset: Сколько элементов из попавших в выборку нужно пропустить.
        :param amount: Сколько элементов нужно вернуть.
        :return:
        """
        spec = {'uid': uid, 'data.dtime': {'$exists': False}}
        sort = None
        if order_by:
            sort = [(order_by, int(reverse) or -1)]

        raw_symlinks = link_data.find_all(spec, sort=sort, skip=offset, limit=amount)

        result = []
        for raw_symlink in raw_symlinks:
            if raw_symlink['key'] == '/':
                obj = None
            else:
                obj = Symlink(cls.address_from_key(uid, raw_symlink['key']), data=raw_symlink['data'])
            result.append(obj)
        return result

    def dir(self):
        return {
            'ctime': None,  # дата создания
            'mtime': None,  # дата изменения
            'tgt': None,  # адрес оригинального файла
            'file_id': None,  # file_id оригинального файла
            'resource_id_uid': None,  # uid для resource_id
            'user_ip': None,  # ip пользователя
            'public_uid': None,  # uid опубликовавшего
            'office_access_state': None,  # офисный идентификатор для генерации ссылка на редактирование
            'office_doc_short_id': None,  # офисный идентификатор для генерации ссылка на редактирование
            'password': None,  # пароль ссылки
            'read_only': None,  # read-only атрибут
            'available_until': None,  # время жизни ссылки
            'external_organization_ids': None,  # организации, для которых доступна ссылка
        }

    def dict(self):
        result = {}
        for attr in self.dir().keys():
            result[attr] = deepcopy(getattr(self, attr))
        return result

    @classmethod
    def Create(classname, tgt_address, **kwargs):
        '''
        Создание записи симлинка
        '''
        id = uuid.uuid4().hex
        current_time = datetime.datetime.now()
        dao_item = LinkDataDAOItem()
        dao_item.uid = tgt_address.uid
        dao_item.path = dao_item.build_key(id)
        dao_item.id = dao_item.build_id(dao_item.uid, dao_item.path)
        dao_item.target = tgt_address.id
        # В sql запросе баг. Поле parent не прописано. Создаем записи без него.
        # Это поле не несёт никакой смысловой нагрузки, поэтому всё работает.
        dao_item.parent = dao_item.build_id(dao_item.uid, '/')
        dao_item.type = ResourceType.FILE if tgt_address.is_file else ResourceType.DIR
        dao_item.date_created = current_time
        dao_item.date_modified = current_time
        dao_item.version = generate_version_number()
        dao_item.user_ip = kwargs.get('user_ip')
        dao_item.public_uid = kwargs.get('public_uid')

        resource_id = kwargs.get('resource_id')
        if resource_id:
            dao_item.file_id = resource_id.file_id
            dao_item.resource_id_uid = resource_id.uid
        else:
            dao_item.file_id = None
            dao_item.resource_id_uid = None

        dao_item.password = kwargs.get('password')
        dao_item.read_only = kwargs.get('read_only')
        dao_item.available_until = kwargs.get('available_until')

        classname._dao_obj.create(dao_item.uid, dao_item)
        obj = classname(SymlinkAddress.Make(tgt_address.uid, id),
                        data=dao_item.get_mongo_representation(skip_missing_fields=True)['data'])
        log.info('symlink %s to %s created' % (obj.address.id, tgt_address.id))
        return obj

    def update(self, data={}):
        self.mtime = time.time()
        for k, v in data.iteritems():
            if k in self.dir():
                setattr(self, k, v)
        link_data.put(self.address.uid, self.address.path, self.dict())
        log.info('symlink %s updated to %s' % (self.address.id, self.tgt))

    def update_office_fields(self, data):
        LinkDataDAO().update_office_fields(uid=self.address.uid,
                                           link_data_path=self.address.path,
                                           data=data)

    def set_target(self, tgt_address):
        self.update({'tgt': tgt_address.id})

    def set_resource_id(self, resource_id):
        self.update({'resource_id_uid': resource_id.uid, 'file_id': resource_id.file_id})

    def delete(self):
        data = self.dict()
        data['dtime'] = datetime.datetime.now()
        link_data.put(self.address.uid, self.address.path, data)

    def target_addr(self):
        return self.tgt

    def get_file_id(self):
        return self.file_id

    def get_resource_id_uid(self):
        return self.resource_id_uid

    def get_uid(self):
        return self.address.uid

    def get_office_access_state(self):
        return self.office_access_state

    def get_office_doc_short_id(self):
        return self.office_doc_short_id

    def get_password(self):
        return self.password

    def get_read_only(self):
        return self.read_only

    def get_available_until(self):
        return self.available_until

    def is_expired(self):
        if not self.available_until:
            return False
        return time.time() > self.available_until

    def have_password(self):
        return self.get_password() is not None

    def password_check(self, password):
        if password is None or not self.get_password() == self.hash_password(password):
            raise errors.SymlinkInvalidPassword()

    def get_external_organization_ids(self):
        if self.external_organization_ids is None:
            return []
        return self.external_organization_ids

    def hash_password(self, password):
        salt = CREDENTIAL_CRYPTER_PUBLIC_PASSWORD_SALT
        h = hashlib.sha384()
        h.update(password + self.get_uid() + salt)
        return h.hexdigest()

    @classmethod
    def reset(cls):
        cls._instances = defaultdict(dict)

    @classmethod
    def address_from_key(cls, uid, key):
        return SymlinkAddress.Make(uid, re.search('/([0-9a-f]+)', key).group(1))

    @classmethod
    def find_expired(cls, shard_endpoint, expiration_dt):
        return LinkDataDAO().find_expired(shard_endpoint, expiration_dt)

    def generate_password_token(self, public_hash, file_id, ttl=None):
        ttl = DISK_PUBLIC_PASSWORD_TOKEN_TTL if ttl is None else ttl
        secret_key = CREDENTIAL_CRYPTER_PUBLIC_PASSWORD_TOKEN_SECRET

        crypt_agent = CryptAgent(key=secret_key)
        expires = int(_timestamp_millis() + ttl)
        src = PASSWORD_TOKEN_SRC_SEPARATOR.join([UnicodeBase64.urlsafe_b64encode(public_hash), file_id, str(expires)])

        if isinstance(src, unicode):
            src = src.encode('utf-8')

        return crypt_agent.encrypt(src, urlsafe=True), expires

    def parse_password_token(self, token):
        secret_key = CREDENTIAL_CRYPTER_PUBLIC_PASSWORD_TOKEN_SECRET
        result = None
        if isinstance(token, unicode):
            token = token.encode('utf-8')
        try:
            crypt_agent = CryptAgent(key=secret_key)
            src = crypt_agent.decrypt(token, urlsafe=True)
            public_hash, file_id, expires = src.split(PASSWORD_TOKEN_SRC_SEPARATOR)
            result = (UnicodeBase64.urlsafe_b64decode(public_hash), file_id, int(expires))
        except (errors.DecryptionError, ValueError):
            log.exception('Unable to decrypt password token')
        return result

    def password_token_check(self, token, public_hash, file_id):
        token_data = self.parse_password_token(token)
        if token_data:
            public_hash_, file_id_, expires = token_data
            if public_hash_ == public_hash and file_id_ == file_id and expires > _timestamp_millis():
                return True
            if expires < _timestamp_millis():
                raise errors.SymlinkPasswordTokenExpired()
        raise errors.SymlinkInvalidPasswordToken()
