# -*- coding: utf-8 -*-
from mpfs.dao.base import (
    BaseDAOItem,
    BaseDAO,
    MongoBaseDAOImplementation,
    PostgresBaseDAOImplementation,
    BulkInsertReqGenerator,
)
from mpfs.dao.fields import IntegerField, HidField, FileIdField, UidField, BoolField, MicrosecondsTimestampField
from mpfs.dao.session import Session
from mpfs.metastorage.postgres import global_gallery_queries
from mpfs.metastorage.postgres.global_gallery_queries import (
    SQL_DELETION_LOG_COUNT_BY_UID,
    SQL_DELETION_LOG_DELETE_BY_UID,
    SQL_DELETION_LOG_GET_BY_UID,
    SQL_IS_HID_PRESENTED_IN_DELETION_LOG,
)
from mpfs.metastorage.postgres.schema import deletion_log


class DeletionLogDAOItem(BaseDAOItem):
    mongo_collection_name = None
    postgres_table_obj = deletion_log
    uid_field_name = 'uid'
    is_sharded = True

    uid = UidField(mongo_path='uid', pg_path=deletion_log.c.uid)
    file_id = FileIdField(mongo_path='file_id', pg_path=deletion_log.c.file_id)
    hid = HidField(mongo_path='hid', pg_path=deletion_log.c.storage_id)
    deletion_log_revision = MicrosecondsTimestampField(mongo_path='deletion_log_revision', pg_path=deletion_log.c.deletion_log_revision)
    is_live_photo = BoolField(mongo_path='is_live_photo', pg_path=deletion_log.c.is_live_photo, default_value=False)

    @classmethod
    def build_by_params(cls, uid, file_id, hid, is_live_photo=False):
        deletion_log_record = cls()
        deletion_log_record.uid = uid
        deletion_log_record.file_id = file_id
        deletion_log_record.hid = hid
        deletion_log_record.is_live_photo = is_live_photo
        return deletion_log_record


class DeletionLogDAO(BaseDAO):
    dao_item_cls = DeletionLogDAOItem

    def __init__(self):
        super(DeletionLogDAO, self).__init__()
        self._pg_impl = PostgresDeletionLogDAOImplementation(self.dao_item_cls)
        self._mongo_impl = MongoDeletionLogDAOImplementation(self.dao_item_cls)

    def save(self, item):
        return self._get_impl_by_uid(item.uid).save(item)

    def fetch_with_source_ids_sorted_by_revision(self, uid, start_revision, limit):
        return self._get_impl_by_uid(uid).fetch_with_source_ids_sorted_by_revision(
            uid, start_revision, limit)

    def count_by_uid(self, uid):
        return self._get_impl_by_uid(uid).count_by_uid(uid)

    def remove_by_uid(self, uid):
        return self._get_impl_by_uid(uid).remove_by_uid(uid)

    def fetch_by_uid(self, uid):
        return self._get_impl_by_uid(uid).fetch_by_uid(uid)

    def bulk_insert(self, uid, items):
        return self._get_impl_by_uid(uid).bulk_insert(uid, items)

    def is_hid_presented_in_deletion_log(self, uid, hid, is_live_photo=False):
        return self._get_impl_by_uid(uid).is_hid_presented_in_deletion_log(uid, hid, is_live_photo=is_live_photo)


class PostgresDeletionLogDAOImplementation(PostgresBaseDAOImplementation):

    def save(self, item):
        params = {c.name: v for c, v in item.get_postgres_representation().iteritems()}
        session = Session.create_from_uid(params['uid'])
        session.execute(global_gallery_queries.SQL_ADD_RECORD_TO_DELETION_LOG, params)

    def fetch_with_source_ids_sorted_by_revision(self, uid, start_deletion_log_revision, limit):
        from mpfs.core.global_gallery.dao.source_id import SourceIdDAOItem
        params = {
            'uid': self.get_field_repr('uid', uid),
            'deletion_log_revision': self.get_field_repr('deletion_log_revision', start_deletion_log_revision),
            'limit': limit,
        }
        session = Session.create_from_uid(params['uid'])
        result_proxy = session.execute(global_gallery_queries.SQL_GET_DELETION_RECORDS_SORTED_BY_REVISION, params)
        for doc in result_proxy:
            deletion_log_record = self.dao_item_cls.create_from_pg_data(doc)
            source_ids = [SourceIdDAOItem.build_by_params(
                    deletion_log_record.uid,
                    deletion_log_record.hid,
                    SourceIdDAOItem.convert_from_postgres('source_id', x),
                    deletion_log_record.is_live_photo
                ) for x in doc.source_ids
            ]
            yield (deletion_log_record, source_ids)

    def count_by_uid(self, uid):
        session = Session.create_from_uid(uid)
        return int(session.execute(
            SQL_DELETION_LOG_COUNT_BY_UID, {'uid': self.get_field_repr('uid', uid)}).fetchone()[0])

    def remove_by_uid(self, uid):
        session = Session.create_from_uid(uid)
        session.execute(SQL_DELETION_LOG_DELETE_BY_UID, {'uid': self.get_field_repr('uid', uid)})

    def fetch_by_uid(self, uid):
        session = Session.create_from_uid(uid)
        result_proxy = session.execute(SQL_DELETION_LOG_GET_BY_UID, {'uid': self.get_field_repr('uid', uid)})
        for doc in result_proxy:
            yield self.doc_to_item(doc)

    def bulk_insert(self, uid, items):
        source_ids_gen = BulkInsertReqGenerator(deletion_log, items)
        session = Session.create_from_uid(uid)
        session.execute(
            source_ids_gen.generate_tmpl(),
            source_ids_gen.generate_values()
        )

    def is_hid_presented_in_deletion_log(self, uid, hid, is_live_photo=False):
        params = {
            'uid': self.get_field_repr('uid', uid),
            'storage_id': self.get_field_repr('hid', hid),
            'is_live_photo': self.get_field_repr('is_live_photo', is_live_photo),
        }
        session = Session.create_from_uid(params['uid'])
        return session.execute(SQL_IS_HID_PRESENTED_IN_DELETION_LOG, params).fetchone() is not None


class MongoDeletionLogDAOImplementation(MongoBaseDAOImplementation):
    # Поддерживается только PG реализация

    def save(self, item):
        pass

    def fetch_with_source_ids_sorted_by_revision(self, uid, start_revision, limit):
        return []

    def count_by_uid(self, uid):
        return 0

    def remove_by_uid(self, uid):
        pass

    def fetch_by_uid(self, uid):
        return []

    def bulk_insert(self, uid, items):
        pass

    def is_hid_presented_in_deletion_log(self, uid, hid, is_live_photo=False):
        return False
