# -*- coding: utf-8 -*-
from mpfs.dao.base import (
    BaseDAOItem,
    BaseDAO,
    PostgresBaseDAOImplementation,
    MongoBaseDAOImplementation,
    BulkInsertReqGenerator,
    BulkDeleteReqGenerator,
)
from mpfs.dao.fields import (
    UidField,
    StringField,
    HidField,
    BoolField,
)
from mpfs.dao.session import Session
from mpfs.metastorage.postgres import global_gallery_queries
from mpfs.metastorage.postgres.global_gallery_queries import (
    SQL_DOES_RECORD_EXIST_BY_UID_SOURCE_ID,
    SQL_DOES_RECORD_EXIST_BY_UID_HID_BASE,
    SQL_SOURCE_ID_COUNT_BY_UID,
    SQL_SOURCE_ID_DELETE_BY_UID,
    SQL_SOURCE_ID_GET_BY_UID,
    SQL_GET_ALL_SOURCE_IDS_MAPPED_FOR_IDS,
)
from mpfs.metastorage.postgres.schema import source_ids


class SourceIdDAOItem(BaseDAOItem):
    mongo_collection_name = None
    postgres_table_obj = source_ids
    uid_field_name = 'uid'
    is_sharded = True

    uid = UidField(mongo_path='uid', pg_path=source_ids.c.uid)
    source_id = StringField(mongo_path='source_id', pg_path=source_ids.c.source_id)
    hid = HidField(mongo_path='hid', pg_path=source_ids.c.storage_id)
    is_live_photo = BoolField(mongo_path='is_live_photo', pg_path=source_ids.c.is_live_photo, default_value=False)

    @classmethod
    def build_by_params(cls, uid, hid, source_id, is_live_photo=False):
        upload_log_record = cls()
        upload_log_record.uid = uid
        upload_log_record.hid = hid
        upload_log_record.source_id = source_id
        upload_log_record.is_live_photo = is_live_photo
        return upload_log_record


class SourceIdDAO(BaseDAO):
    dao_item_cls = SourceIdDAOItem

    def __init__(self, session=None):
        super(SourceIdDAO, self).__init__(session=session)
        self._pg_impl = PostgresSourceIdDAOImplementation(self.dao_item_cls)
        self._mongo_impl = MongoSourceIdDAOImplementation(self.dao_item_cls)

    def save(self, item):
        return self._get_impl_by_uid(item.uid).save(item)

    def bulk_insert(self, uid, items):
        if not items:
            return
        return self._get_impl_by_uid(uid).bulk_insert(uid, items)

    def bulk_remove(self, uid, items):
        if not items:
            return
        return self._get_impl_by_uid(uid).bulk_remove(uid, items)

    def set_live_photo_flag_to_source_ids(self, uid, hid):
        return self._get_impl_by_uid(uid).set_live_photo_flag_to_source_ids(uid, hid)

    def get_source_ids_for_hids(self, uid, hids, is_live_photo=None):
        return self._get_impl_by_uid(uid).get_source_ids_for_hids(uid, hids, is_live_photo=is_live_photo)

    def fetch_source_ids_for_ids(self, uid, source_ids):
        return self._get_impl_by_uid(uid).fetch_source_ids_for_ids(uid, source_ids)

    def does_exist_by_hashes(self, uid, hid, is_live_photo=False):
        return self._get_impl_by_uid(uid).does_exist_by_hashes(uid, hid, is_live_photo=is_live_photo)

    def does_exist_by_source_id(self, uid, source_id):
        return self._get_impl_by_uid(uid).does_exist_by_source_id(uid, source_id)

    def count_by_uid(self, uid):
        return self._get_impl_by_uid(uid).count_by_uid(uid)

    def remove_by_uid(self, uid):
        return self._get_impl_by_uid(uid).remove_by_uid(uid)

    def fetch_by_uid(self, uid):
        return self._get_impl_by_uid(uid).fetch_by_uid(uid)


class PostgresSourceIdDAOImplementation(PostgresBaseDAOImplementation):

    def save(self, item):
        params = {c.name: v for c, v in item.get_postgres_representation().iteritems()}
        session = Session.create_from_uid(params['uid'])
        session.execute(global_gallery_queries.SQL_INSERT_UPLOAD_LOG_RECORD, params)

    def bulk_insert(self, uid, items):
        source_ids_gen = BulkInsertReqGenerator(source_ids, items, on_conflict_do_nothing=True)
        session = self.get_session(uid)
        session.execute(
            source_ids_gen.generate_tmpl(),
            source_ids_gen.generate_values()
        )

    def bulk_remove(self, uid, items):
        source_ids_gen = BulkDeleteReqGenerator(
            source_ids,
            items,
            ['uid', 'source_id', 'storage_id', 'is_live_photo'],
        )
        session = self.get_session(uid)
        session.execute(
            source_ids_gen.generate_tmpl(),
            source_ids_gen.generate_values()
        )

    def set_live_photo_flag_to_source_ids(self, uid, hid):
        params = {
            'uid': self.get_field_repr('uid', uid),
            'storage_id': self.get_field_repr('hid', hid)
        }
        session = Session.create_from_uid(params['uid'])
        session.execute(global_gallery_queries.SQL_SET_LIVE_PHOTO_FLAG_TO_SOURCE_IDS, params)

    def get_source_ids_for_hids(self, uid, hids, is_live_photo=None):
        params = {
            'uid': self.get_field_repr('uid', uid),
            'storage_ids': tuple([self.get_field_repr('hid', hid) for hid in hids])
        }

        query = global_gallery_queries.SQL_GET_ALL_SOURCE_IDS_MAPPED_FOR_HIDS
        if is_live_photo is not None:
            params['is_live_photo'] = self.get_field_repr('is_live_photo', is_live_photo)
            query = global_gallery_queries.SQL_GET_ALL_SOURCE_IDS_MAPPED_FOR_HIDS_WITH_IS_LIVE_PHOTO_FLAG

        session = Session.create_from_uid(params['uid'])
        result_proxy = session.execute(query, params)

        r = [self.doc_to_item(doc) for doc in result_proxy]
        return r

    def fetch_source_ids_for_ids(self, uid, source_ids):
        params = {
            'uid': self.get_field_repr('uid', uid),
            'source_ids': [self.get_field_repr('source_id', x) for x in source_ids]
        }
        session = Session.create_from_uid(params['uid'])
        result_proxy = session.execute(SQL_GET_ALL_SOURCE_IDS_MAPPED_FOR_IDS, params)

        for doc in result_proxy:
            yield self.doc_to_item(doc)

    def does_exist_by_hashes(self, uid, hid, is_live_photo=False):
        params = {
            'uid': self.get_field_repr('uid', uid),
            'storage_id': self.get_field_repr('hid', hid),
            'is_live_photo': self.get_field_repr('is_live_photo', is_live_photo),
        }
        session = Session.create_from_uid(params['uid'])
        return session.execute(SQL_DOES_RECORD_EXIST_BY_UID_HID_BASE, params).fetchone() is not None

    def does_exist_by_source_id(self, uid, source_id):
        params = {
            'uid': self.get_field_repr('uid', uid),
            'source_id': self.get_field_repr('source_id', source_id),
        }
        session = Session.create_from_uid(params['uid'])
        return session.execute(SQL_DOES_RECORD_EXIST_BY_UID_SOURCE_ID, params).fetchone() is not None

    def count_by_uid(self, uid):
        session = Session.create_from_uid(uid)
        return int(session.execute(SQL_SOURCE_ID_COUNT_BY_UID, {'uid': self.get_field_repr('uid', uid)}).fetchone()[0])

    def remove_by_uid(self, uid):
        session = Session.create_from_uid(uid)
        session.execute(SQL_SOURCE_ID_DELETE_BY_UID, {'uid': self.get_field_repr('uid', uid)})

    def fetch_by_uid(self, uid):
        session = Session.create_from_uid(uid)
        result_proxy = session.execute(SQL_SOURCE_ID_GET_BY_UID, {'uid': self.get_field_repr('uid', uid)})
        for doc in result_proxy:
            yield self.doc_to_item(doc)


class MongoSourceIdDAOImplementation(MongoBaseDAOImplementation):
    # Поддерживается только PG реализация

    def save(self, item):
        pass

    def bulk_insert(self, uid, items):
        pass

    def bulk_remove(self, uid, items):
        pass

    def set_live_photo_flag_to_source_ids(self, uid, hid):
        pass

    def get_source_ids_for_hids(self, uid, hids, is_live_photo=None):
        return []

    def get_source_ids_for_ids(self, uid, source_ids):
        return []

    def does_exist_by_hashes(self, uid, hid, is_live_photo=False):
        return False

    def does_exist_by_source_id(self, uid, source_id):
        return False

    def count_by_uid(self, uid):
        return 0

    def remove_by_uid(self, uid):
        pass

    def fetch_by_uid(self, uid):
        return []
