# -*- coding: utf-8 -*-
from mpfs.core.address import ResourceId
from mpfs.core.filesystem.hardlinks.common import FileChecksums
from mpfs.core.global_gallery.logic.errors import UploadRecordNotFoundError
from mpfs.core.global_gallery.logic.controller import GlobalGalleryController
from mpfs.core.metastorage.decorators import user_exists_and_not_blocked
from mpfs.core.snapshot.logic.snapshot import _get_request_body


@user_exists_and_not_blocked
def add_source_ids(req):
    """Добавить source_id к файлу
    """
    data = _get_request_body(req)
    hid = FileChecksums(req.md5, req.sha256, req.size).hid
    return GlobalGalleryController.add_source_ids_to_file(req.uid, hid, data['source_ids'],
                                                          is_live_photo=req.is_live_photo)


@user_exists_and_not_blocked
def check_source_id(req):
    """Проверить, есть ли запись по source_id
    """
    if not GlobalGalleryController.does_source_id_exist(req.uid, req.source_id):
        raise UploadRecordNotFoundError()


@user_exists_and_not_blocked
def check_source_ids(req):
    """Проверка по пачке source_ids
    """
    data = _get_request_body(req)
    requested_source_ids_set = set(data['source_ids'])
    if not requested_source_ids_set:
        return {'items': []}
    source_id_dao_items = GlobalGalleryController.fetch_source_ids_for_ids(req.uid, requested_source_ids_set)
    found_source_ids = {x.source_id for x in source_id_dao_items}
    result = [
        {
            'source_id': x,
            'found': x in found_source_ids,
        } for x in requested_source_ids_set
    ]
    return {'items': result}


@user_exists_and_not_blocked
def read_deletion_log(req):
    """Получить записи про удаления ресурсов
    """
    deletion_log_records_with_source_ids = (
        GlobalGalleryController.fetch_with_source_ids_sorted_by_revision(
            req.uid, req.deletion_log_revision))

    items = []
    for deletion_log_record in deletion_log_records_with_source_ids:
        new_item = deletion_log_record[0].get_mongo_representation()
        new_item['source_ids'] = [x.source_id for x in deletion_log_record[1]]
        new_item['resource_id'] = ResourceId(new_item['uid'], new_item['file_id']).serialize()
        del new_item['file_id']
        items.append(new_item)

    result = {
        'items': items,
        'total': len(deletion_log_records_with_source_ids),
        'deletion_log_revision': (req.deletion_log_revision if not deletion_log_records_with_source_ids
                                  else deletion_log_records_with_source_ids[-1][0].deletion_log_revision),
    }
    return result
