# -*- coding: utf-8 -*-
from collections import defaultdict

import mpfs.engine.process
from mpfs.common.static.tags import FILE
from mpfs.common.util.experiments.logic import experiment_gateway_wrapper
from mpfs.common.util.filetypes import MediaType

from mpfs.config import settings
from mpfs.core.filesystem.dao.file import FileDAO

from mpfs.core.filesystem.hardlinks.common import FileChecksums
from mpfs.core.global_gallery.dao.deletion_log import DeletionLogDAO
from mpfs.core.global_gallery.dao.source_id import SourceIdDAOItem, SourceIdDAO
from mpfs.core.global_gallery.logic.errors import UploadRecordNotFoundError
from mpfs.dao.session import Session

default_log = mpfs.engine.process.get_default_log()

GLOBAL_GALLERY_MAX_SOURCE_ID_LIMIT = settings.global_gallery['max_source_id_limit']
GLOBAL_GALLERY_DELETION_LOG_CHUNK_SIZE = settings.global_gallery['deletion_log_chunk_size']
EXPERIMENT_NAME = 'global_gallery'


class GlobalGalleryController(object):
    source_ids_dao = SourceIdDAO()
    deletion_log_dao = DeletionLogDAO()
    file_dao = FileDAO()
    loggable_media_types = {MediaType.IMAGE, MediaType.VIDEO}

    @classmethod
    @experiment_gateway_wrapper(EXPERIMENT_NAME)
    def add_source_ids_to_file(cls, uid, hid, source_ids, is_live_photo=False):
        """
        Добавить source_id запись по хэшам и уиду.

        Добавляем только тем файлам, у которым кол-во source_id < допустимой границы и тем, у которых еще нет
        переданного source_id
        """
        if not mpfs.engine.process.usrctl().is_user_in_postgres(uid):
            return

        r = [x for x in cls.source_ids_dao.get_source_ids_for_hids(uid, [hid]) if x.is_live_photo == is_live_photo]
        if not r and not cls.file_dao.does_exist_by_uid_hid(uid, hid, is_live_photo=is_live_photo):
            default_log.info('File not found in DB, source ids won\'t be added uid=%s hid=%s is_live_photo=%s'
                             % (uid, hid, is_live_photo))
            return

        existing_source_ids = {x.source_id for x in r}
        source_id_dao_items_to_add = []
        for source_id in list(set(source_ids) - existing_source_ids)[:max(
                GLOBAL_GALLERY_MAX_SOURCE_ID_LIMIT - len(existing_source_ids), 0)]:
            source_id_dao_items_to_add.append(SourceIdDAOItem.build_by_params(
                uid, hid, source_id, is_live_photo=is_live_photo))

        if source_id_dao_items_to_add:
            cls.source_ids_dao.bulk_insert(uid, source_id_dao_items_to_add)

    @classmethod
    @experiment_gateway_wrapper(EXPERIMENT_NAME)
    def try_to_write_to_deletion_log_by_file_doc(cls, uid, file_doc, force_live_photo_flag=False):
        """
        Попытаться записать информацию о файле в лог удалений.

        Действие фильтруется по медиа тайпу. Если записать не удалось, то поставим на это задачку.
        :param file_doc: результат resource.dict()
        """
        media_type = file_doc.get('media_type')
        if media_type not in cls.loggable_media_types:
            return

        md5 = file_doc['meta'].get('md5')
        sha256 = file_doc['meta'].get('sha256')
        size = file_doc.get('size')
        file_id = file_doc['meta'].get('file_id')
        is_live_photo = force_live_photo_flag or file_doc.get('is_live_photo', False)

        if any([x is None for x in (md5, sha256, size, file_id)]):
            default_log.info('One of the value is missing md5=%s , sha256=%s , size=%s , file_id=%s . Do nothing'
                             % (md5, sha256, size, file_id))
            return

        hid = FileChecksums(md5, sha256, int(size)).hid
        from mpfs.core.queue import mpfs_queue
        try:
            cls.add_deletion_log_record(uid, file_id, hid, is_live_photo=is_live_photo)
        except Exception:
            default_log.info('Couldn\'t write to deletion log syncronously', exc_info=True)
            mpfs_queue.put({
                'uid': uid, 'hid': hid, 'file_id': file_id},
                'add_deletion_log_record_if_needed_by_file_data')

    @classmethod
    @experiment_gateway_wrapper(EXPERIMENT_NAME)
    def set_live_photo_flag_to_source_ids(cls, uid, hid):
        cls.source_ids_dao.set_live_photo_flag_to_source_ids(uid, hid)

    @classmethod
    @experiment_gateway_wrapper(EXPERIMENT_NAME)
    def set_live_photo_flag_to_source_ids_and_add_one(self, uid, source_ids_to_add, hid):
        session = Session.create_from_uid(uid)
        source_id_dao = SourceIdDAO(session=session)

        source_id_items_to_delete = source_id_dao.get_source_ids_for_hids(uid, [hid], is_live_photo=False)

        with session.begin():
            source_id_dao.bulk_remove(uid, source_id_items_to_delete)

            source_id_item_to_add = source_id_items_to_delete
            for source_id_item in source_id_item_to_add:
                source_id_item.is_live_photo = True
            if source_ids_to_add:
                source_id_item_to_add.extend([
                    SourceIdDAOItem.build_by_params(uid, hid, source_id, True)
                    for source_id in source_ids_to_add
                ])
            source_id_dao.bulk_insert(uid, source_id_item_to_add)

    @classmethod
    @experiment_gateway_wrapper(EXPERIMENT_NAME)
    def add_deletion_log_record(cls, uid, file_id, hid, is_live_photo=False):
        deletion_log_record_dao_item = cls.deletion_log_dao.dao_item_cls.build_by_params(
            uid, file_id, hid, is_live_photo=is_live_photo)
        cls.deletion_log_dao.save(deletion_log_record_dao_item)

    @classmethod
    @experiment_gateway_wrapper(EXPERIMENT_NAME, return_value=[])
    def get_source_ids_for_hids(cls, uid, hids):
        return cls.source_ids_dao.get_source_ids_for_hids(uid, hids)

    @classmethod
    @experiment_gateway_wrapper(EXPERIMENT_NAME, return_value=[])
    def fetch_source_ids_for_ids(cls, uid, source_ids):
        return cls.source_ids_dao.fetch_source_ids_for_ids(uid, source_ids)

    @classmethod
    @experiment_gateway_wrapper(EXPERIMENT_NAME, return_value=[])
    def fetch_with_source_ids_sorted_by_revision(cls, uid, start_revision):
        return list(cls.deletion_log_dao.fetch_with_source_ids_sorted_by_revision(
            uid, start_revision, GLOBAL_GALLERY_DELETION_LOG_CHUNK_SIZE))

    @classmethod
    @experiment_gateway_wrapper(EXPERIMENT_NAME, return_value=False)
    def does_source_id_exist(cls, uid, source_id):
        return cls.source_ids_dao.does_exist_by_source_id(uid, source_id)

    @classmethod
    @experiment_gateway_wrapper(EXPERIMENT_NAME, return_value=False)
    def does_source_id_exist_by_hashes(cls, uid, md5, sha256, size, is_live_photo=False):
        return cls.does_source_id_exist_by_hid(uid, FileChecksums(md5, sha256, size).hid, is_live_photo=is_live_photo)

    @classmethod
    @experiment_gateway_wrapper(EXPERIMENT_NAME, return_value=False)
    def does_source_id_exist_by_hid(cls, uid, hid, is_live_photo=False):
        return cls.source_ids_dao.does_exist_by_hashes(uid, hid, is_live_photo=is_live_photo)

    @classmethod
    @experiment_gateway_wrapper(EXPERIMENT_NAME, return_value=False)
    def is_hid_presented_in_deletion_log(cls, uid, hid, is_live_photo=False):
        return cls.deletion_log_dao.is_hid_presented_in_deletion_log(uid, hid, is_live_photo=is_live_photo)

    @classmethod
    @experiment_gateway_wrapper(EXPERIMENT_NAME)
    def load_and_set_source_ids_for_resources(cls, resources):
        def get_type_func(resource):
            return resource.type

        def get_hid_func(resource):
            return resource.hid

        def get_uid_func(resource):
            return resource.uid

        def get_is_live_photo_func(resource):
            return resource.meta.get('is_live_photo', False)

        def set_source_ids_func(resource, source_ids):
            resource.set_source_ids(source_ids)

        cls.__traverse_objects_and_set_source_ids(resources, get_type_func=get_type_func, get_hid_func=get_hid_func,
                                                  get_uid_func=get_uid_func,
                                                  get_is_live_photo_func=get_is_live_photo_func,
                                                  set_source_ids_func=set_source_ids_func)

    @classmethod
    @experiment_gateway_wrapper(EXPERIMENT_NAME)
    def load_and_set_source_ids_for_resource_docs(cls, resource_docs):
        def get_type_func(resource_doc):
            return resource_doc['type']

        def get_hid_func(resource_doc):
            return FileChecksums(
                resource_doc['md5'], resource_doc['sha256'], resource_doc['size']).hid

        def get_uid_func(resource_doc):
            return resource_doc['uid']

        def get_is_live_photo_func(resource_doc):
            return resource_doc.get('is_live_photo', False)

        def set_source_ids_func(resource_doc, source_ids):
            resource_doc['source_ids'] = source_ids

        cls.__traverse_objects_and_set_source_ids(resource_docs, get_type_func=get_type_func, get_hid_func=get_hid_func,
                                                  get_uid_func=get_uid_func,
                                                  get_is_live_photo_func=get_is_live_photo_func,
                                                  set_source_ids_func=set_source_ids_func)

    @classmethod
    def __traverse_objects_and_set_source_ids(cls, object_collection, get_type_func, get_hid_func, get_uid_func,
                                              get_is_live_photo_func, set_source_ids_func):
        uid_to_hid_to_obj_map = defaultdict(lambda: defaultdict(list))
        for obj in object_collection:
            if get_type_func(obj) != FILE:
                continue
            uid_to_hid_to_obj_map[get_uid_func(obj)][(str(get_hid_func(obj)), get_is_live_photo_func(obj))].append(obj)

        for uid, hid_to_obj_map in uid_to_hid_to_obj_map.iteritems():
            source_id_items = cls.get_source_ids_for_hids(uid, [x for x, _ in hid_to_obj_map.keys()])

            hid_live_photo_flag_to_source_ids_map = defaultdict(list)
            for source_id_item in source_id_items:
                hid_live_photo_flag_to_source_ids_map[source_id_item.hid, source_id_item.is_live_photo].append(
                    source_id_item.source_id)

            for hid_live_photo_flag, source_ids in hid_live_photo_flag_to_source_ids_map.iteritems():
                for obj in hid_to_obj_map[hid_live_photo_flag]:
                    set_source_ids_func(obj, source_ids)
                del hid_to_obj_map[hid_live_photo_flag]

            # Проставляем пустой список для всех ресурсов, у которых нет source_ids
            for objs in hid_to_obj_map.itervalues():
                for obj in objs:
                    set_source_ids_func(obj, [])
