# -*- coding: utf-8 -*-

from mpfs.dao.base import BaseDAO, BaseDAOItem, BulkInsertReqGenerator, PostgresBaseDAOImplementation
from mpfs.dao.fields import DateTimeField, EnumField, JsonField, UidField
from mpfs.metastorage.postgres.schema import InactiveUsersFlowState, inactive_users_flow

SQL_GET_BY_STATE = 'SELECT * FROM disk.inactive_users_flow WHERE state = :state LIMIT :limit'
SQL_GET_BY_STATE_AND_LESS_START_DT = 'SELECT * FROM disk.inactive_users_flow WHERE state = :state AND start_time < :start_time LIMIT :limit'
SQL_GET_BY_UID = 'SELECT * FROM disk.inactive_users_flow WHERE uid = :uid'
SQL_UPDATE_BY_UID = '''
UPDATE disk.inactive_users_flow SET
    state = :state,
    start_time = :start_time,
    update_time = :update_time,
    debug_data = :debug_data
WHERE uid = :uid
'''


class InactiveUsersFlowDAOItem(BaseDAOItem):
    postgres_table_obj = inactive_users_flow
    is_sharded = False
    is_migrated_to_postgres = True

    @classmethod
    def get_postgres_primary_key(cls):
        return 'uid'

    uid = UidField(None, inactive_users_flow.c.uid)
    state = EnumField(None, inactive_users_flow.c.state, InactiveUsersFlowState, default_value=InactiveUsersFlowState.new)
    start_time = DateTimeField(None, inactive_users_flow.c.start_time, default_value=None)
    update_time = DateTimeField(None, inactive_users_flow.c.update_time, default_value=None)
    debug_data = JsonField(None, inactive_users_flow.c.debug_data, default_value=None)


class InactiveUsersFlowDAO(BaseDAO):
    dao_item_cls = InactiveUsersFlowDAOItem

    def __init__(self):
        super(InactiveUsersFlowDAO, self).__init__()
        self._pg_impl = PostgresInactiveUsersFlowDAOImplementation(self.dao_item_cls)

    def get_by_uid(self, uid):
        return self._get_impl(None).get_by_uid(uid)

    def fetch_by_state_and_start_time(self, state, start_time, limit=None):
        return self._get_impl(None).fetch_by_state_and_start_time(state, start_time, limit=limit)

    def fetch_by_state(self, state, limit=None):
        return self._get_impl(None).fetch_by_state(state, limit=limit)

    def bulk_insert(self, items):
        return self._get_impl(None).bulk_insert(items)

    def update_by_uid(self, item):
        return self._get_impl(None).update_by_uid(item)


class PostgresInactiveUsersFlowDAOImplementation(PostgresBaseDAOImplementation):
    def get_by_uid(self, uid):
        session = self._get_session()
        result_proxy = session.execute(
            SQL_GET_BY_UID,
            {'uid': uid}
        )
        return self.fetch_one_item(result_proxy)

    def fetch_by_state_and_start_time(self, state, start_time, limit=None):
        session = self._get_session()
        result_proxy = session.execute(
            SQL_GET_BY_STATE_AND_LESS_START_DT,
            {'state': state.value, 'start_time': start_time, 'limit': limit}
        )
        for doc in result_proxy:
            yield self.doc_to_item(doc)

    def fetch_by_state(self, state, limit=None):
        session = self._get_session()
        result_proxy = session.execute(
            SQL_GET_BY_STATE,
            {'state': state.value, 'limit': limit}
        )
        for doc in result_proxy:
            yield self.doc_to_item(doc)

    def bulk_insert(self, items):
        session = self._get_session()
        tmpl = BulkInsertReqGenerator(inactive_users_flow, items, on_conflict_do_nothing=True)
        session.execute(
            tmpl.generate_tmpl(),
            tmpl.generate_values()
        )

    def update_by_uid(self, item):
        session = self._get_session()
        params = {c.name: v for c, v in item.get_postgres_representation().iteritems()}
        with session.begin():
            session.execute(SQL_UPDATE_BY_UID, params)
