# -*- coding: utf-8 -*-
from datetime import datetime, timedelta

from dateutil import parser
from mpfs.common.errors import NotFound
from mpfs.common.util import chunks2
from mpfs.core.inactive_users_flow.dao import InactiveUsersFlowDAO, InactiveUsersFlowDAOItem, InactiveUsersFlowState


def _fetch_table_from_yt(table_path, start_index=None, end_index=None):
    import yt.wrapper as yt
    from mpfs.core.mrstat.stat_utils import set_yt_proxy

    set_yt_proxy()
    table_path = yt.TablePath(table_path, start_index=start_index, end_index=end_index)
    return yt.read_table(table_path, format=yt.JsonFormat())


def inactive_users_flow_add_from_yt(req):
    """Создать записи флоу по таблице из YT

    Query string аргументы:
        * table_path [обязательный]
        * start_index - с индекса
        * end_index - до индекса
    """

    records = _fetch_table_from_yt(req.table_path, start_index=req.start_index, end_index=req.end_index)
    now_dt = datetime.now()
    dao = InactiveUsersFlowDAO()
    for chunk in chunks2(records, chunk_size=500):
        items = []
        for yt_row in chunk:
            item = InactiveUsersFlowDAOItem()
            item.uid = yt_row['uid']
            item.state = InactiveUsersFlowState.new
            item.start_time = None
            item.update_time = now_dt
            item.debug_data = None
            items.append(item)
        dao.bulk_insert(items)


def inactive_users_flow_add(req):
    """Добавить uid в flow

    Query string аргументы:
        * uid [обязательный]
    """
    item = InactiveUsersFlowDAOItem()
    item.uid = req.uid
    item.state = InactiveUsersFlowState.new
    item.start_time = None
    item.update_time = datetime.now()
    item.debug_data = None
    InactiveUsersFlowDAO().bulk_insert([item])


def inactive_users_flow_update(req):
    """Обновить элемент флоу

    Query string аргументы:
        * uid [обязательный]
        * state
        * delta_days - насколько сдвинуть дату начала
        * start_time - дата начала
    """
    dao = InactiveUsersFlowDAO()
    item = dao.get_by_uid(req.uid)
    if item is None:
        raise NotFound()

    if req.state:
        item.state = InactiveUsersFlowState(req.state)

    if req.start_time:
        item.start_time = parser.parse(req.start_time)
    elif req.delta_days:
        item.start_time = item.start_time + timedelta(days=int(req.delta_days))

    dao.update_by_uid(item)
