# -*- coding: utf-8 -*-
import datetime
import enum
import traceback

import mpfs.engine.process
from dateutil.relativedelta import relativedelta
from mpfs.common.errors import StorageInitUser
from mpfs.config import settings
from mpfs.core.email.logic import send_email_async_by_uid
from mpfs.core.inactive_users_flow.dao import InactiveUsersFlowDAO
from mpfs.core.queue import mpfs_queue
from mpfs.core.services.passport_service import passport
from mpfs.core.services.queller_service import TaskStatus, http_queller
from mpfs.core.support import comment
from mpfs.core.user.base import User
from mpfs.engine.queue2.celery import BaseTask, app
from mpfs.metastorage.postgres.schema import InactiveUsersFlowState
from psycopg2.tz import FixedOffsetTimezone

default_log = mpfs.engine.process.get_default_log()
INACTIVE_USERS_FLOW_ENABLED = settings.inactive_users_flow['enabled']

FLOW_SCHEDULE = {}
for raw_state, days in settings.inactive_users_flow['flow_schedule'].items():
    FLOW_SCHEDULE[InactiveUsersFlowState(raw_state)] = relativedelta(days=days)

BLOCK_COMMENT_TEXT = 'blocked for inactivity'
UNBLOCK_COMMENT_TEXT = 'unblock inactive user'
USER_ACTIVITY_TABLE_NAME = '//home/mpfs-stat/storage/mpfs_last_activity_date'
SAVE_ITEM_LOG_MSG = 'Inactive flow save. uid=%(uid)s, start_time=%(start_time)s, prev_state=%(prev_state)s, new_state=%(new_state)s, exclude_reason=%(exclude_reason)s'


class EmailCampaign(object):
    block_warning = 'inactive_users_flow_block_warning'
    blocked = 'inactive_users_flow_blocked'
    deletion_warning = 'inactive_users_flow_deletion_warning'
    deleted = 'inactive_users_flow_deleted'
    blocked_flow_blocked = 'blocked_flow_blocked'


class ExcludeReason(enum.Enum):
    mailish = 'mailish'
    has_staff = 'has_staff'
    no_last_active_date = 'no_last_active_date'
    active = 'active'
    user_not_found = 'user_not_found'
    stage_skipped = 'stage_skipped'
    blocked = 'blocked'
    not_blocked = 'not_blocked'
    overdrawn = 'overdrawn'
    already_deleted = 'already_deleted'


def can_be_unblock_inactive_flow(user):
    is_blocked = is_blocked_by_inactive_flow(user)
    if is_blocked is False:
        return False
    flow_item = InactiveUsersFlowDAO().get_by_uid(user.uid)
    if flow_item is None or flow_item.state == InactiveUsersFlowState.deleted:
        return False
    return True


def is_blocked_by_inactive_flow(user):
    if not user.is_blocked():
        return False
    comment_obj = get_last_block_comment(user)
    if comment_obj is None:
        return False
    return comment_obj['comment'] == BLOCK_COMMENT_TEXT


def get_last_block_comment(user):
    comments = comment.select(user.uid)
    block_comments = [c for c in comments if c['type'] == 'block_user']
    if len(block_comments) == 0:
        return
    return block_comments[-1]


class FlowItemProcessor(object):
    # сколько времени должно пройти, чтобы пользователь считался неактивным
    inactive_delta = relativedelta(years=2)
    # допустимо обрабатывать стейт, если запуск задержался не более чем на +/- state_active_interval
    state_active_interval = relativedelta(days=4)

    def __init__(self, flow_item):
        self.dao = InactiveUsersFlowDAO()
        self.flow_item = flow_item
        self.now_dt = None
        self.user = None

    def process(self):
        if not INACTIVE_USERS_FLOW_ENABLED:
            default_log.info('Inactive users flow disabled')
            return
        self.now_dt = datetime.datetime.now()
        try:
            self._process()
        except Exception as e:
            self._save(extra_debug_data={
                'last_err_time': datetime.datetime.now(),
                'last_err': '%r' % e,
                'last_err_tb': traceback.format_exc(),
            })
            raise

    def _process(self):
        try:
            self.user = User(self.flow_item.uid)
        except StorageInitUser:
            self._exclude(ExcludeReason.user_not_found)
            return
        if self.user.check_deleted() is not None:
            self._exclude(ExcludeReason.already_deleted)
            return
        if self.user.is_mailish():
            self._exclude(ExcludeReason.mailish)
            return
        if self._has_staff(self.flow_item.uid):
            self._exclude(ExcludeReason.has_staff)
            return
        if self.user.is_overdrawn():
            self._exclude(ExcludeReason.overdrawn)
            return

        if self.flow_item.start_time and self.flow_item.state in FLOW_SCHEDULE:
            state_delta = FLOW_SCHEDULE[self.flow_item.state]
            state_activate_dt = self.flow_item.start_time + state_delta
            now_dt = self.now_dt.replace(tzinfo=FixedOffsetTimezone(180))
            if (
                state_activate_dt - self.state_active_interval > now_dt
                or now_dt > state_activate_dt + self.state_active_interval
            ):
                self._exclude(ExcludeReason.stage_skipped)
                return

        state_func_map = {
            InactiveUsersFlowState.new: self._process_new,  # common entry point
            # inactive user flow
            InactiveUsersFlowState.block_warning_1: self._process_block_warning_1,
            InactiveUsersFlowState.block_warning_2: self._process_block_warning_2,
            InactiveUsersFlowState.blocked: self._process_blocked,
            InactiveUsersFlowState.deletion_warning_1: self._process_deletion_warning_1,
            InactiveUsersFlowState.deletion_warning_2: self._process_deletion_warning_2,
            # no email user flow
            InactiveUsersFlowState.no_email_blocked: self._process_no_email_blocked,
            InactiveUsersFlowState.no_email_deletion_warning_1: self._process_no_email_deletion_warning_1,
            InactiveUsersFlowState.no_email_deletion_warning_2: self._process_no_email_deletion_warning_2,
            # blocked user flow
            InactiveUsersFlowState.blocked_flow_blocked: self._process_blocked_flow_blocked,
            InactiveUsersFlowState.blocked_flow_deletion_warning_1: self._process_blocked_flow_deletion_warning_1,
            InactiveUsersFlowState.blocked_flow_deletion_warning_2: self._process_blocked_flow_deletion_warning_2,
        }

        if self.flow_item.state not in state_func_map:
            raise NotImplementedError("Unsupported state: %s" % self.flow_item)
        else:
            state_func_map[self.flow_item.state]()

    def _process_new(self):
        is_excluded = self._check_and_exclude_if_active()
        if is_excluded:
            return

        passport.reset()
        user_info = passport.userinfo(self.flow_item.uid, all_emails=True)
        address_list = user_info.get('address-list')

        if not address_list:
            # new -> no_email_blocked
            self.user.set_block(1, BLOCK_COMMENT_TEXT)
            self._save(InactiveUsersFlowState.no_email_blocked, start_time=self.now_dt)
        elif self.user.is_blocked():
            # new -> blocked_flow_blocked
            self.user.set_block(1, BLOCK_COMMENT_TEXT)  # add special comment
            self._send_email(EmailCampaign.blocked_flow_blocked, {
                'email': user_info['login'],
            })
            self._save(InactiveUsersFlowState.blocked_flow_blocked, start_time=self.now_dt)
        else:
            # new -> block_warning_1
            template_args = {
                'email': user_info['login'],
                'days_blocked': 14,
            }
            self._send_email(EmailCampaign.block_warning, template_args)
            self._save(InactiveUsersFlowState.block_warning_1, start_time=self.now_dt)

    def _process_block_warning_1(self):
        # block_warning_1 -> block_warning_2
        is_excluded = self._check_and_exclude_if_active()
        if is_excluded:
            return
        if self.user.is_blocked():
            self._exclude(ExcludeReason.blocked)
            return

        user_info = passport.userinfo(self.flow_item.uid)
        template_args = {
            'email': user_info['login'],
            'days_blocked': 7,
        }
        self._send_email(EmailCampaign.block_warning, template_args)
        self._save(InactiveUsersFlowState.block_warning_2)

    def _process_block_warning_2(self):
        # block_warning_2 -> blocked
        is_excluded = self._check_and_exclude_if_active()
        if is_excluded:
            return
        if self.user.is_blocked():
            self._exclude(ExcludeReason.blocked)
            return

        user_info = passport.userinfo(self.flow_item.uid)
        template_args = {
            'email': user_info['login'],
        }
        self._send_email(EmailCampaign.blocked, template_args)
        self.user.set_block(1, BLOCK_COMMENT_TEXT)
        self._save(InactiveUsersFlowState.blocked)

    def _process_blocked(self, new_state=InactiveUsersFlowState.deletion_warning_1):
        # blocked -> deletion_warning_1
        if not self.user.is_blocked():
            self._exclude(ExcludeReason.not_blocked)
            return

        user_info = passport.userinfo(self.flow_item.uid)
        template_args = {
            'email': user_info['login'],
            'days_deleted': 30,
        }
        self._send_email(EmailCampaign.deletion_warning, template_args)
        self._save(new_state)

    def _process_deletion_warning_1(self, new_state=InactiveUsersFlowState.deletion_warning_2):
        # deletion_warning_1 -> deletion_warning_2
        if not self.user.is_blocked():
            self._exclude(ExcludeReason.not_blocked)
            return

        user_info = passport.userinfo(self.flow_item.uid)
        template_args = {
            'email': user_info['login'],
            'days_deleted': 7,
        }
        self._send_email(EmailCampaign.deletion_warning, template_args)
        self._save(new_state)

    def _process_deletion_warning_2(self):
        # deletion_warning_2 -> deleted
        if not self.user.is_blocked():
            self._exclude(ExcludeReason.not_blocked)
            return

        user_info = passport.userinfo(self.flow_item.uid)
        template_args = {
            'email': user_info['login'],
            'days_deleted': 90,
        }
        self._send_email(EmailCampaign.deleted, template_args)
        # TODO add user deletion task
        self._save(InactiveUsersFlowState.deleted)

    def _process_no_email_blocked(self):
        # no_email_blocked -> no_email_deletion_warning_1
        self._process_blocked(new_state=InactiveUsersFlowState.no_email_deletion_warning_1)

    def _process_no_email_deletion_warning_1(self):
        # no_email_deletion_warning_1 -> no_email_deletion_warning_2
        self._process_deletion_warning_1(new_state=InactiveUsersFlowState.no_email_deletion_warning_2)

    def _process_no_email_deletion_warning_2(self):
        # no_email_deletion_warning_2 -> deleted
        self._process_deletion_warning_2()

    def _process_blocked_flow_blocked(self):
        # blocked_flow_blocked -> blocked_flow_deletion_warning_1
        self._process_blocked(new_state=InactiveUsersFlowState.blocked_flow_deletion_warning_1)

    def _process_blocked_flow_deletion_warning_1(self):
        # blocked_flow_deletion_warning_1 -> blocked_flow_deletion_warning_2
        self._process_deletion_warning_1(new_state=InactiveUsersFlowState.blocked_flow_deletion_warning_2)

    def _process_blocked_flow_deletion_warning_2(self):
        # blocked_flow_deletion_warning_2 -> deleted
        self._process_deletion_warning_2()

    def _send_email(self, campaign_name, template_args):
        send_email_async_by_uid(
            self.flow_item.uid,
            campaign_name,
            template_args=template_args,
            default_locale='en',
            allowed_locales=['ru', 'en', 'tr', 'ua', 'uk'],
            send_on_all_emails=True,
        )

    def _check_and_exclude_if_active(self):
        last_active_dt = self._get_last_active_date(self.flow_item.uid)
        if last_active_dt is None:
            self._exclude(ExcludeReason.no_last_active_date)
            return True
        if self.now_dt - self.inactive_delta < last_active_dt:
            self._exclude(ExcludeReason.active)
            return True
        return False

    def _get_last_active_date(self, uid):
        import yt.wrapper as yt
        from mpfs.core.mrstat.stat_utils import set_yt_proxy

        set_yt_proxy()
        table_path = yt.TablePath(USER_ACTIVITY_TABLE_NAME, exact_key=str(uid))
        records = yt.read_table(table_path, format=yt.JsonFormat())
        rows = list(records)
        if len(rows) == 0:
            return
        assert len(rows) == 1
        assert rows[0]['uid'] == uid
        raw_dt = rows[0]['last_activity_date']
        return datetime.datetime.strptime(raw_dt, '%Y-%m-%d')

    @staticmethod
    def _has_staff(uid):
        has_staff = False
        try:
            user_info = passport.userinfo(uid=uid)
            has_staff = user_info.get('has_staff', has_staff)
        except Exception:
            pass
        return has_staff

    def _exclude(self, exclude_reason):
        self._save(InactiveUsersFlowState.excluded, exclude_reason=exclude_reason)

    def _save(self, state=None, start_time=None, exclude_reason=None, extra_debug_data=None):
        flow_item = self.flow_item.copy()
        prev_state = flow_item.state

        new_debug_data = {} if flow_item.debug_data is None else flow_item.debug_data
        if state is not None:
            flow_item.state = state
            new_debug_data[state.value] = self.now_dt
        if start_time:
            flow_item.start_time = start_time
        if exclude_reason is not None:
            new_debug_data['exclude_reason'] = exclude_reason.value
        if extra_debug_data is not None:
            new_debug_data.update(extra_debug_data)
        flow_item.debug_data = new_debug_data
        flow_item.update_time = self.now_dt
        self.dao.update_by_uid(flow_item)
        log_data = {
            'uid': flow_item.uid,
            'start_time': flow_item.start_time.strftime('%Y-%m-%d %H:%M:%S') if flow_item.start_time else '',
            'prev_state': prev_state.value if prev_state else '',
            'new_state': flow_item.state.value if flow_item.state else '',
            'exclude_reason': flow_item.debug_data.get('exclude_reason', '') if flow_item.debug_data else '',
        }
        default_log.info(SAVE_ITEM_LOG_MSG % log_data)


class FlowManager(object):
    worker_task_name = 'mpfs.core.inactive_users_flow.logic.handle_process_inactive_users_flow_item'
    add_tasks_boarder = 2000
    batch_size = 25000

    @classmethod
    def process(cls):
        if not INACTIVE_USERS_FLOW_ENABLED:
            default_log.info('Inactive users flow disabled')
            return

        tasks_in_queue = http_queller.get_tasks_count(cls.worker_task_name, statuses=TaskStatus.not_finished())
        if tasks_in_queue > cls.add_tasks_boarder:
            default_log.info('%s tasks in queue. Skipping...', tasks_in_queue)
            return

        dao = InactiveUsersFlowDAO()
        tasks_created = 0
        now_dt = datetime.datetime.now()
        for state, delta in FLOW_SCHEDULE.items():
            start_time = now_dt - delta
            flow_items_iter = dao.fetch_by_state_and_start_time(state, start_time, limit=cls.batch_size)
            for flow_item in flow_items_iter:
                cls._create_task(flow_item)
                tasks_created += 1
                if tasks_created > cls.batch_size:
                    default_log.info('Create %s tasks.', tasks_created)
                    return

        flow_items_iter = dao.fetch_by_state(InactiveUsersFlowState.new, limit=cls.batch_size)
        for flow_item in flow_items_iter:
            cls._create_task(flow_item)
            tasks_created += 1
            if tasks_created > cls.batch_size:
                break

        default_log.info('Create %s tasks.', tasks_created)

    @classmethod
    def _create_task(cls, flow_item):
        task_data = {'flow_uid': flow_item.uid, 'raw_state': flow_item.state.value}
        mpfs_queue.put(task_data, 'process_inactive_users_flow_item', deduplication_id=str(flow_item.uid))


@app.task(base=BaseTask)
def handle_process_inactive_users_flow_item(flow_uid, raw_state, context=None, **kwargs):
    state = InactiveUsersFlowState(raw_state)
    flow_item = InactiveUsersFlowDAO().get_by_uid(flow_uid)
    if flow_item is None:
        raise ValueError('Flow item not found. uid: %s' % flow_uid)
    if flow_item.state != state:
        raise ValueError('Flow item is in an unexpected state. DB state: %s. Task state: %s' % (flow_item.state, state))

    FlowItemProcessor(flow_item).process()
