# -*- coding: utf-8 -*-


import time
from mpfs.core.operations import manager
from mpfs.core.services.aviary_service import Aviary

import mpfs.engine.process
from mpfs.core.user.constants import *
from mpfs.core.queue import mpfs_queue
from mpfs.engine.queue2.celery import BaseTask, app

log = mpfs.engine.process.get_default_log()
error_log = mpfs.engine.process.get_error_log()

JOB_COMPLETED_CODE = 4
JOB_FAILED_CODE = 40
JOB_RUNNING_CODE = 2


@app.task(base=BaseTask)
def handle_check_aviary_render_status(uid, oid, aviary_job_id, retry=0, context=None, **kwargs):
    operation = manager.get_operation(uid, oid)
    # уже получили колбэк - больше ничего делать не надо
    if operation.is_render_finished():
        return

    aviary = Aviary(request=None)
    try:
        job_status = aviary.get_job_status(aviary_job_id)
    except Exception:
        return

    code = int(job_status['code'])
    if code == JOB_COMPLETED_CODE:
        log.info("Aviary render completed.")
        operation.set_aviary_url(job_status['url'])
    elif code == JOB_FAILED_CODE:
        log.info("Aviary render failed.")
        operation.set_failed({'message': job_status['message']})
    elif code == JOB_RUNNING_CODE:
        _schedule_retry(uid, oid, aviary_job_id, retry + 1)
    else:
        message = 'Unknown job state: %s(%d)' % (job_status['jobStatus'], code)
        log.error(message)
        operation.set_failed({'message': message})


def _get_retry_delay_seconds(current_try):
    if current_try < 10:
        return 1
    elif current_try < 20:
        return 5
    elif current_try < 35:
        return 20
    elif current_try < 50:
        return 20 * 1.2 ** (current_try - 35)
    return None


def _schedule_retry(uid, oid, aviary_job_id, retry):
    delay = _get_retry_delay_seconds(retry)
    if delay is not None:
        log.info("Scheduling retry " + str(retry))
        mpfs_queue.put({'retry': retry, 'uid': uid, 'oid': oid, 'aviary_job_id': aviary_job_id},
                       'aviary_check_render_status', stime=time.time() + delay)
    else:
        log.info("Too many retries " + str(retry))
